package org.gcube.application.geoportal.storage;

import java.io.File;
import java.net.URL;

import org.gcube.application.geoportal.model.content.GeoServerContent;
import org.gcube.application.geoportal.model.fault.SDIInteractionException;
import org.gcube.data.transfer.library.DataTransferClient;
import org.gcube.data.transfer.library.TransferResult;
import org.gcube.data.transfer.library.faults.DestinationNotSetException;
import org.gcube.data.transfer.library.faults.FailedTransferException;
import org.gcube.data.transfer.library.faults.InitializationException;
import org.gcube.data.transfer.library.faults.InvalidDestinationException;
import org.gcube.data.transfer.library.faults.InvalidSourceException;
import org.gcube.data.transfer.library.faults.SourceNotSetException;
import org.gcube.data.transfer.model.Destination;
import org.gcube.data.transfer.model.DestinationClashPolicy;
import org.gcube.spatial.data.gis.GISInterface;
import org.gcube.spatial.data.gis.is.AbstractGeoServerDescriptor;

import lombok.extern.log4j.Log4j2;
import lombok.extern.slf4j.Slf4j;


@Slf4j
public class SDIManager {

	private GISInterface gis;
	private DataTransferClient dtGeoServer; 
	private String geoserverHostName;


	public SDIManager() throws SDIInteractionException {
		try{
			log.debug("Initializing GIS Interface..");
			gis=GISInterface.get();
			AbstractGeoServerDescriptor geoserver=gis.getCurrentGeoServer();						
			if(geoserver==null)	
				throw new Exception("Unable to contact data transfer for geoserver ");

			log.debug("Found geoserver descriptor "+geoserver);
			geoserverHostName=new URL(gis.getCurrentGeoServer().getUrl()).getHost();

			log.debug("Contacting Data Transfer from geoserver {} ",geoserverHostName);
			dtGeoServer=DataTransferClient.getInstanceByEndpoint("http://"+geoserverHostName);

		}catch(Exception e) {
			throw new SDIInteractionException("Unable to initialize SDI Manager",e);
		}
	}



	// GEOSERVER-PERSISTENCE-ID / GNA / PROJECT-ID /LAYER-ID/ FILENAME

	public GeoServerContent pushLayerFile(File theFile, String filename, long projectID, long layerID) throws SDIInteractionException {
		try {

			Destination destination=new Destination(filename);
			destination.setCreateSubfolders(true);
			destination.setOnExistingFileName(DestinationClashPolicy.REWRITE);
			destination.setPersistenceId("geoserver");
			destination.setSubFolder("GNA/"+projectID+"/"+layerID);
			TransferResult result=dtGeoServer.localFile(theFile, destination);

			GeoServerContent content=new GeoServerContent();
			content.setFilename(filename);
			content.setGeoserverPath(result.getRemotePath());
			content.setGeoserverHostName(geoserverHostName);
			return content;
		} catch (InvalidSourceException | SourceNotSetException | FailedTransferException | InitializationException
				| InvalidDestinationException | DestinationNotSetException e) {
			throw new SDIInteractionException("Unable to transfer file "+filename,e);
		}
	}


}
