package org.gcube.application.geoportal.service.rest;

import lombok.extern.slf4j.Slf4j;
import org.gcube.application.geoportal.common.model.legacy.Concessione;
import org.gcube.application.geoportal.common.model.rest.AddSectionToConcessioneRequest;
import org.gcube.application.geoportal.common.model.rest.Configuration;
import org.gcube.application.geoportal.common.model.rest.QueryRequest;
import org.gcube.application.geoportal.common.rest.InterfaceConstants;
import org.gcube.application.geoportal.service.engine.mongo.ConcessioniMongoManager;
import org.gcube.application.geoportal.service.engine.postgis.PostgisIndex;
import org.gcube.application.geoportal.service.model.internal.faults.DeletionException;
import org.json.JSONArray;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

@Path(InterfaceConstants.Methods.MONGO_CONCESSIONI)
@Slf4j
public class ConcessioniOverMongo {


	@GET
	@Path(InterfaceConstants.Methods.CONFIGURATION_PATH)
	@Produces(MediaType.APPLICATION_JSON)
	public Configuration getConfiguration(){
		return new GuardedMethod<Configuration>(){

			@Override
			protected Configuration run() throws Exception, WebApplicationException {
				Configuration toReturn = new Configuration();
				toReturn.setIndex(new PostgisIndex().getInfo());
				log.info("Returning configuration {} ",toReturn);
				return toReturn;
			}
		}.execute().getResult();
	}

	@PUT
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Concessione replace(Concessione c) {
		return new GuardedMethod<Concessione> () {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				//Concessione c=Serialization.read(jsonString, Concessione.class);
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				manager.replace(c);
				
//				return Serialization.write(manager.getById(c.getMongo_id()));
				return manager.getById(c.getMongo_id());
			}
		}.execute().getResult();
	}
	
	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Concessione createNew(Concessione c) {
		return new GuardedMethod<Concessione>() {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {

				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				return manager.registerNew(c);
			}
		}.execute().getResult();
	}
	
	
	
	@GET
	@Produces(MediaType.APPLICATION_JSON)
	public Iterable<Concessione> list() {
		return new GuardedMethod<Iterable<Concessione>>() {
			protected Iterable<Concessione> run() throws Exception ,WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				JSONArray toReturn=new JSONArray();
				return manager.list();
			
		};
	}.execute().getResult();
				
		
	}
	
	
	// BY ID 
	@GET
	@Produces(MediaType.APPLICATION_JSON)
	@Path("{"+InterfaceConstants.Parameters.PROJECT_ID+"}")
	public Concessione getById(@PathParam(InterfaceConstants.Parameters.PROJECT_ID) String id) {
		return new GuardedMethod<Concessione>() {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();				
				return manager.getById(id);
			}
		}.execute().getResult();
	}
	
	@DELETE
	@Produces(MediaType.APPLICATION_JSON)
	@Path("{"+InterfaceConstants.Parameters.PROJECT_ID+"}")
	public void deleteById(@PathParam(InterfaceConstants.Parameters.PROJECT_ID) String id,
						   @QueryParam(InterfaceConstants.Parameters.FORCE) Boolean forceOption) {
		new GuardedMethod<Concessione> () {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				try{
					Boolean force=(forceOption!=null)?forceOption:false;
					ConcessioniMongoManager manager=new ConcessioniMongoManager();
					manager.deleteById(id,force);
					return null;
				}catch(DeletionException e){
					throw new WebApplicationException("Unable to delete "+id,e, Response.Status.EXPECTATION_FAILED);
				}
			}
		}.execute();
	}
	
	
	@PUT
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("{"+InterfaceConstants.Parameters.PROJECT_ID+"}")
	public Concessione update(@PathParam(InterfaceConstants.Parameters.PROJECT_ID) String id,Concessione c) {
		return new GuardedMethod<Concessione>() {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				c.setMongo_id(id);
				return manager.replace(c);
			}
		}.execute().getResult();
	}
	
	
	@PUT
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/{"+InterfaceConstants.Methods.PUBLISH_PATH+"}/{"+InterfaceConstants.Parameters.PROJECT_ID+"}")
	public Concessione publish(@PathParam(InterfaceConstants.Parameters.PROJECT_ID) String id) {
		return new GuardedMethod<Concessione>() {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();				
				return manager.publish(id);
			}
		}.execute().getResult();
	}

	@DELETE
	@Path("/{"+InterfaceConstants.Methods.PUBLISH_PATH+"}/{"+InterfaceConstants.Parameters.PROJECT_ID+"}")
	public Concessione unpublish(@PathParam(InterfaceConstants.Parameters.PROJECT_ID) String id) {
		log.info("Unpublishing {} ",id);
		return new GuardedMethod<Concessione>() {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				return manager.unpublish(id);
			}
		}.execute().getResult();


	}


	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/"+InterfaceConstants.Methods.REGISTER_FILES_PATH+"/{"+InterfaceConstants.Parameters.PROJECT_ID+"}")
	public Concessione registerFile(@PathParam(InterfaceConstants.Parameters.PROJECT_ID) String id, AddSectionToConcessioneRequest request) {
		return new GuardedMethod<Concessione>() {
			@Override
			protected Concessione run() throws Exception, WebApplicationException {
				log.info("Registering {} file(s) for {} Concessione ID {}",
						request.getStreams().size(),
						request.getDestinationPath(),id);
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				return manager.persistContent(id, request.getDestinationPath(), request.getStreams());
			}
		}.execute().getResult();
	}

	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/"+InterfaceConstants.Methods.SEARCH_PATH)
	public Iterable<Concessione> search(String filter){
		return new GuardedMethod<Iterable<Concessione>>() {
			@Override
			protected Iterable<Concessione> run() throws Exception, WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				return manager.search(filter);
			}
		}.execute().getResult();
	}

	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/"+InterfaceConstants.Methods.QUERY_PATH)
	public Iterable<Concessione> search(QueryRequest queryRequest){
		return new GuardedMethod<Iterable<Concessione>>() {
			@Override
			protected Iterable<Concessione> run() throws Exception, WebApplicationException {
				ConcessioniMongoManager manager=new ConcessioniMongoManager();
				return manager.query(queryRequest);
			}
		}.execute().getResult();
	}
}
