package org.gcube.applicationsupportlayer.social;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.gcube.application.framework.core.session.ASLSession;
import org.gcube.applicationsupportlayer.social.mailing.EmailPlugin;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.portal.databook.shared.ApplicationProfile;
import org.gcube.portal.databook.shared.Notification;
import org.gcube.portal.databook.shared.NotificationChannelType;
import org.gcube.portal.databook.shared.NotificationType;
import org.gcube.portal.databook.shared.RunningJob;
import org.gcube.portal.databook.shared.ex.NotificationChannelTypeNotFoundException;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.vomanagement.usermanagement.UserManager;
import org.gcube.vomanagement.usermanagement.exception.UserManagementPortalException;
import org.gcube.vomanagement.usermanagement.exception.UserManagementSystemException;
import org.gcube.vomanagement.usermanagement.exception.UserRetrievalFault;
import org.gcube.vomanagement.usermanagement.impl.liferay.LiferayUserManager;
import org.gcube.vomanagement.usermanagement.model.UserModel;

/**
 * 
 * @author Massimiliano Assante, ISTI-CNR
 * @version 0.1 Dec 2012
 *
 * use to notify users from within your application 
 */
public class ApplicationNotificationsManager extends SocialPortalBridge implements NotificationsManager {
	static GCUBEClientLog _log = new GCUBEClientLog(ApplicationNotificationsManager.class);
	/**
	 * Use this constructor if you do not need notifications to point back to your applications
	 * @param aslSession the ASLSession instance
	 */
	public ApplicationNotificationsManager(ASLSession session) {
		super(session);
	}
	/**
	 * Use this constructor if you do need notifications to point back to your applications, 
	 * make sure you create your application profile on the infrastructure.
	 *    
	 * @see http://gcube.wiki.gcube-system.org/gcube/index.php/Social_Networking_Library#Create_Your_Application_Profile
	 * 
	 * @param aslSession the ASLSession instance
	 * @param portletClassName your portlet class name will be used ad unique identifier for your applicationProfile
	 */
	public ApplicationNotificationsManager(ASLSession session, String portletClassName) {
		super(session, portletClassName);
	}
	/**
	 * actually save the notification to the store
	 * @param notification2Save the notification instance to save
	 * @return true if the notification was sent ok
	 */
	private boolean saveNotification(Notification notification2Save) {
		_log.trace("Trying to send notification to: " + notification2Save.getUserid() +  " Type: " + notification2Save.getType());
		List<NotificationChannelType> channels = null;
		try {
			channels = getStoreInstance().getUserNotificationChannels(notification2Save.getUserid());
		} catch (NotificationChannelTypeNotFoundException e) {
			e.printStackTrace();
		}
		boolean result = false;
		if (channels.contains(NotificationChannelType.PORTAL)) {
			result = getStoreInstance().saveNotification(notification2Save);
			if (result)
				_log.trace("Notification Saved Successfully! ");
			else
				_log.error("Error While trying to save Notification");
		}
		if (channels.contains(NotificationChannelType.EMAIL))
			EmailPlugin.sendNotification(notification2Save, aslSession.getGroupName());
		
		if (channels.isEmpty()) {
			_log.info("Notification was not needed as the user decided not to be notified");
			result = true;
		}
		return result;
	}
	/**
	 * return the url of the application if exists in the profile
	 * @return .
	 */
	private String getApplicationUrl() {
		if (applicationProfile != null && applicationProfile.getUrl() != null)
			return applicationProfile.getUrl();
		else return "";
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyFolderSharing(String userIdToNotify, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_SHARE, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				"?oid="+sharedFolder.getId(), 
				"shared the workspace folder "+ sharedFolder.getName() +" with you",
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyFolderRenaming(String userIdToNotify, String previousName, String newName, String renamedFolderId) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_SHARE, 
				userIdToNotify, //user no notify
				renamedFolderId,  //the 
				new Date(),
				"?oid="+renamedFolderId, 
				"renamed your shared folder "+ previousName +" as " + newName,
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws UserManagementPortalException 
	 * @throws UserRetrievalFault 
	 * @throws UserManagementSystemException 
	 */
	@Override
	public boolean notifyFolderAddedUser(String userIdToNotify, WorkspaceFolder sharedFolder, String newAddedUserId) throws InternalErrorException, UserManagementSystemException, UserRetrievalFault, UserManagementPortalException {
		UserManager um = new LiferayUserManager();
		UserModel user = um.getUserByScreenName(newAddedUserId);
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_ADDEDUSER, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				"?oid="+sharedFolder.getId(),
				"added "+ user.getFullname() +" to your workspace shared folder "+ sharedFolder.getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyFolderRemovedUser(String userIdToNotify, WorkspaceFolder sharedFolder) throws InternalErrorException, UserManagementSystemException, UserRetrievalFault, UserManagementPortalException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_REMOVEDUSER, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				"?oid="+sharedFolder.getId(), 
				"unshared his shared folder "+ sharedFolder.getName() + " from your workspace",
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyAddedItem(String userIdToNotify, WorkspaceItem item, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_NEW, 
				userIdToNotify, //user no notify
				item.getId(),  //the 
				new Date(),
				"?oid="+item.getId()+"&parentoid="+item.getParent().getId(), 
				"added "+ item.getName() +" to your workspace shared folder "+ sharedFolder.getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyRemovedItem(String userIdToNotify,	WorkspaceItem item, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_DELETE, 
				userIdToNotify, //user no notify
				item.getId(),  //the 
				new Date(),
				"?oid="+item.getId()+"&parentoid="+item.getParent().getId(), 
				"removed "+ item.getName() +" from your workspace shared folder "+ sharedFolder.getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyUpdatedItem(String userIdToNotify,	WorkspaceItem item, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_UPDATED, 
				userIdToNotify, //user no notify
				item.getId(),  //the 
				new Date(),
				"?oid="+item.getId()+"&parentoid="+item.getParent().getId(), 
				" updated "+ item.getName() +" to your workspace shared folder "+ item.getParent().getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyMessageReceived(String userIdToNotify,	String subject) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.MESSAGE, 
				userIdToNotify, //user no notify
				"messageid_not_provided",  //the 
				new Date(),
				"/group/data-e-infrastructure-gateway/messages", 
				"sent you a message with subject: " + escapeHtml(subject),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyOwnCommentReply(String userIdToNotify, String feedid, String feedText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.OWN_COMMENT, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"commented on your post: " + escapeHtml(feedText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyCommentReply(String userIdToNotify, String feedid, String feedText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.COMMENT, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"commented on one of your favorite posts: " + escapeHtml(feedText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyLikedFeed(String userIdToNotify, String feedid, String feedText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.LIKE, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"likes your post: " + escapeHtml(feedText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyJobStatus(String userIdToNotify, ApplicationProfile executingJobApId, RunningJob job) {
		//TODO: missing implementation 
		return false;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowView(String userIdToNotify, String documentWorkflowId, String documentName) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_VIEW, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				"viewed document workflow " + escapeHtml(documentName),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowUpdate(String userIdToNotify, String documentWorkflowId, String documentName) {

		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_EDIT, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				"updated document workflow " + escapeHtml(documentName),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowTaskRequest(String userIdToNotify,	String documentWorkflowId, String documentName, String assignedRoleName) {
				
		String notificationText = "in " + aslSession.getGroupName() + " you are requested to perform a new task in the Document Workflow titled: "
				+  escapeHtml(documentName) + ". Your role is: " + assignedRoleName;
		
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_STEP_REQUEST_TASK, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				notificationText,
						false, 
						aslSession.getUsername(),
						aslSession.getUserFullName(), 
						aslSession.getUserAvatarId());

		return saveNotification(not);

	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowUserForward(String userIdToNotify,	String documentWorkflowId, String documentName, String fromStepName, String toStepName){
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_USER_FORWARD_TO_OWNER, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 

				"forwarded the Document Workflow titled: " + escapeHtml(documentName)  +
						" from status \"" + fromStepName + "\" to status \"" + toStepName +"\". In " + aslSession.getGroupName(),
						false, 
						aslSession.getUsername(),
						aslSession.getUserFullName(), 
						aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowStepForwardComplete(String userIdToNotify, String documentWorkflowId, String documentName, String fromStepName, String toStepName) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_FORWARD_STEP_COMPLETED_OWNER, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				"has performed the last needed forward on a Document Workflow titled: " + escapeHtml(documentName) + ". " + 
						"Step \"" + fromStepName + "\" is now complete. The next step is \"" + toStepName +"\". In " + aslSession.getGroupName(),
						false, 
						aslSession.getUsername(),
						aslSession.getUserFullName(), 
						aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowFirstStepRequest(String userIdToNotify, String documentWorkflowId,	String documentName, String assignedRole) {
		String notificationText = "involved you in the Document Workflow titled: " + escapeHtml(documentName) + ". " +
				"You are requested to perform a task. Your role is: " + assignedRole+"";
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_FIRST_STEP_REQUEST_INVOLVMENT, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				notificationText,
						false, 
						aslSession.getUsername(),
						aslSession.getUserFullName(), 
						aslSession.getUserAvatarId());

		return saveNotification(not);
	}
}
