package org.gcube.common.authorization.utils.manager;

import java.util.ArrayList;
import java.util.List;
import java.util.SortedSet;

import org.gcube.common.authorization.utils.provider.GCubeSecretProvider;
import org.gcube.common.authorization.utils.provider.SecretProvider;
import org.gcube.common.authorization.utils.secret.JWTSecret;
import org.gcube.common.authorization.utils.secret.Secret;
import org.gcube.common.authorization.utils.user.User;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class SecretManager {
	
	public static final InheritableThreadLocal<SecretManager> instance = new InheritableThreadLocal<SecretManager>() {

		@Override
		protected SecretManager initialValue() {
			return new SecretManager();
		}

	};
	
	private List<SecretProvider> secretProviders;
	
	private SecretHolder initialSecretHolder;
	private SecretHolder currentSecretHolder;
	
	private SecretManager(){
		initialSecretHolder = new SecretHolder();
		currentSecretHolder = initialSecretHolder;
		initSecretProviders();
	}
	
	protected void initSecretProviders() {
		secretProviders = new ArrayList<>();
		
		@SuppressWarnings("unchecked")
		Class<SecretProvider>[] classes = new Class[]{
			JWTSecret.class, GCubeSecretProvider.class
		};
		
		for(Class<SecretProvider> clz : classes) {
			try {
				SecretProvider authorizationSecretProvider = clz.newInstance();
				addSecretProvider(authorizationSecretProvider);
			} catch (Exception e) {
				
			}
		}
	}
	
	public List<SecretProvider> getSecretProviders(){
		return secretProviders;
	}
	
	public void addSecretProvider(SecretProvider secretProvider) {
		secretProviders.add(secretProvider);
		Secret secret = secretProvider.getSecret();
		initialSecretHolder.addSecret(secret);
	}
	
	public synchronized void startSession(Secret secret) throws Exception {
		if(currentSecretHolder!=initialSecretHolder) {
			throw new Exception("You are already in a session. You must terminate the session first.");
		}
		currentSecretHolder = new SecretHolder(secret);
		currentSecretHolder.set();
	}
	
	public synchronized  void startSession(SortedSet<Secret> secrets) throws Exception {
		currentSecretHolder = new SecretHolder(secrets);
		currentSecretHolder.set();
	}
	
	public synchronized void endSession() throws Exception {
		if(currentSecretHolder!=initialSecretHolder) {
			initialSecretHolder.set();
			currentSecretHolder = initialSecretHolder;
		}
	}
	
	public void reset() {
		initialSecretHolder.reset();
		if(initialSecretHolder!=currentSecretHolder) {
			currentSecretHolder.reset();
		}
		instance.remove();
	}

	public String getContext() {
		return currentSecretHolder.getContext();
	}
	
	public User getUser() {
		return currentSecretHolder.getUser();
	}

}
