package org.gcube.common.uri;

import java.net.URI;
import java.util.List;

import org.gcube.common.scope.api.ScopeProvider;

/**
 * A {@link Mint} that returns <em>scoped URIs</em>, i.e. URIs with a final parameter of the following form:
 * <p>
 * scope=&lt;<em>scope</em>&gt;
 * <p>
 * where <em>scope</em> is the current scope.
 * <p>
 * A {@link ScopedMint} scopes the URIs generated by an existing {@link Mint}, an {@link HttpMint} by default.
 * 
 * @see ScopeProvider
 * 
 * @author Fabio Simeoni
 *
 */
public class ScopedMint implements Mint {
	
	private static final String SCOPE_PARAM_PREFIX = "scope=";
	
	private final Mint mint;
	
	/**
	 * Creates an instance with an {@link HttpMint}.
	 */
	public ScopedMint() {
		this(new HttpMint());
	}

	/**
	 * Creates an instance with a given {@link Mint}.
	 * @param mint the mint.
	 */
	public ScopedMint(Mint mint) {
		this.mint = mint;
	}
	
	
	@Override
	public URI mint(List<String> elements) {
		
		URI uri = mint.mint(elements);
		
		String scope = ScopeProvider.instance.get();
		
		//we cannot assume the mint is the default one, hence we check the scope.
		if (scope == null || scope.isEmpty())
				throw new IllegalStateException("cannot mint a URI outside a scope");
		
		String query = uri.getQuery();
		String param = SCOPE_PARAM_PREFIX+scope;
		query = query==null?param:query+"&"+param;
		
		try {
			return new URI(uri.getScheme(),uri.getAuthority(),uri.getPath(),query,uri.getFragment());
		}
		catch(Exception e) {
			throw new RuntimeException("resource path is invalid",e);
		}
	}
	
	
}
