package org.gcube.event.publisher;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import org.gcube.oidc.rest.JWTToken;
import org.gcube.oidc.rest.OpenIdConnectRESTHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HTTPEventSender implements EventSender {

    protected static final Logger log = LoggerFactory.getLogger(HTTPEventSender.class);

    private URL endpoint;
    private String clientId;
    private String clientSecret;
    private URL tokenURL;

    public HTTPEventSender(URL endpointURL, String clientId, String clientSecret, URL tokenURL) {
        this.endpoint = endpointURL;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.tokenURL = tokenURL;
    }

    @Override
    public void send(Event event) {
        if (log.isDebugEnabled()) {
            log.debug("Starting HTTP POST thread to: " + endpoint);
        }
        try {
            URL eventEndpoint = new URL(endpoint, event.getName());
            new Thread(new HTTPost(eventEndpoint, event.toJSONString())).start();
        } catch (MalformedURLException e) {
            log.error("Cannot compute event endpoint URL. Event name: " + event.getName() + ", base endpoint: "
                    + endpoint, e);
        }
    }

    public class HTTPost implements Runnable {

        private URL endpoint;
        private String jsonString;

        public HTTPost(URL endpoint, String jsonString) {
            this.jsonString = jsonString;
        }

        @Override
        public void run() {
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Getting token for client: " + clientId);
                }
                JWTToken token = OpenIdConnectRESTHelper.queryClientToken(clientId, clientSecret, tokenURL);
                if (log.isTraceEnabled()) {
                    log.trace("Performing HTTP POST");
                }
                HttpURLConnection con = (HttpURLConnection) endpoint.openConnection();
                con.setDoOutput(true);
                con.setDoInput(true);
                con.setRequestProperty("Content-Type", "application/json");
                con.setRequestProperty("Accept", "application/json");
                con.setRequestProperty("Method", "POST");
                con.setRequestProperty("Authorization", token.getAsBearer());
                OutputStream os = con.getOutputStream();
                if (log.isTraceEnabled()) {
                    log.trace("Sending event JSON");
                }
                os.write(jsonString.getBytes("UTF-8"));
                os.close();

                StringBuilder sb = new StringBuilder();
                int httpResultCode = con.getResponseCode();
                if (log.isTraceEnabled()) {
                    log.trace("HTTP Response code: " + httpResultCode);
                }
                if (log.isTraceEnabled()) {
                    log.trace("Reading response");
                }
                boolean ok = true;
                InputStreamReader isr = null;
                if (httpResultCode == HttpURLConnection.HTTP_OK) {
                    isr = new InputStreamReader(con.getInputStream(), "UTF-8");
                } else {
                    ok = false;
                    isr = new InputStreamReader(con.getErrorStream(), "UTF-8");
                }
                BufferedReader br = new BufferedReader(isr);
                String line = null;
                while ((line = br.readLine()) != null) {
                    sb.append(line + "\n");
                }
                br.close();
                if (ok) {
                    if (log.isDebugEnabled()) {
                        log.debug("Event publish OK. Results: " + sb.toString());
                    }
                } else {
                    log.error("Event publish KO. Results: " + sb.toString());
                }
            } catch (Exception e) {
                log.error("POSTing JSON to: " + endpoint, e);
            }
        }
    }

}