package org.gcube.common.gxrest.request;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Objects;
import java.util.Map.Entry;

import org.gcube.common.gxrest.request.GXConnection.HTTPMETHOD;
import org.gcube.common.gxrest.response.inbound.GXInboundResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Common logic for all GXHTTPRequests.
 * 
 * @author Manuele Simi (ISTI CNR)
 *
 */
abstract class GXHTTPCommon<REQUEST> {

	protected static final Logger logger = LoggerFactory.getLogger(GXHTTPStringRequest.class);

	protected GXConnection connection;

	/**
	 * Sets the identity user agent associated to the request.
	 * 
	 * @param agent
	 * @return the request
	 */
	@SuppressWarnings("unchecked")
	public REQUEST from(String agent) {
		this.connection.setAgent(agent);
		return (REQUEST) this;
	}

	/**
	 * Adds s positional path parameter to the request.
	 * 
	 * @param path
	 * @return the request
	 * @throws UnsupportedEncodingException
	 */
	@SuppressWarnings("unchecked")
	public REQUEST path(String path) throws UnsupportedEncodingException {
		this.connection.addPath(path);
		return (REQUEST) this;
	}

	/**
	 * Sets the query parameters for the request.
	 * 
	 * @param parameters
	 *            the parameters that go in the URL after the address and the
	 *            path params.
	 * @return the request
	 * @throws UnsupportedEncodingException
	 */
	@SuppressWarnings("unchecked")
	public REQUEST queryParams(Map<String, String> parameters) throws UnsupportedEncodingException {
		if (Objects.nonNull(parameters) && !parameters.isEmpty()) {
			StringBuilder result = new StringBuilder();
			boolean first = true;
			for (Entry<String, String> parameter : parameters.entrySet()) {
				if (first) {
					first = false;
				} else {
					result.append(GXConnection.PARAM_SEPARATOR);
				}
				result.append(URLEncoder.encode(parameter.getKey(), GXConnection.UTF8));
				result.append(GXConnection.PARAM_EQUALS);
				result.append(URLEncoder.encode(parameter.getValue(), GXConnection.UTF8));
			}
			connection.setQueryParameters(result.toString());
		}
		return (REQUEST) this;
	}
	
	
	/**
	 * Overrides the default security token.
	 * 
	 * @param token
	 */
	public void setSecurityToken(String token) {
		if (!this.connection.isExtCall())
			this.connection.setProperty(org.gcube.common.authorization.client.Constants.TOKEN_HEADER_ENTRY, token);
		else
			throw new UnsupportedOperationException("Cannot set the security token on an external call");
	}

	/**
	 * Add headers to the request.
	 * 
	 * @param name
	 * @param value
	 */
	@SuppressWarnings("unchecked")
	public REQUEST header(String name, String value) {
		this.connection.setProperty(name, value);
		return (REQUEST) this;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#put()
	 */
	public GXInboundResponse put() throws Exception {
		logger.trace("Sending a PUT request...");
		return this.connection.send(HTTPMETHOD.PUT);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#post()
	 */
	public GXInboundResponse post() throws Exception {
		logger.trace("Sending a POST request...");
		return this.connection.send(HTTPMETHOD.POST);
	}
	/**
	 * Sends the GET request to the web application.
	 * 
	 * @return the response
	 */
	public GXInboundResponse get() throws Exception {
		logger.trace("Sending a GET request...");
		return this.connection.send(HTTPMETHOD.GET);
	}

	/**
	 * Sends the DELETE request to the web application.
	 * 
	 * @return the response
	 * @throws Exception
	 */
	public GXInboundResponse delete() throws Exception {
		logger.trace("Sending a DELETE request...");
		return this.connection.send(HTTPMETHOD.DELETE);
	}

	/**
	 * Sends the HEAD request to the web application.
	 * 
	 * @return the response
	 * @throws Exception
	 */
	public GXInboundResponse head() throws Exception {
		logger.trace("Sending a HEAD request...");
		return this.connection.send(HTTPMETHOD.HEAD);
	}

	/**
	 * Clears all the parameter except the address.
	 */
	public void clear() {
		this.connection.reset();
	}

	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#trace()
	 */
	public GXInboundResponse trace() throws Exception {
		logger.trace("Sending a TRACE request...");
		return this.connection.send(HTTPMETHOD.TRACE);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#patch()
	 */
	public GXInboundResponse patch() throws Exception {
		logger.trace("Sending a TRACE request...");
		return this.connection.send(HTTPMETHOD.PATCH);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#options()
	 */
	public GXInboundResponse options() throws Exception {
		logger.trace("Sending an OPTIONS request...");
		return this.connection.send(HTTPMETHOD.OPTIONS);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#connect()
	 */
	public GXInboundResponse connect() throws Exception {
		logger.trace("Sending a CONNECT request...");
		return this.connection.send(HTTPMETHOD.CONNECT);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.gcube.common.gxrest.request.GXHTTP#isExternalCall(boolean)
	 */
	public void isExternalCall(boolean ext) {
		this.connection.setExtCall(ext);
	}
}
