package org.gcube.common.homelibrary.jcr.workspace.folder.items;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.common.homelibrary.home.workspace.folder.items.AquaMapsItem;
import org.gcube.common.homelibrary.home.workspace.folder.items.File;
import org.gcube.common.homelibrary.home.workspace.folder.items.Image;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;


public class JCRAquaMapsItem extends JCRWorkspaceFolderItem implements
		AquaMapsItem {

	private static final String NT_CONTENT         	= "nthl:aquamapsItemContent";
	private static final String MAP_NAME 			= "hl:mapName";
	private static final String MAP_TYPE 			= "hl:mapType";
	private static final String AUTHOR 				= "hl:author"; 
	private static final String NUMBER_OF_SPECIES 	= "hl:numberOfSpecies";
	private static final String BOUNDING_BOX 		= "hl:boundingBox";
	private static final String PSO_THRESHOLD 		= "hl:PSOThreshold";
	private static final String NUMBER_OF_IMAGES 	= "hl:numberOfImages";
	private static final String IMAGES 				= "hl:images";
	
	private final JCRFile file;
	private final List<Image> listImages;
	
	private final String mapName;
	private final String mapType;
	private final String author;
	private final int numberOfSpecies;
	private final String boundingBox; 
	private final float psoThreshold;
	private final int numberOfGeneratedImages;

	
	public JCRAquaMapsItem(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace, node);
		
		Node contentNode = node.getNode(CONTENT);
		this.mapName = contentNode.getProperty(MAP_NAME).getString();
		this.mapType = contentNode.getProperty(MAP_TYPE).getString();
		this.author = contentNode.getProperty(AUTHOR).getString();
		this.numberOfSpecies = (int)contentNode.getProperty(NUMBER_OF_SPECIES).getLong();
		this.boundingBox = contentNode.getProperty(BOUNDING_BOX).getString();
		this.psoThreshold = (float)contentNode.getProperty(PSO_THRESHOLD).getDouble();
		this.numberOfGeneratedImages = (int)contentNode.getProperty(NUMBER_OF_IMAGES).getLong();
		
		file = new JCRFile(contentNode);
		listImages = new ArrayList<Image>();
		Node nodeImages = contentNode.getNode(IMAGES);
		for(NodeIterator iterator =  nodeImages.getNodes(); iterator.hasNext();) {
			Node nodeImage = iterator.nextNode();
			JCRImage jcrImage = new JCRImage(nodeImage);
			listImages.add(jcrImage);
		}
	}
	
	public JCRAquaMapsItem(JCRWorkspace workspace, Node node, String name, String description,
			String mapName, String mapType, String author,
			int numberOfSpecies, String boundingBox, float psoThreshold,
			int numberOfGeneratedImages, InputStream metadata, Map<String,InputStream> images) throws Exception  {
		super(workspace, node, name, description);
		
		Validate.notNull(mapName, "MapName must be not null");
		Validate.notNull(mapType, "MapType must be not null");
		Validate.notNull(author, "Author must be not null");
		Validate.notNull(boundingBox, "Bounding box must be not null");
		Validate.notNull(metadata, "Metadata must be not null");
		Validate.notNull(images, "Images must be not null");
		
		this.mapName = mapName;
		this.mapType = mapType;
		this.author = author;
		this.numberOfSpecies = numberOfSpecies;
		this.boundingBox = boundingBox;
		this.psoThreshold = psoThreshold;
		this.numberOfGeneratedImages = numberOfGeneratedImages;
		

		node.setProperty(FOLDER_ITEM_TYPE, FolderItemType.AQUAMAPS_ITEM.toString());
		Node contentNode = node.addNode(CONTENT,NT_CONTENT);
		contentNode.setProperty(MAP_NAME,mapName);
		contentNode.setProperty(MAP_TYPE,mapType);
		contentNode.setProperty(AUTHOR,author);
		contentNode.setProperty(NUMBER_OF_SPECIES,numberOfSpecies);
		contentNode.setProperty(BOUNDING_BOX,boundingBox);
		contentNode.setProperty(PSO_THRESHOLD,psoThreshold);
		contentNode.setProperty(NUMBER_OF_IMAGES,numberOfGeneratedImages);
		file = new JCRFile(contentNode, new BufferedInputStream(metadata));

		listImages = new ArrayList<Image>();
		Node nodeImages = contentNode.getNode(IMAGES);
		for(Map.Entry<String, InputStream> image : images.entrySet()) {
			String nodeName = Text.escapeIllegalJcrChars(image.getKey());
			Node nodeImage = nodeImages.addNode(nodeName,ContentType.IMAGE.toString());

			JCRImage jcrImage = new JCRImage(nodeImage,image.getValue());
			listImages.add(jcrImage);
		}
		contentNode.getSession().save();
		
	}
	
	

	@Override
	public long getLength() throws InternalErrorException {
		return file.getLength();
	}

	@Override
	public String getMapName() {
		return mapName;
	}

	@Override
	public String getMapType() {
		return mapType;
	}

	@Override
	public String getAuthor() {
		return author;
	}

	@Override
	public int getNumberOfSpecies() {
		return numberOfSpecies;
	}

	@Override
	public String getBoundingBox() {
		return boundingBox;
	}

	@Override
	public float getPsoThreshold() {
		return psoThreshold;
	}

	@Override
	public int getNumberOfGeneratedImages() {
		return numberOfGeneratedImages;
	}

	@Override
	public File getMetadata() throws InternalErrorException {
		return file;
	}

	@Override
	public List<Image> getImages() {
		return listImages;
	}

	@Override
	public FolderItemType getFolderItemType() {
		return FolderItemType.AQUAMAPS_ITEM;
	}

	@Override
	public void saveContent(Node node) throws RepositoryException {
		file.save(node);
	}

	@Override
	public String getMimeType() throws InternalErrorException {
		return null;
	}
	
	

}
