package org.gcube.common.homelibrary.jcr.workspace.folder.items;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import org.apache.commons.lang.Validate;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.common.homelibrary.home.workspace.folder.items.ts.TimeSeries;
import org.gcube.common.homelibrary.home.workspace.folder.items.ts.TimeSeriesInfo;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.common.homelibrary.util.zip.UnzipUtil;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

public class JCRTimeSeries extends JCRWorkspaceFolderItem implements TimeSeries {

	private static final String NT_CONTENT				= "nthl:timeSeriesItemContent";
	
	private static final String TIMESERIES_ID 			= "hl:id";
	private static final String TIMESERIES_TITLE 		= "hl:title";
	private static final String TIMESERIES_DESCRIPTION 	= "hl:description";
	private static final String TIMESERIES_CREATOR 		= "hl:creator";
	private static final String TIMESERIES_CREATED 		= "hl:created";
	private static final String TIMESERIES_PUBLISHER 	= "hl:publisher";
	private static final String TIMESERIES_SOURCE_ID 	= "hl:sourceId";
	private static final String TIMESERIES_SOURCE_NAME 	= "hl:sourceName";
	private static final String TIMESERIES_RIGHTS 		= "hl:rights";
	private static final String TIMESERIES_DIMENSION 	= "hl:dimension";
	private static final String HEADER_LABELS 			= "hl:headerLabels";
	
	

	private final long dataLength;
	
	public JCRTimeSeries(JCRWorkspace workspace, Node node) throws RepositoryException, InternalErrorException  {
		super(workspace, node);
	
		Node contentNode = node.getNode(CONTENT);		
		this.dataLength = contentNode.getProperty(JCRFile.SIZE).getLong();
		
	}
	
	
	public JCRTimeSeries(JCRWorkspace workspace,Node node, String name, String description,
			String timeseriesId, String title, String creator,
			String timeseriesDescription, String timeseriesCreationDate,
			String publisher, String sourceId, String sourceName,
			String rights, long dimension, List<String> headerLabels,
			InputStream compressedCSV) throws RepositoryException, InternalErrorException, RemoteBackendException  {
		super(workspace, node, name, description);
		
		Validate.notNull(timeseriesId, "TimeSeriesId must be not null");
		Validate.notNull(title, "Title must be not null");
		Validate.notNull(creator, "Creator must be not null");
		Validate.notNull(timeseriesDescription, "TimeseriesDescription must be not null");
		Validate.notNull(timeseriesCreationDate, "TimeSeriesCreationDate must be not null");
		Validate.notNull(publisher, "Publisher must be not null");
		Validate.notNull(sourceId, "SourceId must be not null");
		Validate.notNull(sourceName, "SourceName must be not null");
		Validate.notNull(rights, "Rights must be not null");
		Validate.notNull(headerLabels, "HeaderLabels must be not null");
		Validate.notNull(compressedCSV, "CompressedCSV must be not null");
		
		node.setProperty(FOLDER_ITEM_TYPE, FolderItemType.TIME_SERIES.toString());
		Node contentNode = node.addNode(CONTENT,NT_CONTENT);
		contentNode.setProperty(TIMESERIES_CREATED, timeseriesCreationDate);
		contentNode.setProperty(TIMESERIES_CREATOR, creator);
		contentNode.setProperty(TIMESERIES_DESCRIPTION, description);
		contentNode.setProperty(TIMESERIES_DIMENSION, dimension);
		contentNode.setProperty(TIMESERIES_ID, timeseriesId);
		contentNode.setProperty(TIMESERIES_PUBLISHER, publisher);
		contentNode.setProperty(TIMESERIES_RIGHTS, rights);
		contentNode.setProperty(TIMESERIES_SOURCE_ID, sourceId);
		contentNode.setProperty(TIMESERIES_SOURCE_NAME, sourceName);
		contentNode.setProperty(TIMESERIES_TITLE, title);
		contentNode.setProperty(HEADER_LABELS, headerLabels.toArray(new String[headerLabels.size()]));
		
		JCRFile file = new JCRFile(workspace,contentNode,new BufferedInputStream(compressedCSV));
		this.dataLength = file.getLength();
		
	}

	
	@Override
	public long getLength() throws InternalErrorException {
		return dataLength;
	}

	@Override
	public int getNumberOfColumns() {
		
		ArrayList<String> headerLabels = new ArrayList<String>();

		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			for(Value value: contentNode.getProperty(HEADER_LABELS).getValues()) 
				headerLabels.add(value.getString());

		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	

		return headerLabels.size();
	}

	@Override
	public TimeSeriesInfo getTimeSeriesInfo() {
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			
			String timeseriesId = contentNode.getProperty(TIMESERIES_ID).getString();
			String title = contentNode.getProperty(TIMESERIES_TITLE).getString();
			String creator = contentNode.getProperty(TIMESERIES_CREATOR).getString();
			String timeseriesDescription = contentNode.getProperty(TIMESERIES_DESCRIPTION).getString();
			String timeseriesCreationDate = contentNode.getProperty(TIMESERIES_CREATED).getString();
			String publisher = contentNode.getProperty(TIMESERIES_PUBLISHER).getString();
			String sourceId = contentNode.getProperty(TIMESERIES_SOURCE_ID).getString();
			String sourceName = contentNode.getProperty(TIMESERIES_SOURCE_NAME).getString();
			String rights = contentNode.getProperty(TIMESERIES_RIGHTS).getString();
			long dimension = contentNode.getProperty(TIMESERIES_DIMENSION).getLong();
			
			TimeSeriesInfo timeSeriesInfo = new TimeSeriesInfo(timeseriesId, title, creator, timeseriesDescription,
					timeseriesCreationDate, publisher, sourceId, sourceName, rights, dimension);
			return timeSeriesInfo;
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
		
	}

	@Override
	public List<String> getHeaderLabels() {
		
		ArrayList<String> headerLabels = new ArrayList<String>();
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node contentNode = node.getNode(CONTENT);
			for(Value value: contentNode.getProperty(HEADER_LABELS).getValues()) 
				headerLabels.add(value.getString());
			
		} catch (Exception e) {
			throw new RuntimeException(e) ;
		} finally {
			if(session != null)
				session.logout();
		}	
		
		return headerLabels;
	}

	@Override
	public InputStream getData() throws InternalErrorException {
	
		try {
			return UnzipUtil.unzipToTmp(getCompressedData());
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public InputStream getCompressedData() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			
			Node node = session.getNodeByIdentifier(identifier).getNode(CONTENT);
			JCRFile file = new JCRFile(workspace, node);
			return file.getData();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public FolderItemType getFolderItemType() {
		return FolderItemType.TIME_SERIES;
	}


	@Override
	public void saveContent(Node node) throws RepositoryException {
		
	}

	@Override
	public String getMimeType() throws InternalErrorException {
		return null;
	}

}
