package org.gcube.common.homelibrary.jcr.workspace;


import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.jackrabbit.util.Text;
import org.gcube.common.homelibrary.home.Home;
import org.gcube.common.homelibrary.home.HomeManager;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.common.homelibrary.home.workspace.WorkspaceSharedFolder;
import org.gcube.common.homelibrary.home.workspace.accessmanager.ACLType;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.common.homelibrary.home.workspace.usermanager.GCubeGroup;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.workspace.accessmanager.JCRAccessManager;
import org.gcube.common.homelibrary.jcr.workspace.accessmanager.JCRPrivilegesInfo;
import org.gcube.common.homelibrary.jcr.workspace.usermanager.JCRUserManager;
import org.gcube.common.homelibrary.util.WorkspaceUtil;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;
import org.omg.CORBA.OMGVMCID;


public class JCRWorkspaceSharedFolder extends JCRAbstractWorkspaceFolder implements WorkspaceSharedFolder {

	public static final String USERS 			=	"hl:users";
	public static final String MEMBERS 			=	"hl:members";
	public static final String IS_VRE_FOLDER	=	"hl:isVreFolder";
	public static final String DISPLAY_NAME 	=	"hl:displayName";

	private static final String READ_ONLY 		= "jcr:read";
	private static final String WRITE_OWNER 	= "jcr:write";
	private static final String WRITE_ALL 		= "hl:writeAll";
	private static final String ADMINISTRATOR 	= "jcr:all";

	private String applicationName;
	private String destinationFolderId;
	private boolean isVreFolder;
	private String displayName;
	private String absPath;


	private String itemName;
	//resolved groups (just users)
	private List<String> users;


	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node) throws RepositoryException, InternalErrorException {
		super(workspace,node);	

		this.absPath = node.getPath();

		try{
			this.isVreFolder = node.getProperty(IS_VRE_FOLDER).getBoolean();
		}catch (Exception e) {
			this.isVreFolder = false;
		}

		try{
			this.displayName = node.getProperty(DISPLAY_NAME).getString();
		}catch (Exception e) {}

		//		try{
		//			this.groupIds = getGroupIds(node);
		//		}catch (Exception e) {}	

	}

	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node,
			String name, String description, String originalDestinationFolderId, List<String> users, String applicationName, String itemName) throws RepositoryException, InternalErrorException {		

		super(workspace,node,name,description);

		// Create a sharable node below Share root node.
		super.save(node);

		//		System.out.println("originalDestinationFolderId " + originalDestinationFolderId);
		//		System.out.println("destinationFolder " + applicationName);

		this.destinationFolderId = originalDestinationFolderId;
		this.applicationName = applicationName;
		this.itemName = itemName;

		this.users = listUsers(users);

	}



	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node,
			String name, String description, String originalDestinationFolderId, List<String> users, String applicationName, String itemName, String displayName, boolean isVreFolder) throws RepositoryException, InternalErrorException {		

		super(workspace,node,name,description);

		// Create a sharable node below Share root node.
		super.save(node);

		//		System.out.println("originalDestinationFolderId " + originalDestinationFolderId);
		//		System.out.println("destinationFolder " + applicationName);

		this.destinationFolderId = originalDestinationFolderId;
		this.applicationName = applicationName;
		this.itemName = itemName;

		//resolve groupId
		this.users = listUsers(users);

		this.displayName = displayName; 
		this.isVreFolder = isVreFolder;
		this.absPath = node.getPath();

		node.setProperty(IS_VRE_FOLDER,isVreFolder);
		node.setProperty(DISPLAY_NAME,displayName);

		node.getSession().save();

	}


	/**
	 * Resolve groupIds and add user/group to member Node
	 * @param usersList
	 * @return a list of users (no group ids)
	 * @throws InternalErrorException
	 */
	private List<String> listUsers(List<String> usersList) throws InternalErrorException {

		List<String> groups = new ArrayList<String>();
		List<String> users = new ArrayList<String>();

		//get a list of groups
		JCRUserManager userManager = new JCRUserManager();
		List<GCubeGroup> groupsList = userManager.getGroups();		
		for (GCubeGroup group : groupsList){
			groups.add(group.getName());
		}

		Session session = JCRRepository.getSession(); 
		try {	
			for (String user: usersList){	

				//add users to members node
				Node sharedNode = session.getNodeByIdentifier(getId());
				Node memberIds = null;
				try{
					memberIds = sharedNode.getNode(MEMBERS);									
				}catch (Exception e) {
					memberIds = sharedNode.addNode(MEMBERS);
					session.save();
					//					e.printStackTrace();
					logger.warn(" Members_ids Node already exists");
				}

				try{
					memberIds.getNode(user);					
				}catch (Exception e) {
					if(!user.endsWith("-Manager")){
						memberIds.addNode(user);
						session.save();
						//					e.printStackTrace();
						//					System.out.println(user + " is already in membersList");
						logger.warn(user + " add to membersList");
					}
				}

				//resolve groups
				if (groups.contains(user)) {
					//if a user is a group, resolve group
					List<String> userList = workspace.resolveGroupId(user);
					users.addAll(userList);
				}else
					users.add(user);
			}
			session.save();
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

		return users;
	}





	@Override
	public void save(Node node) throws RepositoryException {

		//				System.out.println("save");
		//				System.out.println("users " + users.toString());
		try {
			// The save method creates a clone of the sharable node
			//for the owner and all users below user roots.
			//			if (!isVreFolder())
			addUser(node, workspace.getOwner().getPortalLogin(), destinationFolderId);
			//			else
			//				addUser(node,workspace.getOwner().getPortalLogin(), getWorkspace().getMySpecialFoldersId());

			logger.trace("Share with " + users.toString());
			for (String user : users) {

				//				(this.users = workspace.resolveGroupId(groupId);)
				//							System.out.println("user " + user);
				HomeManager homeManager = workspace.getHome().getHomeManager();
				//GCUBEScope scopeOwner = workspace.getOwner().getScope();
				Home home = homeManager.getHome(user);

				if (applicationName==null){
					//								System.out.println("save application name == null");

					if (isVreFolder()){
						//						addUser(node, user, home.getWorkspace().getMySpecialFoldersId());
						addUser(node, user, home.getWorkspace().getMySpecialFolders().getId());
					}
					else
						addUser(node, user, home.getWorkspace().getRoot().getId());
				}
				else{
					//									System.out.println("save application name != null");
					addUser(node, user, home.getDataArea().getApplicationRoot(applicationName).getId());
					//					System.out.println("home.getDataArea().getApplicationRoot(applicationName).getId() " + home.getDataArea().getApplicationRoot(applicationName).getId());
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new RepositoryException(e);
		}
	}


	public Node getUserNode(Node node, String user) throws RepositoryException,
	InternalErrorException {

		Node usersNode = node.getNode(USERS);
		logger.trace("Looking for user: " + user + " in node: " + usersNode.getPath());
		String value = usersNode.getProperty(user).getString();

		String[] values = value.split(workspace.getPathSeparator());
		if (values.length < 2)
			throw new InternalErrorException("Path node corrupt");

		String parentId = values[0];
		String nodeName = values[1];
		Node parentNode = node.getSession().getNodeByIdentifier(parentId);

		return node.getSession().getNode(parentNode.getPath() + 
				workspace.getPathSeparator() + Text.escapeIllegalJcrChars((nodeName)));

	}

	private Node getUserNode(Node node) throws RepositoryException,
	InternalErrorException {
		return getUserNode(node, workspace.getOwner().getPortalLogin());
	}

	private String getNodeName(Node node) throws RepositoryException,
	InternalErrorException {

		String[] names = node.getPath().split(
				workspace.getPathSeparator());

		return names[names.length - 1];
	}

	@Override
	public String getName() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node userNode = getUserNode(sharedNode);
			return getNodeName(userNode);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}		
	}

	@Override
	public void internalRename(Node node, String newName) throws ItemAlreadyExistException, InternalErrorException {


		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {

			Node userNode = getUserNode(node);

			if (workspace.exists(nodeNewName, userNode.getParent().getIdentifier())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			String newPath = userNode.getParent().getPath() 
					+ workspace.getPathSeparator() + nodeNewName;

			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.RENAMED.toString());
			node.getSession().save();

			String path = userNode.getPath();
			node.getSession().getWorkspace().move(path, newPath);

			Node usersNode = node.getNode(USERS);
			String value = userNode.getParent().getIdentifier() + workspace.getPathSeparator() + newName;
			usersNode.setProperty(workspace.getOwner().getPortalLogin(), value);

			node.getSession().save();

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		} finally {

		}

	}

	@Override
	public void internalMove(Node destinationFolderNode) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {

		try {

			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getIdentifier());

			Session session = destinationFolderNode.getSession();

			Node node = session.getNodeByIdentifier(identifier);

			if (workspace.exists(node.getName(), destinationFolderNode.getIdentifier())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + node.getName() + " already exists");
			}
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.MOVED.toString());
			node.getSession().save();

			//	System.out.println("internalMove:getUserNode ");
			Node userNode = getUserNode(node);
			String userNodeName = getNodeName(userNode);

			String newPath = destinationFolderNode.getPath() 
					+ workspace.getPathSeparator() + userNodeName;

			String value = destinationFolderNode.getIdentifier() +
					workspace.getPathSeparator() + userNodeName;

			session.getWorkspace().clone(session.getWorkspace().getName(),
					node.getPath(), newPath, false);


			Node usersNode = node.getNode(USERS);
			usersNode.setProperty(workspace.getOwner().getPortalLogin(), value);

			session.removeItem(userNode.getPath());

			session.save();
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} 
	}


	public Node unShareNode(Node sharedNode, Node destinationNode, boolean localCopy) throws RepositoryException,
	InternalErrorException, InsufficientPrivilegesException, ItemAlreadyExistException,
	WorkspaceFolderNotFoundException, WrongDestinationException, ItemNotFoundException,
	RemoteBackendException {

		//		System.out.println("localCopy? " + localCopy);
		logger.debug("unShare Node: "+ sharedNode.getPath() + " -  by user: " + workspace.getOwner().getPortalLogin());

		Session session = sharedNode.getSession();
		Node userNode = getUserNode(sharedNode);

		String folderName = getNodeName(userNode);
		String description = getDescription();

		// shareNode parent it's the same of destinationNode
		if (destinationNode.getIdentifier().equals(userNode.getParent().getIdentifier())) {
			removeUserSharedFolder(sharedNode);
		}

		if (localCopy){
			String unSharedFolderId = workspace.createFolder(folderName, description, destinationNode.getIdentifier()).getId();
			Node nodeFolder = session.getNodeByIdentifier(unSharedFolderId);

			for (NodeIterator iterator = sharedNode.getNodes(); iterator.hasNext();) {
				Node child = (Node) iterator.next();

				if (!child.getName().startsWith(JCRRepository.HL_NAMESPACE) 
						&& !child.getName().startsWith(JCRRepository.JCR_NAMESPACE)
						&& !child.getName().startsWith(JCRRepository.REP_NAMESPACE)) {			
					session.getWorkspace().copy(child.getPath(), nodeFolder.getPath() 
							+ workspace.getPathSeparator() + child.getName());
				}
			}

			//		session.save();
			//TODO temporarily 

			logger.debug("copyremotecontent from "+  nodeFolder.getPath() + " to " + destinationNode.getPath());
			workspace.copyRemoteContent(nodeFolder, destinationNode);

			session.save();		

			JCRWorkspaceItem itemUnshared = (JCRWorkspaceItem) workspace.getItem(unSharedFolderId);
			//add UNSHARE operation in History
			itemUnshared.setUnshareHistory(workspace.getOwner().getPortalLogin());
			//change owner
			itemUnshared.setOwnerToCurrentUser(itemUnshared);
			// Check if it's possible remove shared node completely
			//			checkRemoveSharedFolder(sharedNode);

			return nodeFolder;
		}
		return null;
	}


	//	private void checkRemoveSharedFolder(Node sharedNode) throws  InternalErrorException, RemoteBackendException, RepositoryException{
	//		logger.trace("checkRemoveSharedFolder");
	//		if (getUsers().isEmpty()) {
	//			logger.trace("shared Node is empty, will be removed: " + sharedNode.getPath());
	//			try {
	//				//TODO temporarily solution
	//				try{		
	//					GCUBEStorage.removeRemoteFolder(sharedNode.getPath(), workspace.getOwner().getPortalLogin());
	//					//						GCUBEStorage.removeRemoteFolder(sharedNode.getPath());
	//				}catch (Exception e) {
	//					logger.warn(sharedNode.getName() + " not available", e);
	//				}			
	//				sharedNode.remove();
	//				sharedNode.getSession().save();
	//			} catch (Exception e) {
	//				throw new InternalErrorException(e);
	//			}
	//		}			
	//	}

	@Override
	public Node internalCopy(Node nodeFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{

		Session session = nodeFolder.getSession();

		Node node = session.getNodeByIdentifier(identifier);
		String pathNewNode = nodeFolder.getPath()
				+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);

		try {

			if(session.getNode(pathNewNode) != null)
				throw new ItemAlreadyExistException(newName + " already exist");

		} catch(RepositoryException e) {}


		String description = getDescription();
		try {

			String unSharedFolderId = workspace.createFolder(Text.escapeIllegalJcrChars(newName), description, nodeFolder.getIdentifier()).getId();
			Node newNodeFolder = session.getNodeByIdentifier(unSharedFolderId);

			for (NodeIterator iterator = node.getNodes(); iterator.hasNext();) {
				Node child = (Node) iterator.next();

				if (!child.getName().startsWith(JCRRepository.HL_NAMESPACE) 
						&& !child.getName().startsWith(JCRRepository.JCR_NAMESPACE)
						&& !child.getName().startsWith(JCRRepository.REP_NAMESPACE)) {				

					session.getWorkspace().copy(child.getPath(), newNodeFolder.getPath() 
							+ workspace.getPathSeparator() + child.getName());
				}
			}

			return newNodeFolder;
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}

	@Override
	public WorkspaceFolder getParent() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			//	System.out.println("getParent:getUserNode");
			return workspace.getParent(getUserNode(node));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public String getPath(Node node) throws RepositoryException, InternalErrorException {

		Node userNode =  getUserNode(node);
		return ((JCRWorkspaceFolder)getParent(userNode)).getPath(userNode.getParent())
				+ workspace.getPathSeparator() + getNodeName(userNode);		
	}



	//	private void checkRemoveSharedFolder(Node sharedNode) throws  InternalErrorException, RemoteBackendException {
	//
	//		if (getUsers().isEmpty()) {
	//
	//			try {
	//				//TODO temporarily solution
	//				try{
	//					workspace.removeRemoteContent(sharedNode, null);
	//				}catch (Exception e) {
	//					logger.warn(sharedNode.getName() + " payload not available", e);
	//				}
	//
	//				sharedNode.remove();
	//				sharedNode.getSession().save();
	//			} catch (RepositoryException e) {
	//				throw new InternalErrorException(e);
	//			}
	//		}
	//
	//	}


	@Override
	public void remove() throws InternalErrorException,
	InsufficientPrivilegesException {

		logger.debug("remove shared");
		logger.debug("portalLogin: " + workspace.getOwner().getPortalLogin() + " - owner: "+ getOwner().getPortalLogin());
		Session session = JCRRepository.getSession();
		try {

			Node sharedNode = session.getNodeByIdentifier(getId());

			if (!JCRPrivilegesInfo.canDelete(getOwner().getPortalLogin(), workspace.getOwner().getPortalLogin(), getSharePath(), true)) 
				throw new InsufficientPrivilegesException("Insufficient Privileges to remove the node");
			if (isVreFolder())
				throw new InternalErrorException("A VRE folder cannot be removed");
			if (sharedNode.getPath().equals(workspace.mySpecialFoldersPath))
				throw new InternalErrorException("This folder cannot be removed");

			try {

				//	System.out.println("trash id: " + trashFolderId);
				WorkspaceFolder unsharedFolder = unShare();

				logger.trace("unsharedFolder: " + unsharedFolder.getPath());
				Node usharedNode = session.getNodeByIdentifier(unsharedFolder.getId());

				workspace.moveToTrash(usharedNode);

				//				logger.trace("remove clones");
				//				removeClones(sharedNode);

				//				logger.trace("remove sharedNode: " + sharedNode.getPath());
				//				sharedNode.remove();
				session.save();

			} catch (WrongDestinationException | ItemAlreadyExistException
					| WorkspaceFolderNotFoundException | ItemNotFoundException e) {
				throw new InternalErrorException(e);
			}


			//			removeUserSharedFolder(sharedNode);
			//			checkRemoveSharedFolder(sharedNode);
			//			System.out.println(getSharePath() + " is shared and can be deleted by " + JCRRepository.portalLogin);
			//			System.out.println("remove");

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}



	@Override
	public List<String> getMembers() throws InternalErrorException {

		ArrayList<String> list = new ArrayList<String>();
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());
			Node members = node.getNode(MEMBERS);
			NodeIterator children = members.getNodes();
			while (children.hasNext()){
				String name = children.nextNode().getName();
				if (!name.startsWith(JCRRepository.JCR_NAMESPACE) && !name.startsWith(JCRRepository.HL_NAMESPACE)){
					list.add(name);
				}
			}

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}


	@Override
	public List<String> getUsers() throws InternalErrorException {

		ArrayList<String> list = new ArrayList<String>();
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());
			Node usersNode = node.getNode(USERS);
			for (PropertyIterator iterator = usersNode.getProperties(); iterator.hasNext();) {
				Property property  = iterator.nextProperty();
				String name = property.getName();
				if (!name.startsWith(JCRRepository.JCR_NAMESPACE) &&
						!name.startsWith(JCRRepository.HL_NAMESPACE))
					list.add(name);
			} 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}

	private void addUser(Node sharedNode, String user, String destinationFolderId) throws InternalErrorException, RepositoryException {

		logger.trace("addUser("+sharedNode.getPath()+", " + user +", " + destinationFolderId +");");

		Session	session = sharedNode.getSession();
		//		Session session = JCRRepository.getSession(); 
		try {


			//		Node destinationNode = session.getNodeByIdentifier(destinationFolderId);

			HomeManager homeManager = workspace.getHome().getHomeManager();
			WorkspaceFolder userRoot = (WorkspaceFolder)homeManager.getHome(user).getWorkspace().getItem(destinationFolderId);

			Node rootNode =  session.getNodeByIdentifier(userRoot.getId());

			String sharedFolderName = WorkspaceUtil.getUniqueName(
					sharedNode.getProperty(TITLE).getString(),
					userRoot);

			String pathUser = null;

			Node usersNode = sharedNode.getNode(USERS);

			if (applicationName != null){

				pathUser = rootNode.getPath() + workspace.getPathSeparator() + sharedFolderName;
				//				itemName
				session.getWorkspace().clone(session.getWorkspace().getName(),
						sharedNode.getPath()+ workspace.getPathSeparator() + itemName, pathUser, false);


				logger.trace("clone from " + sharedNode.getPath()+ workspace.getPathSeparator() + itemName + " to "+ pathUser);

				session.save();

			}else {		

				pathUser = rootNode.getPath() + workspace.getPathSeparator() + sharedFolderName;
				try {
					if (usersNode.getProperty(user) != null){
						return;
					}
				} catch (PathNotFoundException e) {
					logger.debug("User is not present");
				}


				session.getWorkspace().clone(session.getWorkspace().getName(),
						sharedNode.getPath(), pathUser, false);

				logger.trace("Clone from " + sharedNode.getPath() + " to "+ pathUser);

				session.save();

				//				Node path = session.getNode(pathUser);


			}

			String value = userRoot.getId() + workspace.getPathSeparator() 
					+ sharedFolderName;

			logger.trace("usersNode: " + usersNode.getPath() + " - set value " + value + " to: " + user);
			usersNode.setProperty(user, value);
			//			usersNode.getSession().save();
			session.save();


		} catch (Exception e) {
			//			e.printStackTrace();
			throw new InternalErrorException(e);
		}
		//		finally{
		//			session.logout();
		//		}

		//		System.out.println("*********** " + sharedNode.getPath()+" add to " + user);

	}


	@Override
	public void addUser(String user) throws InsufficientPrivilegesException,
	InternalErrorException {

		//System.out.println("addUser(String user) user: " + user);
		Session session = JCRRepository.getSession(); 
		try {	 
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node usersNode = sharedNode.getNode(USERS);

			try {
				if (usersNode.getProperty(user) != null){
					logger.trace(user + " is already in share");
					return;
				}
			} catch (PathNotFoundException e) {
				logger.debug("User "+ user + " is not present");
			}

			HomeManager homeManager = workspace.getHome().getHomeManager();
			//GCUBEScope scopeOwner = workspace.getOwner().getScope();
			Home home = homeManager.getHome(user);
			//	System.out.println("addUser(sharedNode,user, home.getWorkspace().getRoot().getId())");
			if (isVreFolder)
				addUser(sharedNode, user, home.getWorkspace().getMySpecialFolders().getId());
			else
				addUser(sharedNode, user, home.getWorkspace().getRoot().getId());


		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.SHARED_FOLDER;
	}

	@Override
	public WorkspaceFolder unShare() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);	
			Node userNode = getUserNode(node);	
			//			WorkspaceItem userItem = workspace.getItem(userNode.getIdentifier());

			//			Node unsharedNode = unShareNode(node, userNode.getParent());
			//			Node rootNode = unShareNode(node, userNode.getParent());
			Node unsharedNode = null;
			JCRWorkspaceFolder folder = null;

			boolean flag = false;
			if (getOwner().getPortalLogin().equals(workspace.getOwner().getPortalLogin()) || (getACLUser().equals(ACLType.ADMINISTRATOR))){
				flag= true;
			}
			try{

				unsharedNode = unShareNode(node, userNode.getParent(), flag);
			}catch (Exception e) {
				// TODO: handle exception
			}

			if (unsharedNode!=null){
				folder = new JCRWorkspaceFolder(workspace, unsharedNode);

				logger.trace("remove clones");
				removeClones(node);

				logger.trace("remove sharedNode: " + node.getPath());
				node.remove();
				session.save();
			}else {

				//remove user from ACL
				JCRAccessManager accessManager = new JCRAccessManager();
				List<String> userToRemove = new ArrayList<String>();
				userToRemove.add(workspace.getOwner().getPortalLogin());
				accessManager.deleteAces(getSharePath(), userToRemove);	
			}
			//create a local copy
			return folder;

		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}




	@Override
	public WorkspaceFolder unShare(String user) throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);	
			Node userNode = node.getNode(USERS);

			if (!user.startsWith(JCRRepository.JCR_NAMESPACE) &&
					!user.startsWith(JCRRepository.HL_NAMESPACE) && 
					!user.equals(workspace.getOwner().getPortalLogin()))	{	
				//remove clone
				try{
					Node cloneNode = getUserNode(node, user);
					logger.trace("remove clone " + cloneNode.getPath());
					session.removeItem(cloneNode.getPath());
					session.save();
				}catch (Exception e) {
					logger.error("Error removing clone ");
				}
				//remove user in userNode
				try{						
					userNode.setProperty(user, (String)null);
					userNode.getSession().save();
					logger.trace(user + "  deleted from share " + node.getPath());
				}catch (Exception e) {					
					logger.error("Error removing user from node Users ");
				}
			}

			//remove user from ACL
			JCRAccessManager accessManager = new JCRAccessManager();
			List<String> userToRemove = new ArrayList<String>();
			userToRemove.add(user);
			accessManager.deleteAces(getSharePath(), userToRemove);	

			//set history			
			setUnshareHistory(user);
			//			return new JCRWorkspaceFolder(workspace, unsharedNode);
			return null;
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	//	private void removeSingleUser(String user, Node node) {
	//		if (!user.startsWith(JCRRepository.JCR_NAMESPACE) &&
	//				!user.startsWith(JCRRepository.HL_NAMESPACE) && 
	//				!user.equals(workspace.getOwner().getPortalLogin()))	{	
	//			//remove clones
	//			try{
	//				Node cloneNode = getUserNode(node, user);
	//				logger.trace("remove clone " + cloneNode.getPath());
	//				node.getSession().removeItem(cloneNode.getPath());
	//				node.getSession().save();
	//			}catch (Exception e) {
	//				logger.error("Error removing clone ");
	//			}
	//			//remove user in userList
	//			try{						
	//				Node userNode = getUserNode(node);
	//				userNode.setProperty(user, (String)null);
	//				userNode.getSession().save();
	//				logger.trace(user + "  deleted from share " + node.getPath());
	//			}catch (Exception e) {					
	//				logger.error("Error removing user from node Users ");
	//			}
	//		}
	//
	//	}

	@Override
	public WorkspaceSharedFolder share(List<String> usersList) throws InsufficientPrivilegesException,
	WrongDestinationException, InternalErrorException {

		List<String> userIds = listUsers(usersList);

		for (String user : userIds) {
			addUser(user);
		}

		//set history
		//		try{
		//			setShareHistory(userIds, workspace.getOwner().getPortalLogin());
		//		}catch (Exception e) {
		//			// TODO: handle exception
		//		}



		//	System.out.println("return WorkspaceSharedFolder");
		return this;
	}



	@Override
	public String getName(String user) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node userNode = getUserNode(sharedNode, user);
			return getNodeName(userNode);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}

	//get path in root folder /Share (e.g. /Share/52fb2641-c9ad-4008-8205-8ea3359dc271)
	public String getSharePath() throws InternalErrorException{
		Session session = JCRRepository.getSession();
		String path = null;
		try {
			Node node = session.getNodeByIdentifier(identifier);
			path = node.getPath();	
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return path;
	}

	@Override
	public void setACL(List<String> users, ACLType privilege) throws InternalErrorException {

		List<String> notAdmins = new ArrayList<String>(users);
		List<String> admins = getAdministrators();
		if (admins!=null){
			notAdmins.removeAll(admins);
			logger.info("notAdmin users " + notAdmins.toString());
		}else
			logger.info("No Administrators on " + absPath);

		String absPath = null;
		boolean flag = false;
		JCRAccessManager accessManager = new JCRAccessManager();

		int i = 0;
		while ((flag==false) && (i<3)){
			i++;
			try{
				absPath = getSharePath();

				if (absPath == null)
					throw new InternalErrorException("Absolute path cannot be null setting ACL");

				switch(privilege){

				case READ_ONLY:
					if (notAdmins.size()>0)
						flag = accessManager.setReadOnlyACL(notAdmins, absPath);		
					break;
				case WRITE_OWNER:	
					if (notAdmins.size()>0)
						flag = accessManager.setWriteOwnerACL(notAdmins, absPath);		
					break;
				case WRITE_ALL:
					if (notAdmins.size()>0)
						flag = accessManager.setWriteAllACL(notAdmins, absPath);	
					break;
				case ADMINISTRATOR:
					flag = accessManager.setAdminACL(users, absPath);	
					break;
				default:
					break;
				}

				if (flag==false)
					Thread.sleep(1000);
				
			}catch (Exception e) {
				logger.error("an error occurred setting ACL on: " + absPath);
			}
		}


		logger.info("Has ACL been modified correctly for users " + users.toString() + "in path " + absPath + "? " + flag);
		//set administators

		if (!isVreFolder()){
			//				System.out.println("***getOwner().getPortalLogin(): " + workspace.getOwner().getPortalLogin());
			//			if (admins!=null){
			//				try{
			//					accessManager.setAdminACL(admins, absPath);
			//				}catch (Exception e) {
			//					logger.error("Error setting owner as administator on " + absPath);
			//				}
			//			}
			//			else{

			if (admins==null){	
				boolean isSet = false;

				int j = 0;
				while ((isSet==false) && (j<3)){
					j++;
					try{
						String owner = workspace.getOwner().getPortalLogin();
						List<String> adminList =new ArrayList<String>();
						adminList.add(owner);
						logger.info("Set " + owner + " ad administrator");
						isSet = accessManager.setAdminACL(adminList, absPath);
						logger.info("Has ACL been modified correctly for users " + adminList.toString() + "in path " + absPath + "? " + isSet);
						if (isSet==false)
							Thread.sleep(1000);
					}catch (Exception e) {
						logger.error("Error setting administators on " + absPath);
					}
				}
			}
		}
		//		else{
		//			if (admins!=null){
		//				try{
		//					accessManager.setAdminACL(admins, absPath);
		//				}catch (Exception e) {
		//					logger.error("Error setting owner as administator on " + absPath);
		//				}
		//			}
		//
		//		}

	}

	@Override
	public void deleteACL(List<String> users) throws InternalErrorException {

		String absPath = null;
		JCRAccessManager accessManager = new JCRAccessManager();
		try{

			absPath = getSharePath();
			accessManager.deleteAces(absPath, users);	
			//			setAcesHistory();

		}catch (Exception e) {
			logger.error("an error occurred setting ACL on: " + absPath);
		}

	}


	@Override
	public Map<ACLType, List<String>> getACLOwner() throws InternalErrorException {
		
		String absPath = null;
		JCRAccessManager accessManager = null;
		Map<String, List<String>> aclMap = null;		
		Map<ACLType, List<String>> map = new HashMap<ACLType, List<String>>();

		try{
			accessManager = new JCRAccessManager();
			absPath = getSharePath();
			aclMap  = accessManager.getEACL(absPath);

			Set<String> keys = aclMap.keySet();

			for (final String user : keys){

				JCRUserManager um = new JCRUserManager();
				GCubeGroup group = null;
				try{

					//if the user is a group and this is empty, skip
					group = um.getGroup(user);
					if (group!=null){
						if (group.getMembers().isEmpty()){
							continue;	
						}
					}

					//					List<String> acl = aclMap.get(user);
					ACLType aclType = getACLTypeByKey(aclMap.get(user));
					List<String> users = null;
					try{					
						users = map.get(aclType);

						users.add(user);
						map.put(aclType, users);

					}catch (Exception e) {

						//if the key does not exist, create a new list
						map.put(aclType, new ArrayList<String>(){
							private static final long serialVersionUID = 1L;
							{add(user);}});
					}


				}catch (Exception e) {
					logger.error(e.getMessage());
				}
			} 

		}catch (Exception e) {
			logger.error("an error occurred setting ACL on: " + absPath);
		}

		return	map;
	}



	private ACLType getACLTypeByKey(List<String> list) {
		switch(list.get(0)){

		case READ_ONLY:
			return ACLType.READ_ONLY;	

		case WRITE_OWNER:	
			return ACLType.WRITE_OWNER;		

		case WRITE_ALL:
			return ACLType.WRITE_ALL;	

		case ADMINISTRATOR:
			return ACLType.ADMINISTRATOR;		

		default:
			return ACLType.READ_ONLY;

		}		
	}

	public void removeClones(Node sharedNode) throws InternalErrorException, RepositoryException {

		Session session = sharedNode.getSession();
		try {

			Node userNode = sharedNode.getNode(USERS);
			PropertyIterator usersList = userNode.getProperties();
			while(usersList.hasNext()){

				String user = usersList.nextProperty().getName();
				if (user.startsWith("jcr:"))
					continue;
				//				System.out.println("remove clone for user " + user);
				logger.trace("user " + user);
				logger.trace("workspace.getOwner().getPortalLogin() " + workspace.getOwner().getPortalLogin());
				//				logger.trace("-> user: " + user);
				//  &&  (!user.equals(workspace.getOwner().getPortalLogin())
				if (!user.startsWith(JCRRepository.JCR_NAMESPACE) &&
						!user.startsWith(JCRRepository.HL_NAMESPACE))	{	
					//remove clones
					try{
						Node cloneNode = getUserNode(sharedNode, user);
						logger.trace("remove clone " + cloneNode.getPath());
						session.removeItem(cloneNode.getPath());
						session.save();
					}catch (Exception e) {
						//						e.printStackTrace();
						logger.error("Error removing clone " + e);
					}

					//					System.out.println("removeClones");
					//					remove user in userList
					try{		
						logger.trace("remove user in userList");
						logger.trace(user + "  deleted from share " + sharedNode.getPath());
						userNode.setProperty(user, (String)null);
						userNode.getSession().save();
						logger.trace(user + "  deleted from share " + sharedNode.getPath());
					}catch (Exception e) {		
						//						e.printStackTrace();
						logger.error("Error removing user from node Users " + e);
					}
					//remove user in memberList
					try{
						Node memberNode = sharedNode.getNode(MEMBERS);
						memberNode.getNode(user).remove();
						memberNode.getSession().save();
					}catch (Exception e) {
						logger.error("Error removing user from members node");
					}
				}
			}
			session.save();
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			//			session.logout();
		}

	}


	public void removeUserSharedFolder(Node sharedNode) throws InternalErrorException, RepositoryException {
		Session session = sharedNode.getSession();
		try {

			Node userNode = getUserNode(sharedNode);

			// Remove sharedNode from user workspace
			session.removeItem(userNode.getPath());
			session.save();

			// Remove user in sharingSet
			try{
				Node usersNode = sharedNode.getNode(USERS);
				usersNode.setProperty(workspace.getOwner().getPortalLogin(), (String)null);
				usersNode.getSession().save();
			}catch (Exception e) {
				logger.error("Error removing user from users node");
			}

			// Remove user in member node
			try{
				Node memberNode = sharedNode.getNode(MEMBERS);
				memberNode.getNode(workspace.getOwner().getPortalLogin()).remove();
				memberNode.getSession().save();
			}catch (Exception e) {
				logger.error("Error removing user from members node");
			}


		}catch (Exception e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public boolean isVreFolder() {
		return isVreFolder;
	}

	@Override
	public String getDisplayName() {
		return displayName;
	}


	//	public List<String> getGroupIds(Node node) throws PathNotFoundException, RepositoryException {
	//		List<String> groupIdsList = null;
	//
	//		try{
	//			groupIdsList = new ArrayList<String>();
	//			NodeIterator groups = node.getNode(GROUP_IDS).getNodes();
	//			while (groups.hasNext())
	//				groupIdsList.add(groups.nextNode().getName());
	//		}catch (Exception e) {
	//			// TODO: handle exception
	//		}
	//
	//		return groupIdsList;
	//	}

	//	@Override
	//	public List<String> getGroupIds() {
	//		return groupIds;
	//	}

	@Override
	public ACLType getACLUser() throws InternalErrorException {
		return JCRPrivilegesInfo.getACLByUser(workspace.getOwner().getPortalLogin(), getAbsPath());
	}


	private String getAbsPath() {
		return absPath;
	}


	@Override
	public boolean setAdmin(final String username) throws InsufficientPrivilegesException,
	InternalErrorException {

		//		if (!getUsers().contains(username))
		try {
			share(new ArrayList<String>(){/**
			 * 
			 */
				private static final long serialVersionUID = 1L;

				{add(username);}});
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
		if (isAdmin(workspace.getOwner().getPortalLogin())){
			try {
				List<String> administator = new ArrayList<String>();
				administator.add(username);
				this.setACL(administator, ACLType.ADMINISTRATOR);

			}catch (Exception e) {
				logger.error(e.getMessage());
				return false;
			}
			return true;
		}
		throw new InsufficientPrivilegesException("Insufficient Privileges to set administrators");

	}

	@Override
	public List<String> getAdministrators() throws InternalErrorException {
		List<String> list = null;
		try{
			list = getACLOwner().get(ACLType.ADMINISTRATOR);
		}catch (Exception e) {
			logger.error("no administrators");
		}
		return list;
	}


	public boolean isAdmin(String username) throws InternalErrorException {
		if (getACLUser().equals(ACLType.ADMINISTRATOR))
			return true;
		return false;
	}





}
