package org.gcube.common.homelibrary.jcr.workspace;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockException;
import javax.jcr.lock.LockManager;

import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.homelibrary.home.User;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.Properties;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.common.homelibrary.home.workspace.accounting.AccountingEntry;
import org.gcube.common.homelibrary.home.workspace.accounting.AccountingEntryRead;
import org.gcube.common.homelibrary.home.workspace.acl.Capabilities;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.common.homelibrary.home.workspace.folder.FolderItem;
import org.gcube.common.homelibrary.jcr.JCRUser;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.common.homelibrary.jcr.resolver.UriResolverReaderParameter;
import org.gcube.common.homelibrary.jcr.shortner.UrlShortener;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntry;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryAddACL;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryCreate;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryDeleteACL;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryModifyACL;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryPaste;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryRead;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryRenaming;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryShare;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryType;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryUnshare;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingEntryUpdate;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryAdd;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryCut;
import org.gcube.common.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryRemoval;
import org.gcube.common.homelibrary.jcr.workspace.folder.items.JCRFile;
import org.gcube.common.homelibrary.jcr.workspace.util.HttpRequestUtil;
import org.gcube.common.homelibrary.jcr.workspace.util.StringUtil;
import org.gcube.common.homelibrary.util.WorkspaceUtil;
import org.gcube.common.scope.api.ScopeProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class JCRWorkspaceItem implements WorkspaceItem {


	public static final String TITLE 				= "jcr:title"; 
	public static final String CREATED 				= "jcr:created";
	public static final String LAST_MODIFIED 		= "jcr:lastModified";
	public static final String OWNER 				= "hl:owner";
	public static final String PORTAL_LOGIN  		= "hl:portalLogin";
	public static final String ISVREFOLDER 			= "hl:isVreFolder";	
	public static final String DISPLAY_NAME 		= "hl:displayName";

	protected static final String LAST_MODIFIED_BY 	= "jcr:lastModifiedBy";
	protected static final String DESCRIPTION 		= "jcr:description";
	protected static final String LAST_ACTION 		= "hl:lastAction";



	protected static final String READERS 			= "hl:readers";
	protected static final String NT_READERS		= "nthl:readersSet";

	protected static final String ACCOUNTING		= "hl:accounting";
	protected static final String NT_ACCOUNTING		= "nthl:accountingSet";

	public static final String USERS 				="hl:users";

	//private static final String SCOPE               = "hl:scope";

	private static final String USER_ID 			= "hl:uuid";
	private static final String NT_USER				= "nthl:user";
	//private static final String NT_BOOKMARK 		= "nthl:bookmark";

	public static final String NT_WORKSPACE_FILE 				= "nthl:externalFile";
	public static final String NT_WORKSPACE_IMAGE 				= "nthl:externalImage";
	public static final String NT_WORKSPACE_PDF_FILE 			= "nthl:externalPdf";

	private static final String CONTENT							= "jcr:content";
	private static final String READ							= "nthl:accountingEntryRead";

	protected final JCRWorkspace workspace;
	protected String identifier;

	protected String userId;
	protected String portalLogin;

	protected Calendar creationDate;
	protected JCRProperties properties;


	//	public int deep;


	protected static Logger logger = LoggerFactory.getLogger(JCRWorkspaceItem.class);

	public JCRWorkspaceItem(JCRWorkspace workspace, Node node) throws RepositoryException {
		//System.out.println("here 1");
		//		System.out.println("JCRWorkspaceItem --> " +node.getPath());
		this.workspace = workspace;
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

		//		node.getProperties(new String [] {CREATED, TITLE, DESCRIPTION, "hl:portalLogin"});
		//get Owner
		try{
			this.portalLogin = node.getProperty("hl:portalLogin").getString();
		}catch (Exception e) {
			// TODO: handle exception
		}		

		//		this.userId = portalLogin;

		if (portalLogin==null){
			Node nodeOwner;
			try {
				nodeOwner = node.getNode(OWNER);
				this.userId = nodeOwner.getProperty(USER_ID).getString();
				this.portalLogin = nodeOwner.getProperty(PORTAL_LOGIN).getString();
				//				node.setProperty(PORTAL_LOGIN, portalLogin);
				node.getSession().save();
			} catch (PathNotFoundException e) {
				try {
					nodeOwner = node.addNode(OWNER,NT_USER);
					setOwnerNode(nodeOwner);		
				} catch (RepositoryException e1) {}
			}

		}

	}

	public JCRWorkspaceItem(JCRWorkspace workspace, Node node, String name,
			String description) throws  RepositoryException  {			

		Validate.notNull(name, "Name must be not null");
		Validate.notNull(description, "Description must be not null");

		this.workspace = workspace;
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

		node.setProperty(LAST_MODIFIED_BY, workspace.getOwner().getPortalLogin());
		node.setProperty(DESCRIPTION, description);

		//		String title = Text.escapeIllegalJcrChars(name);
		node.setProperty(TITLE, name);

		node.setProperty(LAST_ACTION, WorkspaceItemAction.CREATED.toString());
		node.setProperty(PORTAL_LOGIN, workspace.getOwner().getPortalLogin());

		//		Node nodeOwner = null;
		//		try {
		//			nodeOwner = node.getNode(OWNER);
		//		} catch (PathNotFoundException e) {
		//			nodeOwner = node.addNode(OWNER,NT_USER);
		//		}
		//
		//		setOwnerNode(nodeOwner);

	}


	public void setOwnerNode(Node nodeOwner) throws RepositoryException {
		//nodeOwner.setProperty(SCOPE,(workspace.getOwner().getScope() != null)?workspace.getOwner().getScope().toString():null);
		nodeOwner.setProperty(PORTAL_LOGIN,workspace.getOwner().getPortalLogin());
		nodeOwner.setProperty(USER_ID, UUID.randomUUID().toString());
	}

	@Override
	public User getOwner() {

		return new JCRUser(userId, portalLogin);

	}

	public void save(Node node) throws RepositoryException {

		node.getSession().save();		
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

	}	

	@Override
	public String getId() throws InternalErrorException {
		return identifier;
	}



	@Override
	public String getName() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			String name = node.getProperty(TITLE).getString();
			return name;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public String getDescription() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(DESCRIPTION).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}


	@Override
	public void setDescription(String description)
			throws InternalErrorException {	

		Session session = JCRRepository.getSession();
		try {
			internalDescription(session.getNodeByIdentifier(identifier), description);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void rename(String name) throws InternalErrorException,
	InsufficientPrivilegesException, ItemAlreadyExistException {

		try {
			workspace.renameItem(getId(), name);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public Calendar getCreationTime() throws InternalErrorException {		
		return creationDate;
	}

	@Override
	public Calendar getLastModificationTime() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(LAST_MODIFIED).getDate();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	@Override
	public WorkspaceItemAction getLastAction() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return WorkspaceItemAction.valueOf(node.getProperty(LAST_ACTION).getString());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Capabilities getCapabilities() {
		return null;
	}

	@Override
	public Properties getProperties() throws InternalErrorException {		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return new JCRProperties(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	public void addAccountingEntry(AccountingEntry entry) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);

			if (!node.hasNode(ACCOUNTING)){
				node.addNode(ACCOUNTING, NT_ACCOUNTING);
				session.save();
			}
			Node accountingNode = node.getNode(ACCOUNTING);
			JCRAccountingEntryType nodeType = JCRAccountingEntryType.valueOf(entry.getEntryType().toString());
			logger.debug("Accountin node type " + nodeType.getNodeTypeDefinition());
			Node entryNode = accountingNode.addNode(UUID.randomUUID().toString(),
					nodeType.getNodeTypeDefinition());

			((JCRAccountingEntry) entry).save(entryNode);

			session.save();	
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
	}


	@Override
	public List<AccountingEntry> getAccounting() {
		List<AccountingEntry> list = new ArrayList<AccountingEntry>();


		Session session = null;
		try {

			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);


			JCRAccountingEntryCreate entry = new JCRAccountingEntryCreate(getOwner().getPortalLogin(),
					getCreationTime(), node.getName());
			list.add(entry);

			Node accountingNode = node.getNode(ACCOUNTING);

			for(NodeIterator iterator = accountingNode.getNodes();iterator.hasNext();) {
				Node entryNode = (Node)iterator.next();
				try {
					switch (JCRAccountingEntryType.getEnum(
							entryNode.getPrimaryNodeType().getName())) {
							case CUT:
								list.add(new JCRAccountingFolderEntryCut(entryNode));
								break;
							case PASTE:
								list.add(new JCRAccountingEntryPaste(entryNode));
								break;
							case REMOVAL:
								list.add(new JCRAccountingFolderEntryRemoval(entryNode));
								break;
							case RENAMING:
								list.add(new JCRAccountingEntryRenaming(entryNode));
								break;
							case ADD:
								list.add(new JCRAccountingFolderEntryAdd(entryNode));
								break;								
							case UPDATE:
								list.add(new JCRAccountingEntryUpdate(entryNode));
								break;
							case READ:
								list.add(new JCRAccountingEntryRead(entryNode));
								break;
							case SHARE:
								list.add(new JCRAccountingEntryShare(entryNode));
								break;
							case UNSHARE:
								list.add(new JCRAccountingEntryUnshare(entryNode));
								break;
							case ADD_ACL:
								list.add(new JCRAccountingEntryAddACL(entryNode));
								break;
							case MODIFY_ACL:
								list.add(new JCRAccountingEntryModifyACL(entryNode));
								break;
							case DELETE_ACL:
								list.add(new JCRAccountingEntryDeleteACL(entryNode));
								break;

							default:
								break;												
					}
				} catch (Exception e) {
					logger.error("Accounting entry skipped "+ entryNode.getPrimaryNodeType().getName(),e);
				}

			}

			try{
				List<AccountingEntryRead> readersList = getReadersNode();

				for (Iterator<AccountingEntryRead> iterator = readersList.iterator(); iterator.hasNext();){
					AccountingEntryRead reader = iterator.next();
					list.add(reader);
				}
			}catch (Exception e) {
				logger.debug("Node Readers does not exist in " + node.getPath());
			}

			return list;

		} catch (Exception e) {
			logger.error("Error to retrieve accounting entries ",e);
			return list;
		} finally {
			if (session != null)
				session.logout();
		}

	}


	//get old readers list
	public List<AccountingEntryRead> getReadersNode() throws InternalErrorException {
		List<AccountingEntryRead> list = new ArrayList<AccountingEntryRead>();

		Session session = JCRRepository.getSession();
		try {

			Node node = session.getNodeByIdentifier(identifier);
			Node readersNode = node.getNode(READERS);

			for (NodeIterator iterator = readersNode.getNodes(); iterator.hasNext();) {
				Node reader = iterator.nextNode();
				if ( JCRAccountingEntryType.getEnum(
						reader.getPrimaryNodeType().getName()) != null)
					list.add(new JCRAccountingEntryRead(reader));
			}
			return list; 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}



	@Override
	public WorkspaceFolder getParent() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			return workspace.getParent(session.getNodeByIdentifier(identifier));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	protected JCRAbstractWorkspaceFolder getParent(Node node) throws InternalErrorException {
		try {
			return workspace.getParent(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} 
	}

	@Override
	public boolean isShared () throws InternalErrorException {
		return (getIdSharedFolder() != null)?true:false;
	}

	@Override
	public String getIdSharedFolder() throws InternalErrorException {

		if (isRoot()) {
			return null;
		}
		//		if (getType() == WorkspaceItemType.SHARED_FOLDER || (getParent().getType() == WorkspaceItemType.SHARED_FOLDER))
		if (getType() == WorkspaceItemType.SHARED_FOLDER) 
			return getId();

		return  ((JCRWorkspaceItem)getParent()).getIdSharedFolder();
	}

	@Override
	public String getPath() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			return getPath(session.getNodeByIdentifier(identifier));
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	public String getPath(Node node) throws RepositoryException, InternalErrorException {
		if (isRoot(node)) return workspace.getPathSeparator() + node.getProperty(TITLE).getString();
		return getParent(node).getPath(node.getParent()) + workspace.getPathSeparator() 
				+ node.getProperty(TITLE).getString();	
	}

	@Override
	public boolean isRoot() throws InternalErrorException { 
		return getParent() == null;
	}

	public boolean isRoot(Node node) throws RepositoryException, InternalErrorException {
		return workspace.getParent(node) == null;
	}

	@Override
	public void remove() throws InternalErrorException, InsufficientPrivilegesException {
		try {			
			workspace.removeItem(getId());

		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} 

	}

	@Override
	public void move(WorkspaceFolder destination)
			throws InternalErrorException, WrongDestinationException,
			InsufficientPrivilegesException, ItemAlreadyExistException {

		try {

			workspace.moveItem(getId(), destination.getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkspaceItem cloneItem(String cloneName)
			throws InternalErrorException, InsufficientPrivilegesException,
			ItemAlreadyExistException {

		try {
			return workspace.cloneItem(getId(), cloneName);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}


	//copy a node
	public Node internalCopy(Node nodeFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{
		Node node = nodeFolder.getSession().getNodeByIdentifier(identifier);

		String pathNewNode = null;

		try {
			pathNewNode = nodeFolder.getPath()+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);
			if(node.getSession().getNode(pathNewNode) != null){
				//				throw new ItemAlreadyExistException(newName + " already exist");

				WorkspaceFolder destinationFolder;

				destinationFolder = (WorkspaceFolder) workspace.getItem(nodeFolder.getSession(), nodeFolder.getIdentifier());

				newName = WorkspaceUtil.getCopyName(Text.escapeIllegalJcrChars(newName), destinationFolder);
				//				newName = WorkspaceUtil.getUniqueName(Text.escapeIllegalJcrChars(newName), destinationFolder);

				pathNewNode = nodeFolder.getPath()+ workspace.getPathSeparator() + newName;
			}

		} catch(RepositoryException | ItemNotFoundException e) {
			//e.printStackTrace();
		}

		try {
			logger.trace("copy: " + node.getPath() + " - to: " + pathNewNode);
			node.getSession().getWorkspace().copy(node.getPath(),pathNewNode);
			node.getSession().save();

			Node newNode = node.getSession().getNode(pathNewNode);			

			newNode.setProperty(LAST_MODIFIED,Calendar.getInstance());
			newNode.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			newNode.setProperty(TITLE, Text.escapeIllegalJcrChars(newName));
			newNode.setProperty(LAST_ACTION,WorkspaceItemAction.CLONED.toString());
			newNode.getSession().save();

			return newNode;

		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}



	}



	//	public Node copyAsBookmark(Node nodeFolder, String newName) throws InternalErrorException,
	//	ItemAlreadyExistException, WrongDestinationException, RepositoryException{
	//
	//		Node node = nodeFolder.getSession().getNodeByIdentifier(identifier);
	//
	//		String pathNewNode = nodeFolder.getPath()
	//				+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);
	//
	//		try {
	//
	//			if(node.getSession().getNode(pathNewNode) != null)
	//				throw new ItemAlreadyExistException(newName + " already exist");
	//		} catch(RepositoryException e) {
	//
	//		}
	//
	//		try {
	//			node.getSession().getWorkspace().copy(node.getPath(),pathNewNode);
	//
	//			Node newNode = node.getSession().getNode(pathNewNode);
	//			newNode.setPrimaryType(NT_BOOKMARK);
	//			newNode.setProperty(LAST_MODIFIED,Calendar.getInstance());
	//			newNode.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
	//			newNode.setProperty(TITLE, newName);
	//			newNode.setProperty(LAST_ACTION,WorkspaceItemAction.CLONED.toString());
	//			newNode.getSession().save();
	//			return newNode;
	//		} catch (ItemExistsException e) {
	//			throw new ItemAlreadyExistException(e.getMessage());
	//		} catch (RepositoryException e) {
	//			throw new InternalErrorException(e);
	//		}
	//
	//	}


	public void internalMove(Node destinationFolderNode) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {

		try {
			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getIdentifier());

			Node node = destinationFolderNode.getSession().getNodeByIdentifier(identifier);

			if (workspace.exists(node.getName(), destinationFolderNode.getIdentifier())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + node.getName() + " already exists");
			}

			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.MOVED.toString());
			node.getSession().save();

			node.getSession().getWorkspace().move(node.getPath(), destinationFolderNode.getPath() 
					+ workspace.getPathSeparator() + node.getName());


		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} 
	}

	public void internalRename(Node node, String newName) throws ItemAlreadyExistException, InternalErrorException {

		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {

			logger.debug("Internal rename item with id " 
					+ getId() + " to destination item with id " + node.getParent().getIdentifier());
			if (workspace.exists(nodeNewName, node.getParent().getIdentifier())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			String newPath = node.getParent().getPath() 
					+ workspace.getPathSeparator() + nodeNewName;

			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.RENAMED.toString());
			node.setProperty(TITLE,newName);
			node.getSession().save();

			String path = node.getPath();
			node.getSession().getWorkspace().move(path, newPath);



		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		}

	}

	public void internalDescription(Node node, String newDescription) throws InternalErrorException {

		Validate.notNull(newDescription, "Description must be not null");
		try {
			node.setProperty(DESCRIPTION, newDescription);
			node.setProperty(LAST_MODIFIED, Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY, workspace.getOwner().getPortalLogin());
			node.getSession().save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}

	//check if a node has been read
	@Override
	public boolean isMarkedAsRead() throws InternalErrorException  {

		Session session = JCRRepository.getSession();
		try {
			return hasReaders(session);
			//			return existReader(session);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}


	//check if a node has readers
	private boolean hasReaders(Session session) throws  RepositoryException,
	InternalErrorException{
		Node node = session.getNodeByIdentifier(getId());
		int count = 0;
		try {

			Node accountingNode = node.getNode(ACCOUNTING);

			for(NodeIterator iteratorEntry = accountingNode.getNodes();iteratorEntry.hasNext();) {
				Node readersNode = (Node)iteratorEntry.next();
				try {
					String entry = readersNode.getPrimaryNodeType().getName();

					if (entry.equals(READ)) 
						count++;

				}catch (Exception e) {
					logger.info("Node ACCOUNTING not found");
				}
			}

			if (count > 0)
				return true;

		} catch(PathNotFoundException e) {
			logger.info("Node READERS has been added to " + node.getPath());
			return false;
		}
		return false;
	}

	@Override
	public List<AccountingEntryRead> getReaders() throws InternalErrorException {
		List<AccountingEntryRead> list = new ArrayList<AccountingEntryRead>();

		Session session = JCRRepository.getSession();
		try {

			Node node = session.getNodeByIdentifier(identifier);
			Node accountingNode = node.getNode(ACCOUNTING);

			for(NodeIterator iteratorEntry = accountingNode.getNodes();iteratorEntry.hasNext();) {
				Node readersNode = (Node)iteratorEntry.next();
				try {
					String entry = readersNode.getPrimaryNodeType().getName();

					if (entry.equals(READ))
						list.add(new JCRAccountingEntryRead(readersNode));

				}catch (Exception e) {
					logger.info("Node ACCOUNTING not found");
				}
			}	

			return list; 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}



	//	private boolean existReader(Session session) throws  RepositoryException,
	//	InternalErrorException{
	//
	//		Node node = session.getNodeByIdentifier(getId());
	//		Node readersNode = null;
	//		try {
	//			readersNode = node.getNode(READERS);
	//			logger.info("Node READERS in " + node.getPath() + " already exists");
	//		} catch(PathNotFoundException e) {
	//			readersNode = node.addNode(READERS, NT_READERS );
	//			session.save();
	//			logger.info("Node READERS has been added to " + node.getPath());
	//			return false;
	//		}
	//
	//		if (isShared()){
	//
	//			if (readersNode.hasNode(workspace.getOwner().getPortalLogin())){
	//				logger.info(getName() + " is Shared; " + workspace.getOwner().getPortalLogin() +  " is already in readers list of " + node.getPath());
	//				return true;
	//			}
	//		}else{
	//			if (readersNode.hasNodes()){
	//				logger.info(getName() +"; " + workspace.getOwner().getPortalLogin() +  " is already in readers list of " + node.getPath());
	//				return true;
	//			}
	//		}
	//
	//		//		if (readersNode.hasNode(workspace.getOwner().getPortalLogin()))
	//		//			return true;
	//
	//		return false;
	//	}


	//set Share Operation in History
	public void setShareHistory(List<String> users, String author) throws InternalErrorException {


		Session session = JCRRepository.getSession();
		try{

			setShare(users, author);
			setHistoryShareUnshare(this, JCRAccountingEntryType.SHARE.getNodeTypeDefinition(), author, users);

		}catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			session.logout();
		}

	}



	private void setHistoryShareUnshare(WorkspaceItem item, String operation, String user, List<String> members) throws RepositoryException, InternalErrorException, ItemNotFoundException {

		for (WorkspaceItem child : item.getChildren()) { 

			try{		
				if(child.getType().equals(WorkspaceItemType.FOLDER_ITEM) || child.getType().equals(WorkspaceItemType.FOLDER)){

					if (operation.equals(JCRAccountingEntryType.UNSHARE.getNodeTypeDefinition()))

						((JCRWorkspaceItem) child).setUnshare(user);
					else if (operation.equals(JCRAccountingEntryType.SHARE.getNodeTypeDefinition()))
						((JCRWorkspaceItem) child).setShare(members, user);

				}
			}catch (Exception e) {
				throw new ItemNotFoundException(e.getMessage());
			}

			//recorsive
			if (child.getChildren().size()>0)
				setHistoryShareUnshare(child, operation, user, members);			
		}

	}


	//set unShare Operation in History
	public void setUnshareHistory(String user) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try{
			//set unshare on folder	
			setUnshare(user);
			//set unshare on children
			try{
				setHistoryShareUnshare(this, JCRAccountingEntryType.UNSHARE.getNodeTypeDefinition(), user, null);
			}catch (Exception e) {
				// TODO: handle exception
			}

		}catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			session.logout();
		}

	}


	public void setShare(List<String> users, String author) throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try{
			Node node = session.getNodeByIdentifier(getId());

			if (!node.hasNode(ACCOUNTING)){
				node.addNode(ACCOUNTING, NT_ACCOUNTING);
				session.save();
			}

			logger.info("Add SHARE operation for user " + author + " to node " + node.getPath());
			Node accounting = node.getNode(ACCOUNTING);

			JCRAccountingEntryShare entry = new JCRAccountingEntryShare(
					author, Calendar.getInstance(), this.getName(), users);

			//add reader to ACCOUNTING node
			entry.save(accounting.addNode(UUID.randomUUID().toString(), 
					JCRAccountingEntryType.SHARE.getNodeTypeDefinition()));

			session.save();

		}catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			session.logout();
		}
	}


	public void setUnshare(String user) throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try{
			Node node = session.getNodeByIdentifier(getId());
			if (!node.hasNode(ACCOUNTING)){
				node.addNode(ACCOUNTING, NT_ACCOUNTING);
				session.save();
			}
			logger.info("Add UNSHARE operation for user " + user);
			Node accounting = node.getNode(ACCOUNTING);

			JCRAccountingEntryUnshare entry = new JCRAccountingEntryUnshare(
					user, Calendar.getInstance(), this.getName());

			//add reader to ACCOUNTING node
			entry.save(accounting.addNode(UUID.randomUUID().toString(), 
					JCRAccountingEntryType.UNSHARE.getNodeTypeDefinition()));

			session.save();
		}catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			session.logout();
		}

	}


	@Override
	public void markAsRead(boolean read) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());

			if (!node.hasNode(ACCOUNTING)){
				node.addNode(ACCOUNTING, NT_ACCOUNTING);
				session.save();
			}

			logger.info("Mark Node " + node.getPath() + " As Read ");
			Node parentNode = node.getParent();
			String parentPath = parentNode.getPath();

			String user = workspace.getOwner().getPortalLogin();

			Node accounting = node.getNode(ACCOUNTING);

			WorkspaceItem item = workspace.getItem(session, node.getIdentifier());		
			String name = item.getName();

			if (read) {

				//set reader in file
				logger.info("Setting " + node.getName() + " as read in " + node.getPath());			

				JCRAccountingEntryRead entry = new JCRAccountingEntryRead(
						user, Calendar.getInstance(), name);

				//add reader to ACCOUNTING node
				entry.save(accounting.addNode(UUID.randomUUID().toString(), 
						JCRAccountingEntryType.READ.getNodeTypeDefinition()));

				session.save();

				//set reader in parent folder
				try{	

					logger.info("Mark Parent Node " + parentNode.getPath() + " As Read ");

					//mark parent folder as read
					Node accountingParentNode = parentNode.getNode(ACCOUNTING);

					JCRAccountingEntryRead entryParent = new JCRAccountingEntryRead( 
							user, Calendar.getInstance(), name);

					//add reader to ACCOUNTING parent node
					entryParent.save(accountingParentNode.addNode(UUID.randomUUID().toString(), 
							JCRAccountingEntryType.READ.getNodeTypeDefinition()));

					session.save();


					logger.info("Set " + name + " as read in " + parentPath);


				}catch (Exception e) {
					logger.info("Error setting " +name + " as read in " + parentPath);
				}


			}
			//			} 

			//unread : not implemented
			//			else {
			//				if (!read)
			//					readers.getNode(user).remove();
			//			}

			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	//change remote path
	@Override
	public String getRemotePath() throws InternalErrorException {
		String remotePath = null;
		Session session = JCRRepository.getSession();
		Node nodeItem;
		try {
			nodeItem = session.getNodeByIdentifier(identifier);
		} catch (javax.jcr.ItemNotFoundException e2) {
			throw new InternalErrorException(e2);
		} catch (RepositoryException e2) {
			throw new InternalErrorException(e2);
		}

		try {
			Node contentNode = nodeItem.getNode(CONTENT);
			if (contentNode.hasProperty(JCRFile.REMOTE_STORAGE_PATH)) {
				remotePath = contentNode.getProperty(JCRFile.REMOTE_STORAGE_PATH).getString();
			}
		} catch (RepositoryException e) {

			try{
				if (nodeItem.hasProperty(JCRFile.REMOTE_STORAGE_PATH)) {
					remotePath = nodeItem.getProperty(JCRFile.REMOTE_STORAGE_PATH).getString();
				}
			}catch (Exception e1) {
				throw new InternalErrorException(e1);
			}

		} finally {
			session.logout();
		}	

		return remotePath;	
	}

	//	//get a remote path of a folder
	//	public String getRemotePathDir(Node nodeItem, int i) throws InternalErrorException, RepositoryException {
	////		System.out.println("getRemotePathDir(child, i) " + nodeItem.getPath() + " - " + i);
	//
	//		String remotePath = null;
	//		NodeIterator children = nodeItem.getNodes();
	//
	//		while (children.hasNext()){
	//			Node child = children.nextNode();
	//			try{				
	//				remotePath = child.getProperty(JCRFile.REMOTE_STORAGE_PATH).getValue().getString();				
	//				this.deep = nodeItem.getDepth();
	//				System.out.println("REMOTE_STORAGE_PATH " + remotePath + " - deep child " + this.deep);
	//			}catch (Exception e) {}
	//
	//			//recorsive
	//			if (child.hasNodes()){
	//				System.out.println(child.getPath() + " has childs " );
	//				remotePath = getRemotePathDir(child, i++);
	//				if (remotePath!=null){					
	//					break;
	//				}
	//			}
	//		}
	//		return remotePath;
	//	}



	//change remote path

	public void setRemotePath(String remotePath, Node node) throws InternalErrorException, RepositoryException{
		Session session;
		if (node!=null)
			session = node.getSession();
		else
			session = JCRRepository.getSession();

		try {
			Node nodeItem = session.getNodeByIdentifier(identifier);

			try{ 
				Node contentNode = nodeItem.getNode(CONTENT);
				if (contentNode.hasProperty(JCRFile.REMOTE_STORAGE_PATH)) 
					contentNode.setProperty(JCRFile.REMOTE_STORAGE_PATH, remotePath);
				session.save();
			}catch (Exception e) {
				//				if (nodeItem.hasProperty(JCRFile.REMOTE_STORAGE_PATH)) 
				//				logger.debug("no content");
				try{
					nodeItem.setProperty(JCRFile.REMOTE_STORAGE_PATH, remotePath);
					session.save();
				}catch (Exception e1) {
					throw new InternalErrorException(e1);
				}
			}
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			if (node==null)
				session.logout();
		}

	}



	public void setNewRemotePath(String remotePath, Node node) throws InternalErrorException, RepositoryException{
		Session session;
		if (node!=null)
			session = node.getSession();
		else
			session = JCRRepository.getSession();

		try {
			Node nodeItem = session.getNodeByIdentifier(identifier);
			nodeItem.setProperty(JCRFile.STORAGE_PATH, remotePath);
			session.save();

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			if (node==null)
				session.logout();
		}

	}

	//unlock a node
	public LockManager getLock(Node node) throws InternalErrorException {
		LockManager lockManager = null;
		try{
			lockManager = node.getSession().getWorkspace().getLockManager();
			Lock lock = null;
			try {
				lock = lockManager.getLock(node.getPath());
			} catch (LockException ex) {                    
			}
			if (lock != null) {
				lockManager.addLockToken(lock.getLockToken());
				lockManager.unlock(node.getPath());
			}
		} catch (RepositoryException e) {
			e.printStackTrace();
			throw new InternalErrorException(e);
		}
		return lockManager;

	}


	//change owner to all files
	public void setOwnerToCurrentUser(WorkspaceItem item) throws RepositoryException, InternalErrorException {

		//		System.out.println("set owner to current user " + item.getPath());
		Session session;
		try {
			session = JCRRepository.getSession();

			if(item.getType().equals(WorkspaceItemType.FOLDER_ITEM) || item.getType().equals(WorkspaceItemType.FOLDER)){
				//				logger.debug("setOwnerToCurrentUser  to " + item.getPath());
				//update owner node
				Node itemUnshared = session.getNodeByIdentifier(item.getId());
				Node nodeOwner = null;
				try{
					nodeOwner = itemUnshared.getNode(OWNER);
				}catch (Exception e) {
					nodeOwner = itemUnshared.addNode(OWNER,NT_USER);
					session.save();
				}
				setOwnerNode(nodeOwner);
				session.save();
			}

			List<? extends WorkspaceItem> children = null;
			try{
				children = item.getChildren();								
			}catch (Exception e) {
				//				System.out.println("no children");
			}
			if (children!=null){
				for (WorkspaceItem child : children) {

					setOwnerToCurrentUser(child);	
				}
			}
		} catch (InternalErrorException e1) {
			throw new InternalErrorException(e1);
		}	

	}


	public List<String> getUsers() throws InternalErrorException {

		ArrayList<String> list = new ArrayList<String>();
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());
			Node usersNode = node.getNode(USERS);
			for (PropertyIterator iterator = usersNode.getProperties(); iterator.hasNext();) {
				Property property  = iterator.nextProperty();
				String name = property.getName();
				if (!name.startsWith(JCRRepository.JCR_NAMESPACE) &&
						!name.startsWith(JCRRepository.HL_NAMESPACE))
					list.add(name);
			} 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}


	@Override
	public boolean isFolder() throws InternalErrorException {
		if (getType().equals(WorkspaceItemType.FOLDER) || getType().equals(WorkspaceItemType.SHARED_FOLDER))
			return true;

		return false;
	}



	/**
	 * Get PublicLink
	 */
	@Override
	public String getPublicLink(boolean shortenUrl) throws InternalErrorException {
		logger.info("get PublicLink for item: " + getName());
		String publicLink = null;
		if(getType().equals(WorkspaceItemType.FOLDER_ITEM)){

			FolderItem folderItem = (FolderItem) this;

			try {
				String smpUri = getPubliLinkForFolderItem(folderItem);
				publicLink = getPublicLinkForFolderItemId(smpUri, folderItem, shortenUrl);
			} catch (Exception e) {
				throw new InternalErrorException("Sorry, Public Link for selected file is unavailable");
			}
		}else{
			logger.warn("ItemId: "+ getId() +" is not a folder item, sent exception Public Link  unavailable");
			throw new InternalErrorException("Sorry, Public Link for selected file is unavailable");
		}
		return publicLink;
	}



	private String getPubliLinkForFolderItem(FolderItem folderItem) throws InternalErrorException {
		return GCUBEStorage.getPublicLink(folderItem.getRemotePath(), portalLogin);

	}


	/**
	 * 
	 * @param itemId
	 * @param shortenUrl
	 * @return
	 */
	public String getPublicLinkForFolderItemId(String smpUri, FolderItem folderItem, boolean shortenUrl) throws InternalErrorException{

		logger.trace("get Public Link ");

		try{			
			if(smpUri==null || smpUri.isEmpty())
				throw new Exception("Sorry, public link on "+ folderItem.getName() +" is not available");

			UriResolverReaderParameter uriResolver = new UriResolverReaderParameter(ScopeProvider.instance.get());

			String uriRequest = "";

			if(uriResolver!=null && uriResolver.isAvailable()){

				String itemName = StringUtil.removeSpecialCharacters(folderItem.getName());

				itemName = StringUtil.replaceAllWhiteSpace(itemName, "_");
				uriRequest =  uriResolver.resolveAsUriRequest(smpUri, itemName, folderItem.getMimeType(), true);


				//VALIDATE CONNECTION
				if(!HttpRequestUtil.urlExists(uriRequest))
					throw new InternalErrorException("Sorry, The Public Link for selected file is unavailable");

				if(shortenUrl)	
					uriRequest = getShortUrl(uriRequest);

				return uriRequest;

			}
			else
				throw new InternalErrorException("Sorry, The Uri resolver service is temporarily unavailable. Please try again later");

		}catch (Exception e) {
			logger.error("Error getPublicLinkForFolderItemId for item: "+folderItem.getId(), e);
			throw new InternalErrorException(e.getMessage());
		}
	}

	public String getShortUrl(String longUrl) throws Exception {

		logger.trace("get short url for "+ longUrl);

		UrlShortener shortener = new UrlShortener(ScopeProvider.instance.get());
		try{
			if(shortener!=null && shortener.isAvailable())
				return shortener.shorten(longUrl);
			return longUrl;

		}catch (Exception e) {
			logger.error("Error get short url for ", e);
			return longUrl;
		}
	}
}
