package org.gcube.common.homelibrary.jcr.repository.external;

import java.io.IOException;
import java.io.InputStream;

import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.workspace.util.Utils;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;
import org.gcube.contentmanager.storageclient.model.protocol.smp.Handler;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GCUBEStorage {

	private static Logger logger = LoggerFactory.getLogger(GCUBEStorage.class);

	public static IClient getStorage(String portalLogin){
		IClient storage = null;
		logger.trace("Calling GCUBEStorage from user: " + portalLogin);
		try {

			String currentScope = ScopeProvider.instance.get();
			String scope = Utils.getRootScope(currentScope);			
			ScopeProvider.instance.set(scope);	

			logger.trace("GCUBE Storage scope : " + scope);

			//			storage = new StorageClient("org.gcube.portlets.user","home-library",
			//					portalLogin, AccessType.SHARED, currentScope, false).getClient();

			storage = new StorageClient("org.gcube.portlets.user", JCRRepository.serviceName,
					portalLogin, AccessType.SHARED, currentScope, false).getClient();

			Handler.activateProtocol();

		} catch (Exception e) {
			logger.error("GCUBE Storage initialization failed",e);
		}
		return storage;
	}


	public static String putStream(InputStream is, String remotePath, String portalLogin) throws RemoteBackendException {

		logger.trace("GCUBE Storage - putStream - remotePath: " + remotePath);

		IClient storage = getStorage(portalLogin);
		storage.put(true).LFile(is).RFile(remotePath);
		try {
			is.close();
			logger.trace("GCUBE Storage inputStream closed");
		} catch (IOException e) {
			logger.error("GCUBE Storage error during to close inputStream",e);
		}

		return getStorage(portalLogin).getUrl().RFile(remotePath);
	}



	public static String putStream(String old_remotePath, String new_remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage -putStream - remotePath: " + new_remotePath);
		//		System.out.println("GCUBE Storage RFile path : " + remotePath);
		IClient storage = getStorage(portalLogin);
		storage.put(true).LFile(old_remotePath).RFile(new_remotePath);
		return getStorage(portalLogin).getUrl().RFile(new_remotePath);
	}

	public static long getRemoteFileSize(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage getRemoteFileSize - storage.getSize().RFile(" + remotePath+ ")");
		IClient storage = getStorage(portalLogin);
		long size =storage.getSize().RFile(remotePath);
		return size;
	}

	/**
	 * delete file	
	 * @param remotePath
	 * @param portalLogin
	 * @throws RemoteBackendException
	 */
	public static void removeRemoteFile(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - removeRemoteFile - remotePath: " + remotePath);
		getStorage(portalLogin).remove().RFile(remotePath);
	}

	/**
	 * Delete folder
	 * @param remotePath
	 * @param portalLogin
	 * @throws RemoteBackendException
	 */
	public static void removeRemoteFolder(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - storage.removeDir().RDir(" + remotePath + ")");
		IClient storage = getStorage(portalLogin);
		storage.removeDir().RDir(remotePath);
	}
	
	/**
	 * Get size of a file by remotepath
	 * @param remotePath
	 * @param portalLogin
	 * @param serviceName
	 * @return size of a file
	 * @throws RemoteBackendException
	 */
	public static long getRemoteFileSize(String remotePath,
			String portalLogin, String serviceName) throws RemoteBackendException {
		logger.trace("GCUBE Storage getRemoteFileSize - storage.getSize().RFile(" + remotePath+ ")");
		IClient storage = getStorage(portalLogin, serviceName);
		long size =storage.getSize().RFile(remotePath);
		return size;
	}
	
	/**
	 * Get the disk usage amount of a user 
	 * @param portalLogin
	 * @return the disk usage amount of a user 
	 * @throws RemoteBackendException
	 */
	public static long getDiskUsageByUser(String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume used by user: " + portalLogin + "");
		IClient storage = getStorage(portalLogin);
		return Long.parseLong(storage.getTotalUserVolume());
	}

	/**
	 * Get tot items of a user
	 * @param portalLogin
	 * @return tot items of a user
	 * @throws RemoteBackendException
	 */
	public static int getTotalItemsByUser(String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get tot items of user " + portalLogin + "");
		IClient storage = getStorage(portalLogin);
		return Integer.parseInt(storage.getUserTotalItems());
	}
	/**
	 * Get folder size
	 * @param remotePath
	 * @param portalLogin
	 * @return folder size
	 * @throws RemoteBackendException
	 */
	public static long getFolderTotalVolume(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume (" + remotePath + ")");
		IClient storage = getStorage(portalLogin);
		return Long.parseLong(storage.getFolderTotalVolume().RDir(remotePath).toString());
	}

	/**
	 * Get last update of a folder
	 * @param remotePath of the folder
	 * @param portalLogin
	 * @return folder size
	 * @throws RemoteBackendException
	 */
	public static String getFolderLastUpdate(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume (" + remotePath + ")");
		IClient storage = getStorage(portalLogin);
		return storage.getFolderLastUpdate().RDir(remotePath);
	}

	/**
	 * Get total items in a folder
	 * @param remotePath of the folder
	 * @param portalLogin
	 * @return total items in a folder
	 * @throws RemoteBackendException
	 */
	public static int getFolderTotalItems(String remotePath, String portalLogin) throws RemoteBackendException {
		logger.trace("GCUBE Storage - get volume (" + remotePath + ")");
		IClient storage = getStorage(portalLogin);
		return Integer.parseInt(storage.getFolderTotalItems().RDir(remotePath).toString());
	}

	/**
	 * Get the inpustream by a remote path
	 * @param remotePath
	 * @param portalLogin
	 * @return the inpustream by a remote path
	 */
	public static InputStream getRemoteFile(String remotePath, String portalLogin) {
		logger.trace("GCUBE Storage -getRemoteFile - stream = storage.get().RFileAsInputStream(" + remotePath+ ")");
		InputStream stream = null;
		IClient storage = getStorage(portalLogin);
		try{
			stream = storage.get().RFileAsInputStream(remotePath);
		}catch (Exception e) {
			//			e.printStackTrace();
			logger.error("no payload found in Storage for remotePath: " + remotePath );
		}
		return stream;
	}


	/**
	 * Get public link by remote path
	 * @param remotePath
	 * @param portalLogin
	 * @return public link 
	 */
	public static String getPublicLink(String remotePath, String portalLogin) {
		logger.trace("GCUBE Storage - Generate Public Link for " + remotePath);
		IClient storage = getStorage(portalLogin);
		String publicUrl = storage.getUrl().RFile(remotePath);
		return publicUrl;
	}

	/**
	 * move dir
	 * @param oldPath
	 * @param newPath
	 * @param portalLogin
	 */
	public static void moveRemoteFolder(String oldPath, String newPath, String portalLogin) {
		logger.trace("GCUBE Storage - storage.moveDir().from(" + oldPath + ").to(" + newPath + ")");
		IClient storage = getStorage(portalLogin);
		storage.moveDir().from(oldPath).to(newPath);
	}

	/**
	 * move file
	 * @param oldPath
	 * @param newPath
	 * @param portalLogin
	 */
	public static void moveRemoteFile(String oldPath, String newPath, String portalLogin) {
		logger.trace("GCUBE Storage - storage.moveFile().from(" + oldPath + ").to(" + newPath + ")");
		IClient storage = getStorage(portalLogin);
		storage.moveFile().from(oldPath).to(newPath);
	}

	/**
	 * copy file
	 * @param oldPath
	 * @param newPath
	 * @param portalLogin
	 */
	public static void copyRemoteFile(String oldPath, String newPath, String portalLogin) {
		logger.trace("GCUBE Storage - storage.copyFile().from(" + oldPath + ").to(" + newPath + ")");
		IClient storage = getStorage(portalLogin);
		storage.copyFile().from(oldPath).to(newPath);
	}

	/**
	 * use remotepath to access storage
	 * @param srcPath
	 * @param destPath
	 * @param portalLogin
	 */
	public static void createHardLink(String srcPath, String destPath, String portalLogin) {
		try{
			logger.trace("GCUBE Storage - storage.linkFile().from(" + srcPath + ").to(" + destPath + ")");
			IClient storage = getStorage(portalLogin);
			storage.linkFile().from(srcPath).to(destPath);
			//			System.out.println("created a hard link");
		}catch (Exception e) {
			logger.error("GCUBE Storage error creating a linkFile",e);
		}
	}

	//	public static InputStream getInputStream(URL url) throws IOException {
	//		
	//		URLConnection uc = url.openConnection();
	//		InputStream is=uc.getInputStream();
	//		return is;
	//	}

	//new repository


	/**
	 * delete folder
	 * @param remotePath
	 * @param portalLogin
	 * @param serviceName
	 * @throws RemoteBackendException
	 */
	public static void removeRemoteFolder(String remotePath, String portalLogin, String serviceName) throws RemoteBackendException {
		logger.trace("GCUBE Storage - storage.removeDir().RDir(" + remotePath + ")");
		IClient storage = getStorage(portalLogin, serviceName);
		storage.removeDir().RDir(remotePath);
	}

	/**
	 * Save inpustream in storage using a remotepath
	 * @param is
	 * @param new_remotePath
	 * @param portalLogin
	 * @param serviceName
	 * @return remote path
	 * @throws RemoteBackendException
	 */
	public static String putStream(InputStream is,  String new_remotePath, String portalLogin, String serviceName) throws RemoteBackendException {

		logger.trace("GCUBE Storage - putStream - remotePath: " + new_remotePath);

		IClient storage = getStorage(portalLogin, serviceName);
		storage.put(true).LFile(is).RFile(new_remotePath);
		try {
			is.close();
			logger.trace("GCUBE Storage inputStream closed");
		} catch (IOException e) {
			logger.error("GCUBE Storage error during to close inputStream",e);
		}

		return getStorage(portalLogin).getUrl().RFile(new_remotePath);
	}


	private static IClient getStorage(String portalLogin, String serviceName) {
		IClient storage = null;
		logger.trace("Calling GCUBEStorage from user: " + portalLogin + " - serviceName: " + serviceName );
		try {

			String currentScope = ScopeProvider.instance.get();
			String scope = Utils.getRootScope(currentScope);			
			ScopeProvider.instance.set(scope);	

			logger.trace("GCUBE Storage scope : " + scope);

			storage = new StorageClient("org.gcube.portlets.user", serviceName,
					portalLogin, AccessType.SHARED, currentScope, false).getClient();

			Handler.activateProtocol();

		} catch (Exception e) {
			logger.error("GCUBE Storage initialization failed",e);
		}
		return storage;
	}



}
