package org.gcube.common.homelibrary.client;

import java.io.File;
import java.net.URL;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.common.homelibrary.client.servlet.Servlets;
import org.gcube.common.homelibrary.client.util.Config;
import org.junit.Assert;

public class FTPClient {

	private String currentPath;

	public FTPClient() {
		this.currentPath = Config.ROOT_PATH;
	}

	/**
	 * This method asks and returns the current working directory.
	 * @return path The path to the current working directory.
	 * @throws Exception If the operation fails.
	 */
	public String currentDirectory() throws Exception {
		if ((currentPath == null) || (!currentPath.contains(Config.ROOT_PATH)))
			throw new Exception("current directory not set");
		return currentPath;
	}


	/**
	 * This method changes the current working directory.
	 * @param path The path to the new working directory.
	 * @throws Exception If the operation fails.
	 */
	public void changeDirectory(String path) throws Exception {
		if (exists(path))
			this.currentPath = currentDirectory() + File.separator + path;
		else
			throw new Exception(path + " does not exist");
	}

	/**
	 * Check if a directory exists
	 * @param dir The path to the new working directory.
	 * @return true if the directory exists.
	 */
	private boolean exists(String dir){
		try {
			Servlets.listFolder(currentDirectory() + File.separator + dir);
		} catch (Exception e) {			
			return false;
		}
		return true;
	}


	/**
	 * This method changes the current working directory to the parent one.
	 * @throws Exception  If the operation fails.
	 */
	public void changeDirectoryUp() throws Exception {

		String path = currentDirectory().
				substring(0,currentDirectory().lastIndexOf(File.separator));

		if (!path.contains(Config.ROOT_PATH))
			throw new Exception("Operation not Allowed");

		this.currentPath = path;

	}

	/**
	 * This method deletes a remote file.
	 * @param path The path to the file.
	 * @throws Exception  If the operation fails.
	 */
	public void deleteFile(String path) throws Exception {
		deleteDirectory(path);
	}

	/**
	 * This method deletes a remote directory.
	 * @param path  The path to the directory.
	 * @throws Exception  If the operation fails.
	 */
	public void deleteDirectory(String path) throws Exception {

		String dir = currentDirectory() + File.separator + path;
		if (dir == Config.ROOT_PATH)
			throw new Exception("Root cannot be deleted");
		if (!dir.contains(Config.ROOT_PATH))
			throw new Exception(currentDirectory() + " cannot be deleted");

		Boolean flag = Servlets.delete(dir);
		if (!flag)
			throw new Exception("Impossible to delete " + path);
	}


	/**
	 * This method lists the entries of the current working directory
	 * @return The list of the files (and directories) in the current working directory.
	 * @throws Exception If the operation fails.
	 */
	public FTPFile[] list() throws Exception {

		FTPFile[] list = null;
		try{
			Map<String, Boolean> files = Servlets.listFolder(currentDirectory());
			list = new FTPFile[files.size()];

			int i = 0;
			Set<String> keys = files.keySet();

			for(String key: keys){
				Boolean isFolder = files.get(key);
				list[i] = new FTPFile(key, isFolder);
				i++;
			}
		}catch (Exception e){
			throw new Exception("Impossible to list " + currentDirectory());
		}
		return list;
	}

	/**
	 * This method creates a new remote directory in the current working one.
	 * @param directoryName The name of the new directory.
	 * @param description A description for the new directory.
	 * @throws Exception If the operation fails.
	 */
	public void createDirectory(String directoryName, String description) throws Exception {
		try{
			Servlets.createFolder(directoryName, description, currentDirectory());
		}catch (Exception e){
			throw new Exception("Impossible to create Directory " + directoryName);
		}
	}

	/**
	 * This method uploads a file to the remote server
	 * @param file The file to upload.
	 * @param name A name for the file.
	 * @param description A description for the file.
	 * @throws Exception If the operation fails.
	 */
	public void upload(File file, String name, String description) throws Exception {

		try{
			Servlets.uploadFile(file, name, description, currentDirectory(), "UTF-8");
		}catch (Exception e){
			throw new Exception("Impossible to upload file " + file.getAbsolutePath());
		}
	}


	/**
	 * This method uploads a file to the remote server
	 * @param file The file to upload.
	 * @throws Exception If the operation fails.
	 */
	public void upload(File file) throws Exception {
		upload(file, null, "");
	}

	/**
	 * Run tests
	 * @throws Exception
	 */
	public void test() throws Exception {
		FTPClient client = new FTPClient();
		System.out.println("Current dir " + client.currentDirectory());
		System.out.println("List Root ");
		FTPFile[] list = client.list();
		for (FTPFile item:list)
			System.out.println("* " + item.getName() + " is folder? " + item.isFolder());

		System.out.println("\n\n");

		String A_name = "A-"+ UUID.randomUUID().toString();
		String B_name = "B-"+ UUID.randomUUID().toString();

		client.createDirectory(A_name, "test A in ROOT");
		System.out.println("Folder "+ A_name + " created in "+ client.currentDirectory());
		client.changeDirectory(A_name);
		System.out.println("Current dir " + client.currentDirectory());
		Assert.assertEquals(client.currentDirectory(), Config.ROOT_PATH + "/" + A_name);

		System.out.println("\n\n");


		client.createDirectory(B_name, "test A in ROOT");
		System.out.println("Folder "+ B_name + " created in "+ client.currentDirectory());
		client.changeDirectory(B_name);
		System.out.println("Current dir " + client.currentDirectory());
		Assert.assertEquals(client.currentDirectory(), Config.ROOT_PATH + "/" + A_name + "/" + B_name);
		System.out.println("\n\n");

		System.out.println("Save images in " + client.currentDirectory());

		URL imageURL = JUnitTest.class.getClassLoader().getResource(Config.DEFAULT_IMAGE);
		for (int i= 0; i< 5; i++){
			client.upload(new File(imageURL.getFile()), "img-"+ UUID.randomUUID().toString()+".jpg", "my description");
		}
		System.out.println("\n\n");

		client.changeDirectoryUp();
		System.out.println("Current directory up " + client.currentDirectory());
		Assert.assertEquals(client.currentDirectory(), Config.ROOT_PATH + "/" + A_name);

		System.out.println("Delete folder " + B_name);
		client.deleteDirectory(B_name);
		System.out.println("\n\n");
		client.changeDirectoryUp();
		Assert.assertEquals(client.currentDirectory(), Config.ROOT_PATH);
		System.out.println("Current directory " + client.currentDirectory());

		System.out.println("Delete folder " + A_name);
		client.deleteDirectory(A_name);

		System.out.println("\n\n");
		System.out.println("HL CLI test successfully completed. ");
	}

}
