package org.gcube.common.keycloak;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import org.gcube.common.keycloak.model.ModelUtils;
import org.gcube.common.keycloak.model.TokenIntrospectionResponse;
import org.gcube.common.keycloak.model.TokenResponse;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.security.AuthorizedTasks;
import org.gcube.common.security.Owner;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultKeycloakClientLegacyIS extends DefaultKeycloakClient implements KeycloakClientLegacyIS {

	protected static Logger logger = LoggerFactory.getLogger(DefaultKeycloakClientLegacyIS.class);

	@Override
	public URL findTokenEndpointURL(String audience) throws KeycloakClientException {
		logger.debug("Checking ScopeProvider's scope presence and format");
		String originalScope = audience;
		if (originalScope == null || !originalScope.startsWith("/") || originalScope.length() < 2) {
			if (SecretManagerProvider.instance.get()!=null)
				originalScope = SecretManagerProvider.instance.get().getContext();
			else throw new KeycloakClientException(originalScope == null ? "Scope not found in ScopeProvider"
					: "Bad scope name found: " + originalScope);
		}
		logger.debug("Assuring use the rootVO to query the endpoint simple query. Actual scope is: {}", originalScope);
		String rootVOScope = "/" + originalScope.split("/")[1];
		logger.debug("Setting rootVO scope into provider as: {}", rootVOScope);

		Callable<List<AccessPoint>> endpoints = new Callable<List<AccessPoint>>() {
			@Override
			public List<AccessPoint> call() throws Exception {
				logger.debug("Creating simple query");
				SimpleQuery query = queryFor(ServiceEndpoint.class);
				query.addCondition(
						String.format("$resource/Profile/Category/text() eq '%s'", CATEGORY))
				.addCondition(String.format("$resource/Profile/Name/text() eq '%s'", NAME))
				.setResult(String.format("$resource/Profile/AccessPoint[Description/text() eq '%s']", DESCRIPTION));

				logger.debug("Creating client for AccessPoint");
				DiscoveryClient<AccessPoint> client = clientFor(AccessPoint.class);

				logger.trace("Submitting query: {}", query);
				return client.submit(query);
			}
		};
		List<AccessPoint> accessPoints = null;
		try {
			accessPoints= AuthorizedTasks.executeSafely( endpoints, new Secret() {

				@Override
				public boolean isRefreshable() {
					// TODO Auto-generated method stub
					return false;
				}

				@Override
				public boolean isExpired() {
					// TODO Auto-generated method stub
					return false;
				}

				@Override
				public Owner getOwner() {
					// TODO Auto-generated method stub
					return null;
				}

				@Override
				public Map<String, String> getHTTPAuthorizationHeaders() {
					// TODO Auto-generated method stub
					return null;
				}

				@Override
				public String getContext() {
					return rootVOScope;
				}
			});

		}catch (Throwable e) {
			throw new KeycloakClientException("error executing query to retrieve url", e);
		}



		if (accessPoints.size() == 0) {
			throw new KeycloakClientException("Service endpoint not found");
		} else if (accessPoints.size() > 1) {
			throw new KeycloakClientException("Found more than one endpoint with query");
		}
		String address = accessPoints.iterator().next().address();
		logger.debug("Found address: {}", address);
		try {
			return new URL(address);
		} catch (MalformedURLException e) {
			throw new KeycloakClientException("Cannot create URL from address: " + address, e);
		}
	}
	
	@Override
	public URL findTokenEndpointURL() throws KeycloakClientException {
		logger.debug("Checking ScopeProvider's scope presence and format");
		String originalScope = SecretManagerProvider.instance.get().getContext();
		return findTokenEndpointURL(originalScope);
	}

	@Override
	public URL computeIntrospectionEndpointURL() throws KeycloakClientException {
		return computeIntrospectionEndpointURL(findTokenEndpointURL());
	}

	@Override
	public TokenResponse queryOIDCToken(String clientId, String clientSecret) throws KeycloakClientException {
		return queryOIDCToken(findTokenEndpointURL(), clientId, clientSecret);
	}

	@Override
	public TokenResponse queryUMAToken(String clientId, String clientSecret, List<String> permissions)
			throws KeycloakClientException {

		return queryUMAToken(clientId, clientSecret, SecretManagerProvider.instance.get().getContext(), permissions);
	}

	@Override
	public TokenResponse queryUMAToken(TokenResponse oidcTokenResponse, String audience, List<String> permissions)
			throws KeycloakClientException {

		return queryUMAToken(findTokenEndpointURL(audience), constructBeareAuthenticationHeader(oidcTokenResponse), audience,
				permissions);
	}

	@Override
	public TokenResponse queryUMAToken(String clientId, String clientSecret, String audience,
			List<String> permissions) throws KeycloakClientException {

		return queryUMAToken(findTokenEndpointURL(audience), clientId, clientSecret, audience, permissions);
	}

	@Override
	public TokenResponse refreshToken(TokenResponse tokenResponse) throws KeycloakClientException {
		return refreshToken((String) null, tokenResponse);
	}

	@Override
	public TokenResponse refreshToken(String clientId, TokenResponse tokenResponse) throws KeycloakClientException {
		return refreshToken(clientId, null, tokenResponse);
	}

	@Override
	public TokenResponse refreshToken(String clientId, String clientSecret, TokenResponse tokenResponse)
			throws KeycloakClientException {

		return refreshToken(findTokenEndpointURL(), clientId, clientSecret, tokenResponse);
	}

	@Override
	public TokenResponse refreshToken(String refreshTokenJWTString) throws KeycloakClientException {
		try {
			String clientId = ModelUtils.getClientIdFromToken(ModelUtils.getRefreshTokenFrom(refreshTokenJWTString));
			return refreshToken(clientId, refreshTokenJWTString);
		} catch (Exception e) {
			throw new KeycloakClientException("Cannot construct access token object from token response", e);
		}
	}

	@Override
	public TokenResponse refreshToken(String clientId, String refreshTokenJWTString) throws KeycloakClientException {
		return refreshToken(clientId, null, refreshTokenJWTString);
	}

	@Override
	public TokenResponse refreshToken(String clientId, String clientSecret, String refreshTokenJWTString)
			throws KeycloakClientException {

		return refreshToken(findTokenEndpointURL(), clientId, clientSecret, refreshTokenJWTString);
	}

	@Override
	public TokenIntrospectionResponse introspectAccessToken(String clientId, String clientSecret,
			String accessTokenJWTString) throws KeycloakClientException {

		return introspectAccessToken(computeIntrospectionEndpointURL(), clientId, clientSecret, accessTokenJWTString);
	}

	@Override
	public boolean isAccessTokenVerified(String clientId, String clientSecret, String accessTokenJWTString)
			throws KeycloakClientException {

		return isAccessTokenVerified(computeIntrospectionEndpointURL(), clientId, clientSecret, accessTokenJWTString);
	}

}
