package org.gcube.common.keycloak;

import java.net.URL;
import java.util.List;
import java.util.Map;

import org.gcube.common.keycloak.model.JSONWebKeySet;
import org.gcube.common.keycloak.model.PublishedRealmRepresentation;
import org.gcube.common.keycloak.model.TokenIntrospectionResponse;
import org.gcube.common.keycloak.model.TokenResponse;

public interface KeycloakClient {

    public static final String PROD_ROOT_SCOPE = "/d4science.research-infrastructures.eu";
    public static final String OPEN_ID_URI_PATH = "protocol/openid-connect";
    public static final String TOKEN_URI_PATH = "token";
    public static final String JWK_URI_PATH = "certs";
    public static final String TOKEN_INTROSPECT_URI_PATH = "introspect";
    public static final String AVATAR_URI_PATH = "account-avatar";
    public final static String D4S_CONTEXT_HEADER_NAME = "X-D4Science-Context";

    public static String DEFAULT_REALM = "d4science";


    /**
     * Returns the Keycloak base {@link URL} for the given context and the default realm (<code>d4science</code>)
     * 
     * @param context the context where the endpoint is needed (e.g. <code>/gcube</code> for DEV)
     * @return the Keycloak <code>token</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL getRealmBaseURL(String context) throws KeycloakClientException;

    /**
     * Returns the Keycloak base {@link URL} for the given context and in the given realm.
     * 
     * @param context the context where the endpoint is needed (e.g. <code>/gcube</code> for DEV)
     * @param realm the realm to use to construct the base URL
     * @return the Keycloak <code>token</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL getRealmBaseURL(String context, String realm) throws KeycloakClientException;

    /**
     * Constructs the Keycloak <code>token</code> endpoint {@link URL} from the realm's base URL.
     * 
     * @param realmBaseURL the realm's base URL to use
     * @return the Keycloak <code>token</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL getTokenEndpointURL(URL realmBaseURL) throws KeycloakClientException;

    /**
     * Constructs the Keycloak <code>JWK</code> endpoint {@link URL} from the realm's base URL.
     * 
     * @param realmBaseURL the realm's base URL to use
     * @return the Keycloak <code>JWK</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL getJWKEndpointURL(URL realmBaseURL) throws KeycloakClientException;

    /**
     * Constructs the Keycloak <code>introspection</code> endpoint {@link URL} from the realm's base URL.
     * 
     * @param realmBaseURL the realm's base URL to use
     * @return the Keycloak <code>introspection</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL getIntrospectionEndpointURL(URL realmBaseURL) throws KeycloakClientException;

    /**
     * Compute the keycloak <code>introspection</code> endpoint {@link URL} starting from the provided token endpoint.
     * 
     * @param tokenEndpointURL the token endpoint to use in the compute
     * @return the keycloak <code>introspection</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL computeIntrospectionEndpointURL(URL tokenEndpointURL) throws KeycloakClientException;

    /**
     * Constructs the Keycloak <code>avatar</code> endpoint {@link URL} from the realm's base URL.
     * 
     * @param realmBaseURL the realm's base URL to use
     * @return the Keycloak <code>avatar</code> endpoint URL
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL getAvatarEndpointURL(URL realmBaseURL) throws KeycloakClientException;

    /**
     * Gets the realm info setup (RSA <code>public_key</code>, <code>token-service</code> URL,
     * <code>account-service</code> URL and <code>tokens-not-before</code> setting)
     * 
     * @param realmURL the realm URL
     * @return the configured realm info
     * @throws KeycloakClientException if something goes wrong getting realm info
     */
    PublishedRealmRepresentation getRealmInfo(URL realmURL) throws KeycloakClientException;

    JSONWebKeySet getRealmJSONWebKeySet(URL jwkURL) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the context's Keycloak server, by using provided clientId and client secret.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(String context, String clientId, String clientSecret) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the context's Keycloak server, by using provided clientId and client secret.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(String context, String clientId, String clientSecret, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided clientId and client secret.
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(URL tokenURL, String clientId, String clientSecret) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided clientId and client secret.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(URL tokenURL, String clientId, String clientSecret, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(String context, String authorization) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(String context, String authorization, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(URL tokenURL, String authorization) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCToken(URL tokenURL, String authorization, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token from the context's Keycloak server, by using provided clientId and client secret, reducing the audience to the requested one.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(String context, String clientId, String clientSecret, String audience)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token from the context's Keycloak server, by using provided clientId and client secret, reducing the audience to the requested one.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(String context, String clientId, String clientSecret, String audience, Map<String, String> extraHeaders)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided clientId and client secret, reducing the audience to the requested one.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(URL tokenURL, String clientId, String clientSecret, String audience)
            throws KeycloakClientException;
    
    /**
     * Queries an OIDC token from the Keycloak server, by using provided clientId and client secret, reducing the audience to the requested one.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(URL tokenURL, String clientId, String clientSecret, String audience, Map<String, String> extraHeaders)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization, reducing the audience to the requested one.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(String context, String authorization, String audience)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization, reducing the audience to the requested one.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(String context, String authorization, String audience, Map<String, String> extraHeaders)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization, reducing the audience to the requested one.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(URL tokenURL, String authorization, String audience)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token from the Keycloak server, by using provided authorization, reducing the audience to the requested one.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenWithContext(URL tokenURL, String authorization, String audience, Map<String, String> extraHeaders)
            throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param username the user's username
     * @param password the user's password
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUser(String context, String clientId, String clientSecret, String username,
            String password) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param username the user's username
     * @param password the user's password
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUser(String context, String clientId, String clientSecret, String username,
            String password, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(String context, String authorization, String username,
            String password, String audience) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the Keycloak server, by using provided clientId and client secret and user's username and password, reducing the audience to the requested one.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(String context, String clientId, String clientSecret, String username,
            String password, String audience) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the Keycloak server, by using provided clientId and client secret and user's username and password, reducing the audience to the requested one.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(String context, String clientId, String clientSecret,
            String username, String password, String audience, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password, reducing the audience to the requested one.
     * 
     * The implementation uses the custom <code>X-D4Science-Context</code> HTTP header that the proper mapper on Keycloak uses to reduce the audience
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(URL tokenURL, String clientId, String clientSecret, String username,
            String password, String audience) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password, , reducing the audience to the requested one.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(URL tokenURL, String clientId, String clientSecret,
            String username, String password, String audience, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(String context, String authorization, String username,
            String password, String audience, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(URL tokenURL, String authorization, String username, String password,
            String audience) throws KeycloakClientException;

    /**
     * Queries an OIDC token for a specific user from the context's Keycloak server, by using provided clientId and client secret and user's username and password.
     * Optionally extra HTTP headers can be provided to be used in the call.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param username the user's username
     * @param password the user's password
     * @param audience an optional parameter to shrink the token's audience to the requested one (e.g. a specific context), by leveraging on the custom HTTP header and corresponding mapper on Keycloak
     * @param extraHeaders extra HTTP headers to add to the request
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryOIDCTokenOfUserWithContext(URL tokenURL, String authorization, String username,
            String password, String audience, Map<String, String> extraHeaders) throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided authorization, for the given audience (context),
     * in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience the audience (context) where to request the issuing of the ticket (URLEncoded)
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String context, String authorization, String audience, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided authorization, for the given audience (context),
     * in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience the audience (context) where to request the issuing of the ticket (URLEncoded)
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, String authorization, String audience, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using access-token provided by the {@link TokenResponse} object
     * for the given audience (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param oidcTokenResponse the previously issued token as {@link TokenResponse} object
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String context, TokenResponse oidcTokenResponse, String audience,
            List<String> permissions) throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using access-token provided by the {@link TokenResponse} object
     * for the given audience (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param oidcTokenResponse the previously issued token as {@link TokenResponse} object
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, TokenResponse oidcTokenResponse, String audience,
            List<String> permissions) throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided clientId and client secret for the given audience
     * (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String context, String clientId, String clientSecret, String audience,
            List<String> permissions) throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided clientId and client secret for the given audience
     * (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, String clientId, String clientSecret, String audience,
            List<String> permissions) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server using the refresh token JWT encoded string in the
     * token response object.
     * 
     * Client id will be read from "issued for" access token's claim and client secret will be not sent.
     * <br><b>NOTE</b>: For <code>public</code> clients types only.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String context, TokenResponse tokenResponse) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server using the refresh token JWT encoded string in the
     * token response object.
     * 
     * Client id will be read from "issued for" access token's claim and client secret will be not sent.
     * <br><b>NOTE</b>: For <code>public</code> clients types only.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(URL tokenURL, TokenResponse tokenResponse) throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server using the refresh token JWT encoded string in the
     * token response object and the provided client id and secret.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the requestor client id, may be <code>null</code> and in this case will be take from the access token "issued for" claim
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String context, String clientId, String clientSecret, TokenResponse tokenResponse)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server using the refresh token JWT encoded string in the
     * token response object and the provided client id and secret.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param clientId the requestor client id, may be <code>null</code> and in this case will be take from the access token "issued for" claim
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param tokenResponse the previously issued token as {@link TokenResponse} object
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(URL tokenURL, String clientId, String clientSecret, TokenResponse tokenResponse)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server by using the client id and secret
     * and the refresh token JWT encoded string obtained with the access token in the previous token response.
     *
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param refreshTokenJWTString the previously issued refresh token JWT string
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(String context, String clientId, String clientSecret, String refreshTokenJWTString)
            throws KeycloakClientException;

    /**
     * Refreshes a previously issued token from the Keycloak server by using the client id and secret
     * and the refresh token JWT encoded string obtained with the access token in the previous token response.
     * 
     * @param tokenURL the token endpoint {@link URL} of the OIDC server
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret, may be <code>null</code> for non-confidential clients
     * @param refreshTokenJWTString the previously issued refresh token JWT string
     * @return the refreshed token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the refresh query
     */
    TokenResponse refreshToken(URL tokenURL, String clientId, String clientSecret, String refreshTokenJWTString)
            throws KeycloakClientException;
    
    TokenResponse exchangeTokenForAccessToken(URL tokenURL, String oidcAccessToken, String clientId,
            String clientSecret, String audience) throws KeycloakClientException;
    
    TokenResponse exchangeTokenForAccessToken(String context, String oidcAccessToken, String clientId,
            String clientSecret, String audience) throws KeycloakClientException;

    TokenResponse exchangeTokenForRefreshToken(URL tokenURL, String oidcAccessToken, String clientId,
            String clientSecret, String audience) throws KeycloakClientException;

    TokenResponse exchangeTokenForRefreshToken(String context, String oidcAccessToken, String clientId,
            String clientSecret, String audience) throws KeycloakClientException;

//    TokenResponse exchangeTokenForOfflineToken(URL tokenURL, String oidcAccessToken, String clientId,
//            String clientSecret, String audience) throws IllegalArgumentException, KeycloakClientException;

//    TokenResponse exchangeTokenForOfflineToken(String context, String oidcAccessToken, String clientId,
//            String clientSecret, String audience) throws IllegalArgumentException, KeycloakClientException;

    /**
     * Introspects an access token against the Keycloak server.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret
     * @param accessTokenJWTString the access token to verify
     * @return a {@link TokenIntrospectionResponse} object with the introspection results; in particular, the <code>active</code> field represents the token validity
     * @throws KeycloakClientException if something goes wrong performing the verification
     */
    TokenIntrospectionResponse introspectAccessToken(String context, String clientId, String clientSecret,
            String accessTokenJWTString) throws KeycloakClientException;

    /**
     * Introspects an access token against the Keycloak server.
     * 
     * @param introspectionURL the introspection endpoint {@link URL} of the Keycloak server
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret
     * @param accessTokenJWTString the access token to verify
     * @return a {@link TokenIntrospectionResponse} object with the introspection results; in particular, the <code>active</code> field represents the token validity
     * @throws KeycloakClientException if something goes wrong performing the verification
     */
    TokenIntrospectionResponse introspectAccessToken(URL introspectionURL, String clientId, String clientSecret,
            String accessTokenJWTString) throws KeycloakClientException;

    /**
     * Verifies an access token against the Keycloak server.
     * 
     * @param context the context where the Keycloak's is needed (e.g. <code>/gcube</code> for DEV)
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret
     * @param accessTokenJWTString the access token to verify
     * @return <code>true</code> if the token is active, <code>false</code> otherwise
     * @throws KeycloakClientException if something goes wrong performing the verification
     */
    boolean isAccessTokenVerified(String context, String clientId, String clientSecret, String accessTokenJWTString)
            throws KeycloakClientException;

    /**
     * Verifies an access token against the Keycloak server.
     * 
     * @param introspectionURL the introspection endpoint {@link URL} of the Keycloak server
     * @param clientId the requestor client id
     * @param clientSecret the requestor client secret
     * @param accessTokenJWTString the access token to verify
     * @return <code>true</code> if the token is active, <code>false</code> otherwise
     * @throws KeycloakClientException if something goes wrong performing the verification
     */
    boolean isAccessTokenVerified(URL introspectionURL, String clientId, String clientSecret,
            String accessTokenJWTString) throws KeycloakClientException;

    byte[] getAvatarData(String context, TokenResponse tokenResponse) throws KeycloakClientException;

    byte[] getAvatarData(URL avatarURL, TokenResponse tokenResponse) throws KeycloakClientException;
 
}