/**
 * University of Illinois/NCSA Open Source License
 *
 * Copyright (c) 2001,2002 The Board of Trustees of the University of Illinois
 * All rights reserved.
 *
 * Developed by:  Open Archives Initiative Metadata Harvesting Project
 *                University of Illinois at Urbana-Champaign
 *                http://oai.grainger.uiuc.edu/
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal with the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 *  . Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimers.
 *  . Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimers in the
 *    documentation and/or other materials provided with the distribution.
 *  . Neither the names of Open Archives Initiative Metadata Harvesting
 *    Project, University of Illinois at Urbana-Champaign, nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this Software without specific prior written permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS WITH THE SOFTWARE.
 *
 * DC.Title:          OAIRecord
 *
 * DC.Contributor:    Yuping Tseng, ytseng1@uiuc.edu, University og Illinois at Urbana-Champaign
 * DC.Contributor:    Tom Habing, thabing@uiuc.edu, University of Illinois at Urbana-Champaign
 *
 * DC.Date.Modified:  5/14/02 3:00p
 *
 * DC:Rights:         University of Illinois/NCSA Open Source License
 *
 * DC.Relation:       For more information see http://oai.grainger.uiuc.edu/
 */

package org.gcube.common;

import java.util.ArrayList;
import java.util.List;

import javax.xml.transform.*;
import org.apache.xpath.*;
import org.w3c.dom.*;

/**
 * This class represents an OAI Record. The record may be an Identifier-only record, meaning it contains only the record identifier 
 *  and possibly a status. If an attempt is made to access the metadata from an Identifier-only record, a GetRecord will be performed, 
 *  turning the recored into a full record.
 *
 * This object is returned either by the OAIRepository.GetRecord method or by the OAIRecordList.CurrentItem property.
 */
public class OAIRecord {

	private List<String> sets;
	private String strStatus;
	private String strIdentifier;
	private String strDatestamp;
	private String strMetadataPrefix;
	private String strMetadata;
	private boolean boolIdOnly;
	private boolean boolValid;
	private Node xmlRecord;
	private OAIRepository repo;

	/**
	 * Constructs an empty OAI record.
	 */
	public OAIRecord() {
		sets = new ArrayList<String>();
		strStatus = "";
		strIdentifier = "";
		strDatestamp = "";
		strMetadataPrefix = "";
		xmlRecord = null;
		boolIdOnly = true;
		boolValid = true;
		strMetadata = "";
	}

	/**
	 * Returns the namespace URI of the metadata section of the record.  If there is no metadata an empty string is returned. 
	 */
	public String getMetadataNamespaceURI() throws OAIException {
		Node node;
		String ret = "";

		priCheckIdOnly();

		try {
			node = XPathAPI.selectSingleNode(xmlRecord, "oai:metadata/*", repo.getNamespaceNode());

			if (node != null) {
				ret = node.getNamespaceURI();
			}
		} catch (TransformerException te) {
			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
		}

		return ret;
	}

	/**
	 * Returns the namespace URI of the first about section of the record.  If there is no about section an empty string is returned.
	 */
	public String getAboutNamespaceURI() throws OAIException {
		return getAboutNamespaceURI(0);  
	}

	/**
	 * Returns the namespace URI of the about section of the record.  If there is no about section an empty string is returned.
	 */
	public String getAboutNamespaceURI(int i) throws OAIException {
		Node node;
		String ret = "";

		priCheckIdOnly();

		try {   
			node = XPathAPI.selectSingleNode(xmlRecord, "oai:about[" + (i + 1) + "]/*", repo.getNamespaceNode());

			if (node != null) {
				ret = node.getNamespaceURI();
			}
		} catch (TransformerException te) {
			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
		}

		return ret;  
	}


	/**
	 * If this is an id-only record, do a refresh to convert it to a complete record
	 */
	private void priCheckIdOnly() throws OAIException {
		if (boolIdOnly) {
			refreshRecord();
		}
	}

	/**
	 * Returns the datestamp of this record as a string.
	 *  If this is a IdOnly record it will be transformed into a complete record
	 */
	public String getDatestamp() throws OAIException {

		if (repo.getProtocolMajorVersion() < 2) {
			priCheckIdOnly();
		}

		return strDatestamp;
	}

	/**
	 * Returns true if the record has a status attribute of 'deleted'. 
	 */
	public boolean deleted() {
		if (strStatus.equals("deleted")) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Returns true if the record appears to be valid (well-formed, and if the Validation is Very Strict also valid according to 
	 *  the XML Schemas); if the Validation is Loose and the record is not well-formed, false is returned. 
	 */
	public boolean isRecordValid() {
		return boolValid;
	}

	/**
	 * Returns the OAI identifier of the record.
	 */
	public String getIdentifier() {
		return strIdentifier;
	}

	/**
	 * Returns the status of the record
	 */ 
	public String getStatus() {
		return strStatus;
	}

	/**
	 * Returns the metadata section of the record as an XML node. If there is no metadata section null node is returned.
	 */
	public Node getMetadata() throws OAIException {
		Node ret = null;

		priCheckIdOnly();

		try {
			Node node = XPathAPI.selectSingleNode(xmlRecord, "oai:metadata/*", repo.getNamespaceNode());

			if (node != null) {
				ret = node.cloneNode(true);
			}
		} catch (TransformerException te) {
			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
		}

		return ret;
	}

	/**
	 * Return the number of about elements associated with the record
	 */
	public int getAboutCount() throws OAIException {
		int ret = 0;

		priCheckIdOnly();

		try {
			NodeList list = XPathAPI.selectNodeList(xmlRecord, "oai:about/*", repo.getNamespaceNode());
			ret = list.getLength();
		} catch (TransformerException te) {
			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
		}  

		return ret;
	}

	/**
	 * Returns the first about section of the record as an XML node. If there is no about section null node is returned.
	 */
	public Node getAbout() throws OAIException {
		return getAbout(0);
	}

	/**
	 * Returns the about section of the record as an XML node. If there is no about section null node is returned.
	 */
	public Node getAbout(int i) throws OAIException {
		Node ret = null;

		priCheckIdOnly();

		try {
			Node node = XPathAPI.selectSingleNode(xmlRecord, "oai:about[" + (i + 1) + "]/*", repo.getNamespaceNode());

			if (node != null) {
				ret = node.cloneNode(true);
			}
		} catch (TransformerException te) {
			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
		}

		return ret;
	}

	/**
	 * Returns true if this is an identifier-only record, meaning that it was probably created as a result of a ListIdentifiers 
	 *  request; else false; an identifier-only record can be explicitly converted to a complete record by calling the 
	 *  refreshRecord method; accessing any of the methods which return metadata will also implicitly convert it to 
	 *  a complete record. 
	 */
	public boolean isIdentifierOnly() {
		return boolIdOnly;
	}

	/**
	 * Returns the metadata prefix of the record.
	 */
	public String getMetadataPrefix() throws OAIException {
		if (repo.getProtocolMajorVersion() < 2) {
			priCheckIdOnly();
		}

		return strMetadataPrefix;
	}

	/**
	 * Returns the complete record as an XML node; includes everything starting at the <record> tag.
	 */
	public Node getRecord() throws OAIException {
		priCheckIdOnly();
		return xmlRecord.cloneNode(true);
	}

	/**
	 * This will re-retrieve the record from the repository; this can be used to turn a identifier-only record into a complete 
	 * record; the current metadataPrefix is used. 
	 */
	public void refreshRecord() throws OAIException {
		if (strMetadataPrefix!=null){
			if (strMetadataPrefix.length() == 0) {
				strMetadataPrefix = "oai_dc";
			}
		}else
			strMetadataPrefix = "oai_dc";
		refreshRecord(strMetadataPrefix);
	}

	/**
	 * This will re-retrieve the record from the repository; this can be used to turn a identifier-only record into a complete 
	 * record or to change the record to a different metadataPrefix.
	 */
	public void refreshRecord(String meta) throws OAIException {
		// Use GetRecord to retrieve the record, possibly using a new metadata prefix
		// Update this record with the new record values
		OAIRecord rec = repo.getRecord(strIdentifier, meta);

		if (rec == null) {
			throw new OAIException(OAIException.FAILED_REFRESH_ERR, "Record failed to refresh, illegal or nonexisting identifier?");
		} else {
			this.frndSetDatestamp(rec.getDatestamp());
			this.frndSetIdentifier(rec.getIdentifier());
			this.frndSetIdOnly(rec.isIdentifierOnly());
			this.frndSetMetadataPrefix(meta);
			this.frndSetStatus(rec.deleted() ? "deleted" : "");
			this.frndSetRecord(rec.getRecord());
			this.frndSetValid(rec.isRecordValid());
		}

		boolIdOnly = false;
	}   

	/**
	 * Return the number of setSpec strings for the record. OAI 1.x repositories will always return 0
	 */
	public int getSetSpecCount() throws OAIException {

		int ret = 0;

		try {
			NodeList list = XPathAPI.selectNodeList(xmlRecord, "//oai:setSpec", repo.getNamespaceNode());
			ret = list.getLength();
		} catch (TransformerException te) {
			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
		}

		return ret;

	}

	/**
	 * Get the first setSpec strings associated with the record.  Only valid for OAI 2.0 repositories
	 */
	public String getSetSpec() throws OAIException {
		return getSetSpec(0);
	}


	/**
	 * Get the list of sets associated with the record.
	 */
	public List<String> getSetSpeList() throws OAIException {

		return this.sets;
	}


	//	/**
	//	 * Get the first setSpec strings associated with the record.  Only valid for OAI 2.0 repositories
	//	 */
	//	public int getSetSpeCount1() throws OAIException {
	//
	//		int ret = 0;
	//
	//		priCheckIdOnly();
	//		System.out.println(xmlRecord.toString());
	//		try {
	//			NodeList list = XPathAPI.selectNodeList(xmlRecord, "//oai:setSpec", repo.getNamespaceNode());
	//			ret = list.getLength();
	//		} catch (TransformerException te) {
	//			throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
	//		}  
	//
	//		return ret;
	//
	//	}


	/**
	 * Get one of the setSpec strings associated with the record.  Only valid for OAI 2.0 repositories
	 */
	public String getSetSpec(int i) throws OAIException {
		String ret = "";

		if (repo.getProtocolMajorVersion() < 2) {
			throw new OAIException(OAIException.OAI_2_ONLY_ERR, "'setSpec' of header is not supported.");
		}

		if (getSetSpecCount() > 0 && i < getSetSpecCount()) {
			try {
				Node n = XPathAPI.selectSingleNode(xmlRecord, "//oai:setSpec[" + (i + 1) + "]/text()", repo.getNamespaceNode());
				if (n != null) {
					ret = n.getNodeValue();
				}
			} catch (TransformerException te) {
				throw new OAIException(OAIException.CRITICAL_ERR, te.getMessage());
			}
		}

		return ret;
	}

	/**
	 * Purpose: Set the identifier of the record
	 */
	protected void frndSetIdentifier(String i) {
		strIdentifier = i;
	}

	/**
	 * Purpose: Set the datestamp of the record
	 */
	protected void frndSetDatestamp(String d) {
		strDatestamp = d;
	}

	/**
	 * Purpose: Set the repository from which the record is derived
	 */
	protected void frndSetRepository(OAIRepository r) {
		repo = r;
	}

	/**
	 * Purpose: Set the metadata prefix of the record
	 */
	protected void frndSetMetadataPrefix(String m) {
		strMetadataPrefix = m;
	}

	/**
	 * Purpose: Set whether the record is valid or not
	 */
	protected void frndSetValid(boolean v) {
		boolValid = v;
	}

	/**
	 * Purpose: Set whether this record is an id-only record
	 */
	protected void frndSetIdOnly(boolean b) {
		boolIdOnly = b;
	}

	/**
	 * Purpose: Set the DOM Node of the record
	 */
	protected void frndSetRecord(Node n) {
		xmlRecord = n;
	}

	/**
	 * Purpose: Set the status of the record
	 */
	protected void frndSetStatus(String s) {
		strStatus = s;
	}

	/**
	 * Purpose: Set the status of the record
	 */
	protected void addSet(String s) {
		sets.add(s);
	}

	public void setMetadataRecord(String s) {
		strMetadata = s;

	}

	public String getMetadataRecord() {
		return this.strMetadata;

	}






}
