package org.gcube.contentmanagement.timeseriesservice.impl.codelist.wrappers;

import java.util.Arrays;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.Limit;
import org.gcube.common.dbinterface.Order;
import org.gcube.common.dbinterface.Specification;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.persistence.annotations.FieldDefinition;
import org.gcube.common.dbinterface.persistence.annotations.TableRootDefinition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.tables.Table;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.entities.TableField.ColumnType;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListCuration;
import org.gcube.contentmanagement.codelistmanager.util.Constants;
import org.gcube.contentmanagement.timeseriesservice.impl.editing.CodeListRelationEdit;
import org.gcube.contentmanagement.timeseriesservice.impl.editing.ColumnEditor;
import org.gcube.contentmanagement.timeseriesservice.impl.exceptions.EditNotFinishedException;
import org.gcube.contentmanagement.timeseriesservice.impl.exceptions.NotInEditModeException;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;

@TableRootDefinition
public class CLCurationWrapper {

	private static GCUBELog logger = new GCUBELog(CLCurationWrapper.class);
	
	@FieldDefinition()
	private ColumnEditor edit;	
	
	@FieldDefinition(precision={60}, specifications={Specification.NOT_NULL})
	private String owner;
	
	@FieldDefinition(precision={60}, specifications={Specification.NOT_NULL, Specification.PRIMARY_KEY})
	private String codelistId;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private GCUBEScope scope;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private Status status= Status.Initializing;
	
	@FieldDefinition(precision={60})
	private String parentId;
	
	
	
	/**
	 * @return the parentId
	 */
	public String getParentId() {
		return parentId;
	}

	/**
	 * @return the status
	 */
	public Status getStatus() {
		return status;
	}

	/**
	 * @param status the status to set
	 */
	public void setStatus(Status status) {
		this.status = status;
	}

	/**
	 * @return the owner
	 */
	public String getOwner() {
		return owner;
	}

	/**
	 * @param owner the owner to set
	 */
	public void setOwner(String owner) {
		this.owner = owner;
	}

	/**
	 * @return the codelistId
	 */
	public String getCodelistId() {
		return codelistId;
	}

	/**
	 * @return the scope
	 */
	public GCUBEScope getScope() {
		return scope;
	}

	/**
	 * @param scope the scope to set
	 */
	public void setScope(GCUBEScope scope) {
		this.scope = scope;
	}

	@SuppressWarnings("unused")
	private CLCurationWrapper(){};
	
	public CLCurationWrapper(String owner, String codelistId, GCUBEScope scope) {
		super();
		this.owner = owner;
		this.codelistId = codelistId;
		this.scope = scope;
		this.edit =null;
	}	
	
	
	/**
	 * @return the codelistId
	 */
	public CodeListCuration getCodelist() throws Exception {
		return CodeListCuration.get(codelistId);
	}
	
	
	public synchronized void edit(String fieldId) throws Exception{
		edit(fieldId, null);
	}
		
	public synchronized void edit(String fieldId, Type type) throws Exception{
		if (edit!=null) edit.dismiss();
		CodeListCuration codelist = this.getCodelist(); 
		SimpleTable table = codelist.getTable();
		table.initializeFieldMapping();
		int[] fieldLength = codelist.getLabelFieldMapping().get(fieldId).getLength();
		logger.trace("fieldLength for "+fieldId+" is "+Arrays.toString(fieldLength));
		
		
		if (codelist.getLabelFieldMapping().get(fieldId).getColumnReference().getType()==ColumnType.ParentCode || codelist.getLabelFieldMapping().get(fieldId).getColumnReference().getType()==ColumnType.HLParentCode || codelist.getLabelFieldMapping().get(fieldId).getColumnReference().getType()==ColumnType.HLChildCode)
			switch (codelist.getCodelistType()) {
			case Simple:
				List<String> codesId = codelist.getFieldsIdByColumnType(ColumnType.Code);
				if (codesId.size()==0) throw new Exception(); //TODO: new Exception
				logger.debug("code id is "+codesId.get(0));
				logger.debug("the type is "+table.getFieldsMapping().get(codesId.get(0)).getType().getValue());
				this.edit = new CodeListRelationEdit(this.getCodelistId(), fieldId, table, fieldLength, false, codesId.get(0), Constants.ID_LABEL);
				break;
			case Hierarchical:
				logger.debug("entering in the hierarchical case");
				CodeList referenceCodelist= CodeList.get(codelist.getLabelFieldMapping().get(fieldId).getColumnReference().getCodelistReferenceId());
				this.edit = new CodeListRelationEdit(this.getCodelistId(), fieldId, table, fieldLength, false, referenceCodelist.getCodeColumnId(), referenceCodelist.getTable(), Constants.ID_LABEL);				
				break;
			default:
				throw new Exception(); //TODO: new Exception
			}
		else if (type!=null){
			this.edit = new ColumnEditor(this.getCodelistId(), fieldId, table, fieldLength, false, type, Constants.ID_LABEL);
		}else throw new Exception(); //TODO: create new Exception
		this.edit.initialize();
		if (!this.store()){
			logger.warn("error storing the curation");
			edit.dismiss();
			edit=null;
			logger.error("error entering in edit mode");
			throw new Exception("error entering in edit mode");
		}

	}
	
	public Status getEditStatus(){
		if (edit!=null && edit.getIsUnderInitialization()==Status.Open) return Status.Initializing;
		if (edit==null) return Status.Open;
		if (edit.getIsUnderInitialization()==Status.Error){
			edit.dismiss();
			edit=null;
			this.store();
			return Status.Error;
		}
		return edit.getIsUnderInitialization();
	}
	
	public boolean isUnderEditing(){
		return (edit!=null);
		
	}
	
	public synchronized void removeEdit() throws NotInEditModeException, Exception{
		if (edit!=null){
			edit.dismiss();
			edit=null;
			this.store();
		} else throw new NotInEditModeException();
	}
	
	public synchronized void saveEdit() throws NotInEditModeException, EditNotFinishedException, Exception{
		logger.debug("called saveEdit in CLCurationWrapper with edit "+(edit==null?"null":"not null") );
		if (edit!=null){
			if (this.getErrorCount()>0) throw new EditNotFinishedException();
			CodeListCuration codelist = this.getCodelist();
			try{
				edit.save();
				codelist.setLabelFieldMapping(this.getEditModeColumnsDefinition());
			}catch (Exception e) {
				logger.debug("error saving the edit",e);
			}
			codelist.getTable().initializeFieldMapping();
			codelist.updateCount();
			codelist.store();
			try{
				edit.dismiss();
			}catch (Exception e) {
				logger.debug("error dismissing edit",e);
			}
			edit = null;
			this.store();
		} else throw new NotInEditModeException();
	}
	
	public synchronized boolean remove(){
		try {
			getCodelist().remove();
			try{
				this.removeEdit();
			}catch (NotInEditModeException e) {
				logger.debug("revoming a codelist not in edit mode");
			}catch (Exception e) {
				logger.warn("error dismissing edit");
			}
			ObjectPersistency.get(CLCurationWrapper.class).deleteByKey(this.getCodelistId());
		} catch (Exception e) {
			return false;
		}
		return true;
		
	}
	
	public synchronized boolean store(){
		try{
			ObjectPersistency<CLCurationWrapper> op =ObjectPersistency.get(CLCurationWrapper.class);
			if (!op.existsKey(this.getCodelistId()))
				op.insert(this);
			else op.update(this);
		}catch (Exception e) {
			logger.error("error storing curation wrapper on DB",e);
			return false;
		}
		return true;
	}
	
	public static CLCurationWrapper get(String id) throws Exception{
		return ObjectPersistency.get(CLCurationWrapper.class).getByKey(id);
	}
	
	
	
	public String getDataAsJson(Select query) throws Exception {
		if (edit!=null)	return edit.getResultAsJson(query, true);
		else{
			query.setTables(new Table(this.getCodelist().getTableName()));
			return query.getResultAsJSon(true);
		}
	}
	
	public void replaceValue(String fieldId, int row, Object value) throws NotInEditModeException, Exception {
		if (edit!=null && edit.getFieldId().equals(fieldId))
			edit.replaceValue(row, value);
		else this.getCodelist().replaceValue(fieldId, value.toString(), row);
	}
	
	public TableField getColumnInEditMode() throws NotInEditModeException, Exception{
		if (edit==null) throw new NotInEditModeException();
		CodeListCuration codelist = this.getCodelist();
		TableField tf =codelist.getLabelFieldMapping().get(edit.getFieldId());
		//returning the column definition with the old DataType (portlet requirement)
		return new TableField(edit.getFieldId(), tf.getFieldName(), tf.getColumnReference(), tf.getDataType());
	}
	
	
	public Hashtable<String, TableField > getEditModeColumnsDefinition() throws Exception{
		if (edit==null) throw new NotInEditModeException();
		else{
			CodeListCuration codelist = this.getCodelist();
			Hashtable<String, TableField > tableFields = new Hashtable<String, TableField >();
			for (TableField tf : codelist.getLabelFieldMapping().values())
				if (tf.getId().equals(edit.getFieldId())) tableFields.put(edit.getFieldId(), new TableField(edit.getFieldId(), tf.getFieldName(), tf.getColumnReference(),edit.getDataType().getType(), tf.getLength()));
				else tableFields.put(tf.getId(), tf);
			return tableFields;
		}
	}
	
	public int getErrorCount() throws NotInEditModeException, Exception{
		if (edit==null) throw new NotInEditModeException(); 
		return (int)edit.errorCount();
	}
	
	public void removeAllErrors() throws NotInEditModeException, Exception{
		if (edit==null) throw new NotInEditModeException(); 
		edit.removeAllErrors();
	}
	
	public int getSize() throws Exception{
		SimpleTable table = new SimpleTable(this.getCodelist().getTableName());
		table.initializeCount();
		return table.getCount();
	}
	
	public String getCodesAsJson(Limit limit, Order order) throws Exception{
		Select select = DBSession.getImplementation(Select.class);
		if (this.getCodelist().getFieldsIdByColumnType(ColumnType.Code).size()==0) throw new Exception(); //TODO: new Exception
		select.setTables(new Table(this.getCodelist().getTableName()));
		select.setLimit(limit);
		select.setOrders(order);
		return select.getResultAsJSon(true);
	}
	
	public static Iterator<CLCurationWrapper> getByUser(String user) throws Exception{
		return ObjectPersistency.get(CLCurationWrapper.class).getObjectByField("owner", user).iterator();
	}
	
	public static Iterator<CLCurationWrapper> getAll() throws Exception{
		return ObjectPersistency.get(CLCurationWrapper.class).getAll().iterator();
	}
}
