package org.gcube.contentmanagement.timeseriesservice.impl.editing;

import java.io.Serializable;
import java.util.LinkedList;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.ColumnDefinition;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.conditions.OperatorCondition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Delete;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.utils.Utility;
import org.gcube.contentmanagement.timeseriesservice.impl.exceptions.OperationNotSupportedException;
import org.gcube.contentmanagement.timeseriesservice.impl.history.CurationHistoryItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;
public abstract class Edit extends EditProxy implements Serializable{

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private static GCUBELog logger= new GCUBELog(Edit.class);

	public enum TYPE{Dimesion, Column, CodeListRelation};
	
	protected TYPE editorType= null;
	
	protected String ROW_ID_LABEL= "ID";
	protected final String CURATION_TABLE_ALIAS= "curtab";
	protected final String DIMENSION_TABLE_ALIAS="dimtab";
	protected final String COUNT_ALIAS="count";
	
	private SimpleTable correctsTable= null;
	private SimpleTable wrongsTable = null;
	
	protected String fieldId;
	protected boolean withoutErrors;
	protected String resourceId;
	protected int[] fieldlength;
	protected Status isUnderInitialization;
	protected SimpleTable resourceTable;	
	protected ColumnDefinition[] internalColumndefinition;
	protected LinkedList<CurationHistoryItem> historyEditorList;
	
	
	
	public Edit(String resourceId, String fieldId, SimpleTable resourceTable, int[] fieldlength, boolean withoutError, String ... rowIdLabel){
		this.historyEditorList=new LinkedList<CurationHistoryItem>();
		this.resourceId= resourceId;
		this.resourceTable= resourceTable;
		this.fieldId = fieldId;
		//this.fieldlength= new HashMap<String, Integer>();
		this.fieldlength= fieldlength;
		this.withoutErrors= withoutError;
		if (rowIdLabel.length>0) this.ROW_ID_LABEL = rowIdLabel[0];
	}
	
	
	
	/**
	 * @return the fieldId
	 */
	public String getFieldId() {
		return fieldId;
	}

	
	
	/**
	 * @return the editorType
	 */
	public TYPE getEditorType() {
		return editorType;
	}



	/**
	 * @return the fieldlength
	 */
	public int[] getFieldLength() {
		return fieldlength;
	}


	protected SimpleTable getCorrectsTable(){
		if (this.correctsTable==null){
			this.correctsTable =new SimpleTable(getCorrectsTableName());
			try{
				this.correctsTable.initializeCount();
				this.correctsTable.initializeFieldMapping();
			}catch (Exception e) {
				logger.warn("error initializing corrects table", e);
			}
		} 
		return this.correctsTable;
	}
	
	protected SimpleTable getWrongsTable(){
		if (this.wrongsTable==null){
			this.wrongsTable =new SimpleTable(getWrongsTableName());
			try{
				this.wrongsTable.initializeCount();
				this.wrongsTable.initializeFieldMapping();
			}catch (Exception e) {
				logger.warn("error initializing wrongs table", e);
			}
		} 
		return this.wrongsTable;
	}
	
	public void initialize(){
		setIsUnderInitialization(Status.Open);
		DBSession session= null;
		logger.trace("entering in edit mode");
		try{
			session =DBSession.connect();
			long start= System.currentTimeMillis();
			internalInitialize(session);
			logger.debug("Enter in EditMode time is:"+(System.currentTimeMillis()-start)/1000+" sec" );
			setIsUnderInitialization(Status.Close);
		} catch (Exception e) {
			logger.error("failed trying to enter in edit mode", e);
			this.dismiss();
			setIsUnderInitialization(Status.Error);
		}finally {
			if (session!=null) session.release();
		}
	}
	
	
	protected abstract void internalInitialize(DBSession session) throws Exception;
	public abstract long check() throws Exception;
	
	
	
	
	/**
	 * saves the current set
	 * 
	 * @throws Exception -
	 */
	public void save() throws Exception{
		logger.debug("save started");
		if (this.errorCount()>0) throw new Exception("the Time Series is not totally curated");
		DBSession session= null;
		try{
			session = DBSession.connect();
			long start= System.currentTimeMillis();
			
			try{
				Utility.drop(getWrongsTableName()).execute(session);
			}catch (Exception e) {
				logger.debug("the wrong table have been already deleted");
			}
			String temporaryTable= this.resourceTable.getTableName()+"_TMP";
			
			Utility.renameTable(this.resourceTable, temporaryTable).execute(session);
			
			Utility.renameTable(getCorrectsTable(), this.resourceTable.getTableName()).execute(session);
					
			Utility.drop(temporaryTable).execute(session);
			
			
			this.resourceTable.initializeCount();
			this.resourceTable.initializeFieldMapping();
			logger.trace("save column time with drop "+(System.currentTimeMillis()-start)/1000+"s");
			logger.trace("save column time is "+(System.currentTimeMillis()-start)/1000+"s");
		}finally{
			if (session!=null)
				session.release();
		}
	}
	
	
	/**
	 * allows user to retrieve all the information in the table as JSon
	 * 
	 * @param limit the query limit
	 * @param onlyErrors if true the method returns only the wrong entries  
	 * @return a JSon string
	 * @throws Exception -
	 */
	public String getResultAsJson(Select query,boolean onlyErrors) throws OperationNotSupportedException, Exception{
		DBSession session = null;
		if (onlyErrors){
			try{
				session= DBSession.connect();
				query.setTables(getWrongsTable());
				return query.getResultAsJSon(session, true);
			}finally{if (session != null) session.release();}
		} else throw new OperationNotSupportedException();
	}
	
	/**
	 * return the number of errors for this column setting 
	 * 
	 * @return the error count
	 * @throws Exception
	 */
	public long errorCount() throws Exception{
		if (this.withoutErrors) return 0;
		getWrongsTable().initializeCount();
		return getWrongsTable().getCount();
	}
	
	/**
	 * the total entries count
	 * 
	 * @return the total entries count
	 * @throws Exception
	 */
	public long totalCount() throws Exception{
		getWrongsTable().initializeCount();
		getCorrectsTable().initializeCount();
		return getWrongsTable().getCount()+getCorrectsTable().getCount();
	}
	
	
	/**
	 *  exits from edit mode and removes all the temporary tables
	 */
	public void dismiss(){
		DBSession session=null;
		try{
			session=DBSession.connect();
			try{
				Utility.drop(this.getWrongsTableName()).execute(session);
			}catch (Exception e) {logger.warn("error deleting wrongs fields table dismissing the field editor");}
			try{
				Utility.drop(this.getCorrectsTableName()).execute(session);
			}catch (Exception e) {logger.warn("error deleting corrects fields table dismissing the field editor");}
		}catch(Exception e){logger.warn("an error occurs connection on the db");}
		finally{if (session != null) session.release();}
	}
		
		
	/**
	 * set the Status for the edit initialization
	 * 
	 * @return the Status
	 */
	public Status getIsUnderInitialization() {
		return isUnderInitialization;
	}

	/**
	 * sets the initialization Status
	 * 
	 * @param isUnderInitialization the Status
	 */
	public void setIsUnderInitialization(Status isUnderInitialization) {
		this.isUnderInitialization = isUnderInitialization;
	}
	
	/**
	 * 
	 * @param isUnderInitialization Status as string
	 */
	public void setIsUnderInitialization(String isUnderInitialization) {
		this.isUnderInitialization = Status.fromString(isUnderInitialization);
	}
	
	/**
	 * 
	 * @return
	 */
	public LinkedList<CurationHistoryItem> getHistoryItems() {
		return historyEditorList;
	}

	/**
	 * 
	 * @param historyEditorList
	 */
	public void setHistoryItems(
			LinkedList<CurationHistoryItem> historyEditorList) {
		this.historyEditorList = historyEditorList;
	}
	
	
	/**
	 * 
	 * @return the remained lines
	 * @throws Exception
	 */
	public int removeAllErrors() throws Exception{
		DBSession session = null;
		try{
			session = DBSession.connect();
			Delete delete = DBSession.getImplementation(Delete.class);
			delete.setTable(getWrongsTable());
			delete.execute(session);
		}finally{
			if (session!=null) session.release();
		}
		getCorrectsTable().initializeCount();
		return getCorrectsTable().getCount();
	}



	public int removeSingleError(long rowId) throws Exception{
		DBSession session = null;
		try{
			Delete delete = DBSession.getImplementation(Delete.class);
			delete.setTable(getWrongsTable());
			delete.setFilter(new OperatorCondition<SimpleAttribute, Long>(new SimpleAttribute(this.ROW_ID_LABEL), rowId, "="));
			delete.execute(session);
		}finally{
			if (session!=null) session.release();
		}
		
		return (int)this.totalCount();
	}
	
	protected String getCorrectsTableName(){
		return "C"+fieldId+resourceId.replaceAll("-", "_");
	}
	
	protected String getWrongsTableName(){
		return "W"+fieldId+resourceId.replaceAll("-", "_");
	}
}
