package org.gcube.contentmanagement.timeseriesservice.impl.codelist;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.Limit;
import org.gcube.common.dbinterface.Order;
import org.gcube.common.dbinterface.Order.OrderType;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.persistence.ObjectNotFoundException;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListCuration;
import org.gcube.contentmanagement.timeseriesservice.impl.codelist.wrappers.CLCurationWrapper;
import org.gcube.contentmanagement.timeseriesservice.impl.codelist.wrappers.CLManagerWrapper;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Export;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.AccessFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodeListType;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnsDefinitionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistIdentifier;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.DataAsJsonRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CodelistItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.CodelistItemsArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.CreateCodeListRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ItemNotFoundFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.CLExportRequest;

public class CodelistManager {

	private static GCUBELog logger = new GCUBELog(CodelistManager.class);
	
	public String createCodeList(CreateCodeListRequest request) throws GCUBEFault{
		final CodeListCuration curation;
		try{
			CLCurationWrapper curationWrapper = CLCurationWrapper.get(request.getCurationId());
			if (!curationWrapper.getOwner().equals(request.getUser()) || !ServiceContext.getContext().getScope().toString().equals(curationWrapper.getScope().toString()))
				throw new AccessFault();
			curation = curationWrapper.getCodelist();
		}catch (ObjectNotFoundException e) {
			throw new ItemNotFoundFault();
		} catch (Exception e) {
			throw new GCUBEFault(e);
		}
		
		final CodeList codelist= new CodeList(curation.getAgencyId(), curation.getName(), curation.getDescription(), curation.isFinal());
			
	
		final CLManagerWrapper managerWrapper = new CLManagerWrapper(codelist.getId(), request.getUser(), ServiceContext.getContext().getScope(), request.getSuperUsers());
		final Thread managerThread = new Thread(){
			public void run(){
				managerWrapper.setStatus(Status.Open);
				managerWrapper.store();
				if (!codelist.initialize(curation))	
					managerWrapper.setStatus(Status.Error);
				else
					managerWrapper.setStatus(Status.Close);
				managerWrapper.store();
			}
		};
		try {
			ServiceContext.getContext().useServiceCredentials(managerThread);
		} catch (Exception e) {
			logger.error("error initializing the codelist",e);
			throw new GCUBEFault(e);
		}
		
		managerThread.start();
		
		return codelist.getId();
		
	}
	
	public Status getCodeListStatus(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(identifier.getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			return clManagerWrapper.getStatus();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting data as json",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public String getDataAsJson(DataAsJsonRequest request) throws GCUBEFault {
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(request.getIdentifier().getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			Select selectQuery= DBSession.getImplementation(Select.class);
			selectQuery.setLimit(new Limit(request.getLimit().getLowerLimit(), request.getLimit().getUpperLimit()));
			selectQuery.setOrders(new Order((request.getOrder().getOrder()==org.gcube.contentmanagement.timeseriesservice.stubs.types.OrderType.Ascending)?OrderType.ASC:OrderType.DESC, new SimpleAttribute(request.getOrder().getField())));
			return clManagerWrapper.getDataAsJson(selectQuery);
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting data as json",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public CodelistColumnsDefinitionArray getColumnsDefinition(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(identifier.getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			return clManagerWrapper.getColumnsDefinition();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting columns defintion",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public CodelistItemsArray getCodelists(VOID req) throws GCUBEFault{
		try {
			Iterator<CLManagerWrapper> clManagerIterator = CLManagerWrapper.getByScope(ServiceContext.getContext().getScope());
			List<CodelistItem> clItemsArray = new ArrayList<CodelistItem>();
			while (clManagerIterator.hasNext()){
				CLManagerWrapper clWrapper = clManagerIterator.next();
				try{
					CodeList codelist = clWrapper.getCodelist();
					clItemsArray.add(new CodelistItem(codelist.getAgency(), CodeListType.fromValue(codelist.getCodelistType().toString()), codelist.getCreationDate(), clWrapper.getCreator(), codelist.getDescription(), codelist.getId(), codelist.getModificationDate(),  codelist.getName(), codelist.getCount(),  clWrapper.getStatus() , codelist.getVersion()));
				}catch (ObjectNotFoundException e) {
					logger.warn("the wrapper "+clWrapper.getCodelistId()+" is not bound to a codelist",e);
					clWrapper.remove();
				}
			}
			return new CodelistItemsArray(clItemsArray.toArray(new CodelistItem[clItemsArray.size()]));
		} catch (Exception e1) {
			logger.error("error getting codelists",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public CodelistItem getCodelistItem(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(identifier.getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			CodeList codelist = clManagerWrapper.getCodelist();
			return new CodelistItem(codelist.getAgency(), CodeListType.fromValue(codelist.getCodelistType().toString()),  codelist.getCreationDate(), clManagerWrapper.getCreator(), codelist.getDescription(), codelist.getId(), codelist.getModificationDate(),  codelist.getName(), codelist.getCount(), clManagerWrapper.getStatus() , codelist.getVersion());
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting columns defintion",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public VOID remove(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(identifier.getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			clManagerWrapper.remove();
			return new VOID();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting columns defintion",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public int getSize(String id) throws GCUBEFault{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(id);
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			return clManagerWrapper.getCodelist().getCount();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting columns defintion",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public String getCodes(DataAsJsonRequest request) throws GCUBEFault{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(request.getIdentifier().getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			Select selectQuery= DBSession.getImplementation(Select.class);
			selectQuery.setAttributes(new SimpleAttribute(clManagerWrapper.getCodelist().getCodeColumnId()));
			selectQuery.setLimit(new Limit(request.getLimit().getLowerLimit(), request.getLimit().getUpperLimit()));
			selectQuery.setOrders(new Order((request.getOrder().getOrder()==org.gcube.contentmanagement.timeseriesservice.stubs.types.OrderType.Ascending)?OrderType.ASC:OrderType.DESC, new SimpleAttribute(request.getOrder().getField())));
			return clManagerWrapper.getDataAsJson(selectQuery);
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting data as json",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	public String export(CLExportRequest request) throws Exception{
		try {
			CLManagerWrapper clManagerWrapper = CLManagerWrapper.get(request.getIdentifier().getId());
			if (!clManagerWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString())) throw new AccessFault();
			CodeList codeList= clManagerWrapper.getCodelist();
			return Export.codeListAsCsv(codeList.getTable(), codeList.getName(), request.isHeader());
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting data as json",e1);
			throw new GCUBEFault(e1);
		}
	}

	
}
