package org.gcube.contentmanagement.timeseriesservice.impl.timeseries;


import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.axis.components.uuid.UUIDGen;
import org.apache.axis.components.uuid.UUIDGenFactory;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableFault;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.state.GCUBEWSResourceKey;
import org.gcube.common.core.types.StringMap;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.contentmanagement.timeseriesservice.stubs.CreateFromTsRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CreateFromTsResponse;
import org.gcube.contentmanagement.timeseriesservice.stubs.CreateTsRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CreateTsResponse;
import org.gcube.contentmanagement.timeseriesservice.stubs.OpenMsg;
import org.gcube.contentmanagement.timeseriesservice.stubs.ServiceProperties;
import org.gcube.contentmanagement.timeseriesservice.stubs.TimeSeriesItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.TimeSeriesItemsArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.ServiceMode;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ColumnDefinition;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.EntryType;
import org.gcube.contentmanagement.timeseriesservice.impl.context.CurationContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.TimeSeriesContext;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.state.CurationResource;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.state.TimeSeriesResource;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Util;


/**
 * 
 * @author lucio
 *
 */
public class TimeSeriesFactory {
	
	GCUBELog logger= new GCUBELog(TimeSeriesFactory.class);
	
	private static final UUIDGen uuidGen = UUIDGenFactory.getUUIDGen();
	
	
	/**
	 * 	
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public CreateTsResponse create(CreateTsRequest request) throws GCUBEFault{
		logger.debug("trying to create TS "+request.getTitle());
		String id= uuidGen.nextUUID();
		CurationResource curationResource;
		TimeSeriesResource resource=null;
		try{
			curationResource=(CurationResource) CurationContext.getPortTypeContext().getWSHome().find(CurationContext.getPortTypeContext().makeKey(request.getRelatedCurationId()));
		}catch (Exception e){
			logger.error("the Curation with selected Id does not exist",e);
			throw new GCUBEUnrecoverableFault(e, "the Curation with selected Id does not exist");
		}
		for (ColumnDefinition columnDefinition: curationResource.getColumnDefinition())
			if (columnDefinition.getColumnType()==EntryType.Undefined){
				logger.error("the selected curation resource is not completly defined");
				throw new GCUBEUnrecoverableFault("the selected curation resource is not completly defined");
			}
		
		try{
			resource =(TimeSeriesResource)TimeSeriesContext.getPortTypeContext().getWSHome().create(TimeSeriesContext.getPortTypeContext().makeKey(id), id, request, Boolean.TRUE);
			resource.store();
			return new CreateTsResponse(resource.getEPR(), resource.getId());
		}catch(Exception e){
			if (resource!=null) resource.remove();
			logger.error("error creating TS" , e);
			throw new GCUBEUnrecoverableFault(e,"error creating TS");
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public CreateFromTsResponse createFromTs(CreateFromTsRequest request) throws GCUBEFault{
		logger.debug("trying to create TS "+request.getTitle() +" from "+request.getTsSourceId());
		String id= uuidGen.nextUUID();
		try{
			TimeSeriesResource resource =(TimeSeriesResource)TimeSeriesContext.getPortTypeContext().getWSHome().create(TimeSeriesContext.getPortTypeContext().makeKey(id), id, request, Boolean.FALSE);
			resource.store();
			return new CreateFromTsResponse(resource.getEPR(), resource.getId());
		}catch(Exception e){
			logger.error("error creating TS" , e);
			throw new GCUBEUnrecoverableFault(e,"error creating TS");
		}
	}
	
	/**
	 * 
	 * @param timeSeriesId
	 * @return
	 * @throws GCUBEFault
	 */
	public EndpointReferenceType open(OpenMsg request) throws GCUBEFault{
		logger.debug("opening time series with Id "+request.getTsId());
		GCUBEWSResourceKey timeSeriesKey= TimeSeriesContext.getPortTypeContext().makeKey(request.getTsId());
		EndpointReferenceType resourceEpr= null;
		TimeSeries ts = null;
		try{
			ts = ObjectPersistency.get(TimeSeries.class).getByKey(request.getTsId());
			if (ts==null) throw new Exception();
			
			if(!ServiceContext.getContext().getScope().isEnclosedIn(GCUBEScope.getScope(ts.getScope()))) throw new Exception();
			
			if (!ts.getUsers().contains(" "+request.getUser()+" ")){
				ts.setUsers(ts.getUsers()+", "+request.getUser()+" ");
				ts.store();
			}
		}catch (Exception e) {
			logger.error("the requested timeSeries has been not found or has been created in a different scope");
			throw new GCUBEUnrecoverableFault("the requested timeSeries has been not found or has been created in a different scope");
		}	
			
		try{
			resourceEpr=TimeSeriesContext.getPortTypeContext().getWSHome().find(timeSeriesKey).getEPR();
		}catch(Exception nsre){
			try{
				if (ts!=null)ts.delete();
			}catch (Exception e) {
				logger.warn("error removing ts from timeseries table");
			}
			logger.error("the requested timeSeries "+request.getTsId()+" doesn't exist ",nsre);	
			throw new GCUBEUnrecoverableFault("the requested timeSeries "+request.getTsId()+" doesn't exist");
		} 
		return  resourceEpr;
	}
	
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public TimeSeriesItemsArray getImportedTimeSeries(String user) throws GCUBEFault{
		TimeSeriesItemsArray response= new TimeSeriesItemsArray();
		try {
			/*ANDCondition cond = new ANDCondition(new ORCondition(new OperatorCondition<SimpleAttribute, String>(new SimpleAttribute("creator"),user,"="), 
					new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute("position(' "+user+" ' in users)"),0,"!=") ),
					new OperatorCondition<SimpleAttribute, Boolean>(new SimpleAttribute("published"),false,"="));*/
			Iterator<TimeSeries> it = ObjectPersistency.get(TimeSeries.class).getObjectByField("published", false).iterator();
			logger.trace("retrieved imported items");
			List<TimeSeriesItem> listTimeSeries= new ArrayList<TimeSeriesItem>();
			while (it.hasNext()){
				TimeSeries ts= it.next();
				if ((ts.getUsers().contains(" "+user+" ") || ts.getCreator().equals(user))
						&& (ServiceContext.getContext().getScope().isEnclosedIn(GCUBEScope.getScope(ts.getScope()))))
					listTimeSeries.add(createTSItem(ts));
			}
			response.setItems(listTimeSeries.toArray(new TimeSeriesItem[listTimeSeries.size()]));
		}catch(Exception e){
			logger.error("error retrieving imported Time Series",e);
			throw new GCUBEUnrecoverableFault(e,"error retrieving imported Time Series");
		}
		return response;
	}
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public TimeSeriesItemsArray getPublishedTimeSeries(VOID request) throws GCUBEFault{

		TimeSeriesItemsArray response= new TimeSeriesItemsArray();
		DBSession session=null;
		try {
			Iterator<TimeSeries> it = ObjectPersistency.get(TimeSeries.class).getObjectByField("published", true).iterator();
			List<TimeSeriesItem> listTimeSeries= new ArrayList<TimeSeriesItem>();
			while (it.hasNext()){
				TimeSeries ts = it.next();
				if(ServiceContext.getContext().getScope().isEnclosedIn(GCUBEScope.getScope(ts.getScope())))
					listTimeSeries.add(createTSItem(ts));
			}
			response.setItems(listTimeSeries.toArray(new TimeSeriesItem[listTimeSeries.size()]));
		}catch(Exception e){
			logger.error("error retrieving imported Time Series",e);
			throw new GCUBEUnrecoverableFault(e,"error retrieving imported Time Series");
		}finally{
			if (session!=null)
				session.release();
		}
		return response;
	}
			
	public TimeSeriesItem getTimeSeriesInfoById(String tsId) throws Exception{
		try {
			TimeSeries ts = ObjectPersistency.get(TimeSeries.class).getByKey(tsId);
			if (ts== null) return null;
			return createTSItem(ts);
		}catch(Exception e){
			logger.error("error retrieving imported Time Series",e);
			throw new GCUBEUnrecoverableFault(e,"error retrieving imported Time Series");
		}
	}
		
	/**
	 * 
	 * @param id
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID removeTimeSeries(String id) throws GCUBEFault{
		DBSession session=null;
		try {
			try{	
				ObjectPersistency.get(TimeSeries.class).deleteByKey(id);
				//new DeleteQuery(new OperatorCondition<String, String>("id",id,"="),new Table(Constants.TimeSeriesTable)).execute();
			}catch(Exception e){logger.error("error deleting entry in TimeSeries table, it's could be already deleted",e);}
			TimeSeriesContext.getPortTypeContext().getWSHome().remove(TimeSeriesContext.getPortTypeContext().makeKey(id));
		} catch (Exception e) {
			logger.error("error removing TimeSeries with id "+id,e);
			throw new GCUBEUnrecoverableFault(e);
		}finally{
			if (session!=null)
				session.release();
		}
		return new VOID();
	}
	
	
	public ServiceProperties getServiceProperties(VOID req) throws GCUBEFault{
		try {
			ServiceProperties properties = new ServiceProperties();
			properties.setServiceMode( ServiceMode.fromValue(ServiceContext.getContext().getTsMode().toString()));
			properties.setProps(new StringMap(ServiceContext.getContext().getProperties()));
			return properties; 
		}  catch (Exception e) {
			logger.error("error getting the serviceMode", e);
			throw new GCUBEFault(e);
		}
	}
	
	private TimeSeriesItem createTSItem(TimeSeries ts) throws Exception{
		TimeSeriesItem tempItem= new TimeSeriesItem();
		tempItem.setCreator(ts.getCreator());
		tempItem.setDate(Util.dateFormatter.format(new Date(ts.getDate().getTime())));
		tempItem.setDescription(ts.getDescription());
		tempItem.setId(ts.getId());
		tempItem.setIsVersionOf(ts.getIsVersionOf());
		tempItem.setLenght(ts.getLegth());
		tempItem.setPublished(ts.isPublished());
		tempItem.setPublisher(ts.getPublisher());
		tempItem.setRights(ts.getRights());
		tempItem.setSourceId(ts.getSourceId());
		tempItem.setSourceName(ts.getSourceName());
		tempItem.setTitle(ts.getTitle());
		tempItem.setType(ts.getType());
		return tempItem;
	}
}
