package org.gcube.data.access.accounting.summary.access.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.LinkedList;

import org.gcube.data.access.accounting.summary.access.AccountingDao;
import org.gcube.data.access.accounting.summary.access.ParameterException;
import org.gcube.data.access.accounting.summary.access.impl.DBStructure.DIMENSIONS;
import org.gcube.data.access.accounting.summary.access.impl.DBStructure.Measure;
import org.gcube.data.access.accounting.summary.access.model.MeasureResolution;
import org.gcube.data.access.accounting.summary.access.model.Record;
import org.gcube.data.access.accounting.summary.access.model.Report;
import org.gcube.data.access.accounting.summary.access.model.ReportElement;
import org.gcube.data.access.accounting.summary.access.model.ScopeDescriptor;
import org.gcube.data.access.accounting.summary.access.model.Series;
import org.gcube.data.access.accounting.summary.access.model.internal.Dimension;

import lombok.extern.slf4j.Slf4j;


@Slf4j
public class AccountingDaoImpl implements AccountingDao{


	private ContextTreeProvider treeProvider=null;
	private ConnectionManager connectionManager=null;

	public AccountingDaoImpl() {
		connectionManager=new BasicConnectionManager();
		treeProvider=new BasicContextTreeProvider();
	}

	public void setTreeProvider(ContextTreeProvider treeProvider) {
		this.treeProvider = treeProvider;
	}

	public AccountingDaoImpl(ContextTreeProvider treeProvider, ConnectionManager connectionManager) {
		super();
		this.treeProvider = treeProvider;
		this.connectionManager = connectionManager;
	}



	//	@Override
	//	public Report getReportByScope(
	//			@NonNull ScopeDescriptor scopeDescriptor, @NonNull Date from, @NonNull Date to, @NonNull MeasureResolution resolution) throws ParameterException {
	//		log.trace("Loading report for {} between {} and {} [{}]",scopeDescriptor,from,to,resolution);
	//		// Evaluate Range based on from/to && resolution - arrays in series will depend on that
	//		int range=getRangeSize(from, to, resolution);
	//		LinkedList<ReportElement> reportItems=new LinkedList<>();
	//		
	//		
	//		
	//		
	//		
	//		// Get connection
	//		Connection conn=connectionManager.getConnection();
	//		
	//		
	//		// Evaluate to produce Report Elements
	//		
	//		PreparedStatement reportsPS=conn.prepareStatement("Select * from dimensions where id in (Select distinct(dimension) from "+resolution.getTableName()+" where context_id in (?))");
	//		
	//		reportsPS.setString(scopeDescriptor., x);
	//		
	//		ResultSet rs=
	//		
	//		
	//		
	//		String.format(format, args)
	//		
	//		
	//		// Select * from monthly_measure inner join dimensions on monthly_measure.dimension=dimensions.id 
	//		// where year => ? AND year =< ? AND month => ? AND month =< ?
	//		// AND context_id= ?
	//				
	//		PreparedStatement measurePS=conn.prepareStatement("Select * from "+resolution.getTableName()+" ");
	//		
	//		
	//		
	//		
	//		
	//		
	//		
	//		// ScopeDescriptor.children.size +1 == number of series 
	//		
	//		// N° of ReportElement == N° of found measures
	//		
	//		return null;
	//	}
	//	


	@Override
	public Report getReportByScope(ScopeDescriptor desc, Date from, Date to, MeasureResolution resolution) throws SQLException, ParameterException {

		DateTimeFormatter formatter=getFormatter(resolution);
		LocalDateTime fromDate=from.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
		LocalDateTime toDate=to.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
		long startReportTime=System.currentTimeMillis();

		log.info("Loading report {} for {} between {} and {} ",resolution,desc.getId(),formatter.format(fromDate),formatter.format(toDate));

		Connection conn=connectionManager.getConnection();
		Queries queries=new Queries(conn);

		int timeSlices=getRangeSize(from, to, resolution);


		//load available dimensions in time slice
		ResultSet dimensionRS=queries.getAvailableDimensions(from, to, desc, resolution);
		LinkedList<Dimension> foundDimensions=new LinkedList<>();
		while(dimensionRS.next()){
			String id=dimensionRS.getString(DIMENSIONS.ID);
			String label=dimensionRS.getString(DIMENSIONS.LABEL);
			String group=dimensionRS.getString(DIMENSIONS.GROUP);
			String aggregatedDim=dimensionRS.getString(DIMENSIONS.AGGREGATED_MEASURE);

			foundDimensions.add(new Dimension(id,label,aggregatedDim,group));
		}
		log.debug("Found {} dimensions to load. ",foundDimensions.size());

		// Prepare reports for each Dimension
		LinkedList<ReportElement> reports=new LinkedList<>();
		for(Dimension entry: foundDimensions){
			String xLabel=entry.getLabel();
			String yLabel="time";
			String category=entry.getGroup();

			// Report 1 series for selected Scope
			reports.add(new ReportElement(desc.getName()+" "+xLabel,category,
					xLabel,yLabel,new Series[]{getSeries(queries, from, to, entry, desc, resolution, timeSlices)}));

			// Report 2 series for each children
			if(desc.hasChildren()) {
				LinkedList<Series> childrenSeries=new LinkedList<>();
				for(ScopeDescriptor child:desc.getChildren()){
					childrenSeries.add(getSeries(queries, from, to, entry, child, resolution, timeSlices));
				}
				reports.add(new ReportElement(desc.getName()+" children "+xLabel,category,
						xLabel,yLabel,childrenSeries.toArray(new Series[childrenSeries.size()])));
			}


			//			PreparedStatement psMeasure=queries.prepareMeasuresByDimension(desc, resolution);
			//			
			//			LocalDateTime toDate=to.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
			//			for(LocalDateTime toAsk=from.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime(); 
			//					toAsk.isAfter(toDate);toAsk=increment(toAsk,resolution)){
			//			// Scan for time slice	
			//			}
		}

		log.info("Loaded {} report elements in {} ms",reports.size(),(System.currentTimeMillis()-startReportTime));


		return new Report(reports);

	}


	private Series getSeries(Queries queries, Date from, Date to, Dimension dim, ScopeDescriptor scope, MeasureResolution res, int timeSlices) throws SQLException{

		Record[] records=new Record[timeSlices];
		PreparedStatement ps=queries.prepareMeasuresByDimension(scope, res);
		DateTimeFormatter formatter=getFormatter(res);


		LocalDateTime toSetStartDate=from.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
		for(int i=0;i<timeSlices;i++){
			toSetStartDate=increment(toSetStartDate,res,i); // Increment Date
			LocalDateTime toSetEndDate=increment(toSetStartDate,res,i+1);


			ps.setLong(1, toSetStartDate.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
			ps.setLong(2, toSetEndDate.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
			ps.setString(3, dim.getId());

			ResultSet rs=ps.executeQuery();
			Record toSet=new Record(formatter.format(toSetStartDate),0l);
			if(rs.next()){
				toSet.setY(rs.getLong(Measure.MEASURE));
			}
			records[i]=toSet;
		}

		return new Series(scope.getName(),records);

	}



	private static LocalDateTime increment(LocalDateTime toIncrement,MeasureResolution res,int offset){
		switch(res){
		case MONTHLY : return toIncrement.plusMonths(1);
		default : throw new RuntimeException("Unexpected Resolution "+res);
		}
	}

	private static final DateTimeFormatter monthFormatter=DateTimeFormatter.ofPattern("yyyy-MM");


	private static DateTimeFormatter getFormatter(MeasureResolution res){
		switch(res){
		case MONTHLY : return monthFormatter;
		default : throw new RuntimeException("Unexpected Resolution "+res);
		}
	}


	@Override
	public ScopeDescriptor getTree(Object request) throws Exception {
		return treeProvider.getTree(request);
	}

	private static final int getRangeSize(Date from, Date to, MeasureResolution resolution) throws ParameterException {
		log.debug("Evaluating time range between {} , {} [{}]",from,to,resolution);

		Period p=Period.between(from.toInstant().atZone(ZoneId.systemDefault()).toLocalDate(), to.toInstant().atZone(ZoneId.systemDefault()).toLocalDate());
		switch(resolution) {
		case MONTHLY : return p.getMonths()+(p.getYears()*12);
		default : throw new ParameterException("Invalid resolution "+resolution);
		}

	}





}
