package org.gcube.common.homelibrary.client;

import java.io.File;
import java.net.URL;
import java.nio.file.Files;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.common.homelibrary.client.servlet.Servlets;
import org.gcube.common.homelibrary.client.util.Config;

public class FTPClient {

	private static final String SEPARATOR = "/";
	private String rootPath;

	public FTPClient() {
		this.rootPath = Config.ROOT_PATH;
	}

	/**
	 * This method asks and returns the current working directory.
	 * @return path The path to the current working directory.
	 * @throws Exception If the operation fails.
	 */
	public String getRootPath() throws Exception {
		if ((rootPath == null) || (!rootPath.contains(Config.ROOT_PATH)))
			throw new Exception("current directory not set");
		return rootPath;
	}

	/**
	 * Check if a directory exists
	 * @param dir The path to the new working directory.
	 * @return true if the directory exists.
	 */
	private boolean exists(String dir, String absPath){
		try {
			Servlets.listFolder(absPath + File.separator + dir);
		} catch (Exception e) {			
			return false;
		}
		return true;
	}

	/**
	 * This method deletes a remote directory.
	 * @param path  The path to the directory.
	 * @throws Exception  If the operation fails.
	 */
	public void delete(String absPath) throws Exception {

		if (absPath == Config.ROOT_PATH)
			throw new Exception("Root cannot be deleted");
		if (!absPath.contains(Config.ROOT_PATH))
			throw new Exception(getRootPath() + " cannot be deleted");

		Boolean flag = Servlets.delete(absPath);
		if (!flag)
			throw new Exception("Impossible to delete " + absPath);
	}


	/**
	 * This method lists the entries of the current working directory
	 * @return The list of the files (and directories) in the current working directory.
	 * @throws Exception If the operation fails.
	 */
	public FTPFile[] list(String absPath) throws Exception {

		FTPFile[] list = null;
		try{
			Map<String, Boolean> files = Servlets.listFolder(absPath);
			System.out.println(files.size());
			list = new FTPFile[files.size()];

			int i = 0;
			Set<String> keys = files.keySet();

			for(String key: keys){
				Boolean isFolder = files.get(key);
				list[i] = new FTPFile(absPath + SEPARATOR + key, isFolder);
				i++;
			}
		}catch (Exception e){
			throw new Exception("Impossible to list " + absPath + ": " + e.getMessage());
		}
		return list;
	}

	/**
	 * This method creates a new remote directory in the current working one.
	 * @param directoryName The name of the new directory.
	 * @param description A description for the new directory.
	 * @throws Exception If the operation fails.
	 */
	public void createDirectory(String directoryName, String description, String absPath) throws Exception {

		try{
			if (exists(directoryName, absPath))
				throw new Exception(directoryName + " already exists");
			Servlets.createFolder(directoryName, description, absPath);
		}catch (Exception e){
			throw new Exception("Impossible to create Directory " + directoryName + ": " + e.getMessage());
		}
	}

	/**
	 * This method uploads a file to the remote server
	 * @param file The file to upload.
	 * @param name A name for the file.
	 * @param description A description for the file.
	 * @throws Exception If the operation fails.
	 */
	public void upload(String absoluteFilePath, String name, String description, String absPath, String mimetype, long size) throws Exception {

		File file = new File(absoluteFilePath);
		try{
			Servlets.uploadFile(file, name, description, absPath, "UTF-8", mimetype, size);
		}catch (Exception e){
			throw new Exception("Impossible to upload file " + file.getPath() + ": " + e.getMessage());
		}
	}


//	/**
//	 * This method uploads a file to the remote server
//	 * @param file The file to upload.
//	 * @throws Exception If the operation fails.
//	 */
//	public void upload(File file, String absPath) throws Exception {
//		upload(file, null, "", absPath);
//	}



	/**
	 * This method uploads a byte[] to the remote server
	 * @param bFile in byte[]
	 * @param name The name of the new file
	 * @param description The description of the new file
	 * @param absPath absolute path where the file will be uploaded
	 * @throws Exception If the operation fails.
	 */
	public void upload(byte[] bFile, String name, String description, String absPath) throws Exception {
		try{
			Servlets.uploadFile(bFile, name, description, absPath, null, 0);
		}catch (Exception e){
			throw new Exception("Impossible to upload file " + name + ": " + e.getMessage());
		}
	}

	/**
	 * This method uploads a byte[] to the remote server with mimetype and size
	 * @param bFile in byte[]
	 * @param name The name of the new file
	 * @param description The description of the new file
	 * @param absPath absolute path where the file will be uploaded
	 * @param mimetype The mimetype of the file
	 * @param size The size of the file
	 * @throws Exception If the operation fails.
	 */
	public void upload(byte[] bFile, String name, String description, String absPath, String mimetype, long size) throws Exception {
		try{
			Servlets.uploadFile(bFile, name, description, absPath, mimetype, size);
		}catch (Exception e){
			throw new Exception("Impossible to upload file " + name + ": " + e.getMessage());
		}
	}


	/**
	 * Run tests
	 * @throws Exception
	 */
	public void test() throws Exception {
		FTPClient client = new FTPClient();
		System.out.println("Current dir " + client.getRootPath());
		System.out.println("List Root ");
		FTPFile[] list = client.list(client.getRootPath());
		for (FTPFile item:list){
			System.out.println("* " + item.getName() + " is folder? " + item.isFolder());
			//			if (item.isFolder())
			//				client.deleteDirectory(item.getPath());
		}

		System.out.println("\n\n");

		String A_name = "A-"+ UUID.randomUUID().toString();
		String B_name = "B-"+ UUID.randomUUID().toString();

		client.createDirectory(A_name, "test A in ROOT", client.getRootPath());
		System.out.println("Folder "+ A_name + " created in "+ client.getRootPath());


		System.out.println("\n\n");


		client.createDirectory(B_name, "test A in ROOT", Config.ROOT_PATH + "/" + A_name);
		System.out.println("Folder "+ B_name + " created in "+ client.getRootPath());

		System.out.println("\n\n");

		System.out.println("Save images in " + Config.ROOT_PATH + "/" + A_name + "/" + B_name);

		URL imageURL = FTPClient.class.getClassLoader().getResource(Config.DEFAULT_IMAGE);
		File file = new File(imageURL.getFile());

		for (int i= 0; i< 5; i++){
			byte[] image = Files.readAllBytes(file.toPath());
			client.upload(image, "img-"+ UUID.randomUUID().toString()+".jpg", "my description", Config.ROOT_PATH + "/" + A_name + "/" + B_name, "image/jpeg", 3000);
		}
		System.out.println("\n\n");


		System.out.println("\n\n");
		System.out.println("HL CLI test successfully completed. ");
	}

}
