/**
 * 
 */
package org.gcube.data.tml.clients;

import static org.gcube.data.tml.clients.APIUtils.*;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.clients.DiscoveryException;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScopeManager;
import org.gcube.data.tml.clients.providers.TReaderProvider;
import org.gcube.data.tml.clients.providers.TWriterProvider;
import org.gcube.data.tml.clients.queries.QueryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Administrative operations over remote collection managers.
 * 
 * @author Fabio Simeoni
 *
 */
public class AdminAPI {

	private static Logger log = LoggerFactory.getLogger(AdminAPI.class);
	
	private static <T> Future<T> execute(Callable<T> call) {
		ExecutorService executor = Executors.newSingleThreadExecutor();
		Future<T> outcome = executor.submit(call);
		executor.shutdown();
		return outcome;
	}
	
	private static <T> List<Future<T>> execute(List<Callable<T>> calls) {
		//launch tasks
		List<Future<T>> output = new ArrayList<Future<T>>();
		
		if (calls.size()>0) {
			
			//setup executor service
			ExecutorService executor = Executors.newFixedThreadPool(calls.size());

			//submit tasks and collect future outcomes
			for (final Callable<T> call : calls)
				output.add(executor.submit(call));
			
			//terminates the executor when all tasks are finished
			executor.shutdown();
		}
		
		return output;
	}
	
	public static Future<?> deleteReader(final EndpointReferenceType endpoint) {
		
		final GCUBEScope scope = GCUBEScopeManager.DEFAULT.getScope();
		return execute(new Callable<Void>() {
					public Void call() throws Exception {
						
						GCUBEScopeManager.DEFAULT.setScope(scope);
						
						log.debug("deleting reader @ {}",endpoint);
						
						new ReaderClient(endpoint).delete();
						
						return null;
					}}
				);
	}
	
	public static Future<?> deleteWriter(final EndpointReferenceType endpoint) throws IllegalArgumentException {
		
		final GCUBEScope scope = GCUBEScopeManager.DEFAULT.getScope();
		
		return execute(new Callable<Void>() {
			public Void call() throws Exception {
				
				GCUBEScopeManager.DEFAULT.setScope(scope);
				
				log.debug("deleting reader @ {}",endpoint);
				
				new WriterClient(endpoint).delete();
				
				return null;
			}}
		);
	}
	
	/**
	 * Deletes readers and writer services for a given source at a given address.
	 * @param sourceId the identifier of the source
	 * @param address the address
	 * @return a two-element list of {@link Future} outcomes of the operation (ReaderResource and WriterResource)
	 * @throws Exception if the operation fails due to a local error
	 */
	public static List<Future<?>> deleteServices(String sourceId, URL address) throws Exception {

		log.info("deleting all reader and writer services for source {} @ {}",sourceId, address);
		
		List<Future<?>> outcome = new ArrayList<Future<?>>();
	
		EndpointReferenceType endpoint = endpoint(address,TReaderProvider.INSTANCE.name(),sourceId);
		outcome.add(deleteReader(endpoint));

		endpoint = endpoint(address,TWriterProvider.INSTANCE.name(),sourceId);
		outcome.add(deleteWriter(endpoint));
		
		return outcome;
	}
	
	public static List<Future<EndpointReferenceType>> deleteReaders(String sourceId) throws DiscoveryException {
		
		log.info("deleting all readers for source {}",sourceId);
		
		//discovery
		List<EndpointReferenceType> endpoints = QueryBuilder.findReadSource().withId(sourceId).build().fire();
		
		//launch tasks
		List<Callable<EndpointReferenceType>> calls = new ArrayList<Callable<EndpointReferenceType>>();
		for (final EndpointReferenceType endpoint : endpoints)
			calls.add(new Callable<EndpointReferenceType>() {
						public EndpointReferenceType call() throws Exception {
							deleteReader(endpoint);
							return endpoint;
						}
					});
		
		return execute(calls);
	}
	
	public static List<Future<EndpointReferenceType>> deleteWriters(String sourceId) throws DiscoveryException {
		
		log.info("deleting all writers for source {}",sourceId);
		
		//discovery
		List<EndpointReferenceType> endpoints = QueryBuilder.findWriteSource().withId(sourceId).build().fire();
		
		//launch tasks
		List<Callable<EndpointReferenceType>> calls = new ArrayList<Callable<EndpointReferenceType>>();
		
		final GCUBEScope scope = GCUBEScopeManager.DEFAULT.getScope();
		
		for (final EndpointReferenceType endpoint : endpoints)
			calls.add(new Callable<EndpointReferenceType>() {
						public EndpointReferenceType call() throws Exception {
							
							GCUBEScopeManager.DEFAULT.setScope(scope);
							
							deleteReader(endpoint);
							
							return endpoint;
						}
					});
		
		return execute(calls);
	}
	
	
	/**
	 * Deletes all reader and writer services for a given data source.
	 * @param sourceId the identifier of the source
	 * @return a list with the {@link Future} outcomes of the operation, one for each service
	 * @throws DiscoveryException if an error occurred during manager discovery
	 * @throws Exception if the operation fails due to a local error
	 */
	static List<Future<EndpointReferenceType>> deleteServices(String sourceId) throws DiscoveryException,Exception {
		
		log.info("deleting all readers and writers for source {}",sourceId);
		
		List<Future<EndpointReferenceType>> outcome = deleteReaders(sourceId);
		outcome.addAll(deleteWriters(sourceId));
		return outcome;
	}


}