package org.gcube.data.tm.state;

import org.gcube.common.core.faults.GCUBEException;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.data.streams.Stream;
import org.gcube.data.tmf.api.SourceWriter;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.tml.outcomes.AddTreeOutcome;
import org.gcube.data.tml.outcomes.UpdateTreeFailure;
import org.gcube.data.trees.data.Tree;
import org.globus.wsrf.ResourceException;

/**
 * A {@link AccessorResource} for write-only operations over a
 * {@link SourceResource}.
 * 
 * @author Fabio Simeoni
 */
public final class TWriterResource extends AccessorResource {

	// little helper
	private SourceWriter writer() throws GCUBEException {
		try {
			return getLocalResource().source().writer();
		} catch (ResourceException e) {
			throw new GCUBEUnrecoverableException(e);
		}
	}

	/**
	 * Adds a {@link Tree} to the source.
	 * 
	 * @param tree
	 *            the tree
	 * @return the input tree, possibly annotated with metadata added at the
	 *         time of addition.
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws InvalidTreeException
	 *             if the tree is invalid for addition
	 * @throws Exception
	 *             if the operation fails for any other problem
	 */
	public Tree add(Tree tree) throws UnsupportedOperationException,
			UnsupportedRequestException, InvalidTreeException, Exception {
		return writer().add(tree);
	}

	/**
	 * Adds many {@link Tree}s to the source and returns the corresponding
	 * {@link AddTreeOutcome}s.
	 * 
	 * @param treeStream
	 *            the trees
	 * @return the outcomes
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws GCUBEException
	 *             if the outcomes could not be returned
	 */
	public Stream<AddTreeOutcome> add(Stream<Tree> treeStream)
			throws UnsupportedOperationException, UnsupportedRequestException,
			Exception {

		return writer().add(treeStream);

	}

	/**
	 * Updates a tree in the source from the delta tree that captures the
	 * changes.
	 * 
	 * @param delta
	 *            the tree delta.
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws UnknownTreeException
	 *             if the delta tree does not identify a tree in the source
	 * @throws InvalidTreeException
	 *             if the delta tree does now qualify for update
	 * @throws Exception
	 *             if the operation fails for any other problem
	 */
	public void update(Tree delta) throws UnsupportedOperationException,
			UnsupportedRequestException, UnknownTreeException,
			InvalidTreeException, Exception {
		writer().update(delta);
	}

	/**
	 * Updates many {@link Tree}s in the source from the delta trees that
	 * capture their changes. It returns the failures that may occur in the
	 * process.
	 * 
	 * @param deltaStream
	 *            the delta trees
	 * @return the failures
	 * @throws UnsupportedOperationException
	 *             if the target plugin does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the target plugin does not support this particular request
	 * @throws GCUBEException
	 *             if the locator of a result set of update failures cannot be
	 *             returned.
	 */
	public Stream<UpdateTreeFailure> update(Stream<Tree> deltaStream)
			throws UnsupportedOperationException, UnsupportedRequestException,
			Exception {
		return writer().update(deltaStream);
	}

}
