package org.gcube.data.analysis.sdmx.datasource.tabman.querymanager.impl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.gcube.data.analysis.sdmx.datasource.data.BasicQuery;
import org.gcube.data.analysis.sdmx.datasource.tabman.querymanager.exceptions.InvalidInformationSystemDataException;
import org.gcube.data.analysis.tabulardata.commons.webservice.exception.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.commons.webservice.exception.NoSuchTabularResourceException;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.Equals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.GreaterOrEquals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.LessOrEquals;
import org.gcube.data.analysis.tabulardata.expression.logical.And;
import org.gcube.data.analysis.tabulardata.expression.logical.Or;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDInteger;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDText;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDTypeValue;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.query.parameters.QueryFilter;
import org.gcube.data.analysis.tabulardata.query.parameters.select.QueryColumn;
import org.gcube.data.analysis.tabulardata.query.parameters.select.QuerySelect;
import org.gcube.data.analysis.tabulardata.service.TabularDataService;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResourceId;
import org.gcube.datapublishing.sdmx.DataInformationProvider;
import org.gcube.datapublishing.sdmx.model.TableIdentificators;
import org.sdmxsource.sdmx.api.model.format.DataQueryFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TabmanQuery extends BasicQuery implements DataQueryFormat<TabmanQuery> 
{
	private Logger logger;
	private QueryFilter filter;
	private QuerySelect querySelectFilter;
	private TableId tableId;
	
	public TabmanQuery() {
		super ();
		this.logger = LoggerFactory.getLogger(this.getClass());
		this.filter = null;
	}


	public void initQuery (TabularDataService service, boolean lastTable) throws NoSuchTabularResourceException, NoSuchTableException, InvalidInformationSystemDataException 
	{
		this.logger.debug("Loading table identificators from the Information System");
		TableIdentificators tableIdentificators = getTableIdentificators(this.dataFlowAgency, this.dataFlowId, this.dataFlowVersion);
		
		if (tableIdentificators == null)
		{
			this.logger.error("Tabular resource not found for the data flow");
			throw new InvalidInformationSystemDataException("Tabular resource not found for the data flow");

		}
		else
		{
			Table table = null;
			
			
			if (lastTable)
			{
				table = service.getLastTable(new TabularResourceId(tableIdentificators.getTabularResourceID()));
				
				if (table == null)
				{
					this.logger.error("Tablular resource not found "+tableIdentificators.getTabularResourceIDString());
					throw new NoSuchTabularResourceException(tableIdentificators.getTabularResourceID());
				}
			}
			else
			{
				table = service.getTable(new TableId(tableIdentificators.getTableID()));
				
				if (table == null)
				{
					this.logger.error("Table not found "+tableIdentificators.getTableIDString());
					throw new NoSuchTableException(new TableId(tableIdentificators.getTableID()));
				}
			}
			
			this.tableId = table.getId();
			this.logger.debug("Table  found: id "+this.tableId.getValue());

			List<QueryColumn> queryColumns = getColumns(table.getColumns());
			
			if (queryColumns.size() > 0) 
			{
				logger.debug("Generating query select filter");
				this.querySelectFilter = new QuerySelect(queryColumns);
			}
			else logger.debug("No query select filter generated");
			this.filter = buildQueryFilter();
		}

		
		logger.debug("Query generated");
	}
	

	private TableIdentificators getTableIdentificators (String dataFlowAgency, String dataFlowId, String dataFlowVersion)
	{
		this.logger.debug("Getting table identificators for agency "+dataFlowAgency+ " data flow "+dataFlowId+ " version "+dataFlowVersion);
		String dataFlowKey = DataInformationProvider.getDataFlowKey(dataFlowAgency, dataFlowId, dataFlowVersion);
		this.logger.debug("Data flow key "+dataFlowKey);
		TableIdentificators identificators = DataInformationProvider.getInstance().getTableId(dataFlowKey);
		this.logger.debug("Operation completed with result "+identificators);
		return identificators;
	}
	
	private List<QueryColumn> getColumns (List<Column> tableColumns)
	{
		logger.debug("Generating the list of requested columns");
		List<QueryColumn> response = new ArrayList<>();
		List<String> columnIds = getColumnIds();
		
		
		for (Column column : tableColumns)
		{
			ColumnLocalId localId = column.getLocalId();
			String localIdString = localId.getValue();
			logger.debug("Column local id "+localIdString);
			
			if (columnIds.remove(localIdString))
			{
				logger.debug("Element found");
				response.add(new QueryColumn (localId));
			}
			else logger.warn("Element "+localIdString+" not found on the registry: extra column?");
			
		}
		
		if (!columnIds.isEmpty()) this.logger.warn("At least one column on the registry has not a corresponding data column!");
		
		return response;
		
	}
	
	private Expression buildSingleColumnExpression (String columnId, Set<String> values)
	{
		if (values.size() == 1)
		{
			this.logger.debug("Generating single value expression");
			return generateEqualExpression(columnId, values.iterator().next());
		}
		else if (values.size()>1)
		{
			this.logger.debug("Generating or expression");
			List<Expression> equalsList = new LinkedList<>();
			Iterator<String> valuesIterator = values.iterator();
			
			while (valuesIterator.hasNext())
			{
				equalsList.add(generateEqualExpression(columnId, valuesIterator.next()));
			}
			return new Or (equalsList);
		}
		else return null;
		
	}
	
	private QueryFilter buildQueryFilter ()
	{
		logger.debug("Generating query filter");
		QueryFilter response = null;
		List<Expression> andExpressionsList = new ArrayList<>();
		
		if (!this.parametersMap.isEmpty())
		{
			Iterator<String> columnIdsIterator = this.parametersMap.keySet().iterator();
			
			while (columnIdsIterator.hasNext())
			{
				String columnId = columnIdsIterator.next();
				Set<String> values = this.parametersMap.get(columnId);
				andExpressionsList.add(buildSingleColumnExpression(columnId, values));
			}
		}
		
		if (timeIntervalMin != -1) andExpressionsList.add(generateNotEqualExpression(this.timeDimension.getId(), this.timeIntervalMin, true));
		
		if (timeIntervalMax != -1) andExpressionsList.add(generateNotEqualExpression(this.timeDimension.getId(),this.timeIntervalMax, false));
	
		if (andExpressionsList.size()==1) response = new QueryFilter(andExpressionsList.get(0));
		else if (andExpressionsList.size() >1) response= new QueryFilter(new And(andExpressionsList));
		
		return response;
	
		
	}
	
	private Expression generateEqualExpression (String columnId, String queryParameter)
	{
		logger.debug("Generating expression for "+columnId+ " with parameter "+queryParameter);
		ColumnReference leftArgument = new ColumnReference(this.tableId, new ColumnLocalId(columnId));
		TDTypeValue rightArgument = new TDText((String) queryParameter);
		return new Equals(leftArgument, rightArgument);
	}
	
	private Expression generateNotEqualExpression (String columnId, int queryParameter, boolean greaterOrEqual)
	{
		logger.debug("Generating expression for "+columnId+ " with parameter "+queryParameter+" greater or equal flag "+greaterOrEqual);
		ColumnReference leftArgument = new ColumnReference(this.tableId, new ColumnLocalId(columnId));
		TDTypeValue rightArgument = new TDInteger(queryParameter);
		return greaterOrEqual ? new GreaterOrEquals(leftArgument, rightArgument): new LessOrEquals (leftArgument, rightArgument); 
	}
	

	
//	private TDTypeValue generateValue (Object queryParameter)
//	{
//		TDTypeValue response = null;
//		
//		if (queryParameter instanceof String)
//		{
//			logger.debug("Query parameter string format");
//			response = new TDText((String) queryParameter);
//		}
//		else if (queryParameter instanceof Integer)
//		{
//			logger.debug("Query parameter integer format");
//			response = new TDInteger((Integer) queryParameter);
//		}
//		else if (queryParameter instanceof Number)
//		{
//			logger.debug("Query parameter integer format");
//			response = new TDNumeric((Double) queryParameter);
//		}
//		else if (queryParameter instanceof Date)
//		{
//			logger.debug("Query parameter integer format");
//			response = new TDDate((Date) queryParameter);
//		}
//		
//		return response;
//	}



	public QueryFilter getFilter() {
		return filter;
	}



	public QuerySelect getQuerySelectFilter() {
		return querySelectFilter;
	}



	
	public TableId getTableId ()
	{
		return this.tableId;
	}
	
}
