package org.gcube.data.analysis.statisticalmanager.persistence;

import static org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager.closeSession;
import static org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager.getSessionFactory;
import static org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager.roolbackTransaction;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.analysis.statisticalmanager.SMOperationStatus;
import org.gcube.data.analysis.statisticalmanager.SMOperationType;
import org.gcube.data.analysis.statisticalmanager.SMResourceType;
import org.gcube.data.analysis.statisticalmanager.stubs.SMAlgorithm;
import org.gcube.data.analysis.statisticalmanager.stubs.SMComputationRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMComputations;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreateTableRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMImporters;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMAbstractResource;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMComputation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMEntries;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMEntry;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMImport;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMOperation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMResource;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMSystemImport;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTableMetadata;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.transform.Transformers;

public class SMPersistenceManager {

	private  static GCUBELog logger = new GCUBELog(SMPersistenceManager.class);

	public static long addImporter(final SMCreateTableRequest request) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			
			SMImport smimport = new SMImport();
			smimport.setFileName(request.getTableName());
			smimport.setObjectType(request.getTableType());
			smimport.setOperationType(SMOperationType.IMPORTED.ordinal());
			smimport.setChecked(false);
			smimport.setPortalLogin(request.getUser());
			smimport.setSubmissionDate(Calendar.getInstance());
			smimport.setDescription(request.getDescription());
			smimport.setOperationStatus(SMOperationStatus.RUNNING.ordinal());
			session.save(smimport);
			t.commit();
			return smimport.getOperationId();

		} catch (Exception e) {
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}
	
	public static long addSystemImporter(String description, SMResource resource) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {

			SMSystemImport smimport = new SMSystemImport();
			smimport.setOperationType(SMOperationType.SYSTEM.ordinal());
			smimport.setSubmissionDate(Calendar.getInstance());
			smimport.setDescription(description);
			session.save(resource);
			
			SMAbstractResource ar = new SMAbstractResource();
			ar.setResource(resource);
			ar.setAbstractResourceId(resource.getResourceId());
			session.save(ar);

			smimport.setAbstractResource(ar);
			smimport.setOperationStatus(SMOperationStatus.COMPLETED.ordinal());
			smimport.setCompletedDate(Calendar.getInstance());
			session.saveOrUpdate(smimport);
			
			t.commit();
			return smimport.getOperationId();
		} catch (Exception e) {
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}
	 
	public static List<SMTableMetadata> getTables(String user, String template) {

		Session session = HibernateManager.getSessionFactory().openSession();
		List<SMTableMetadata> tables = new ArrayList<SMTableMetadata>();
		try {
			Query query = session
			.createQuery("select table.resourceId as tableId, "
					+ "table.name as name, "
					+ "table.description as description, "
					+ "table.template as template, "
					+ "o.completedDate as creationDate, "
					+ "o.operationType as provenance, "
					+ "o.algorithm as algorithm "
					+ "from SMOperation o " 
					+ "join o.abstractResource.resource, SMTable table "
					+ "where o.abstractResource.abstractResourceId = table.resourceId "
					+ "and (o.portalLogin like :name or o.portalLogin = null) "
					+ "and table.template like :template");

			query.setParameter("name",
					(user != null) ? user : "%");
			query.setParameter("template",
					((template != null) && !template.equals(TableTemplates.GENERIC)) ? template : "%");

			query.setResultTransformer(Transformers.aliasToBean(SMTableMetadata.class));
			tables.addAll(query.list());
			
			return tables;

		} finally {
			session.close();
		}

	}

	public static SMOperation getOperation(long operationId) throws Exception{

		Session session = HibernateManager.getSessionFactory().openSession();
		try {

			return (SMOperation)session.get(SMOperation.class, operationId);
		} finally {
			closeSession(session);
		}
	}

	public static SMComputations getComputations(final String user,
			final String algorithm, final String category) throws Exception{

		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			Query query = session
			.createQuery("select computation from SMComputation  computation "
					+ "where computation.portalLogin like :name and "
					+ "computation.algorithm like :algorithm and "
					+ "computation.category like :category");

			query.setParameter("name", (user !=null)?user:"%");
			query.setParameter("algorithm", (algorithm !=null)?algorithm:"%");
			query.setParameter("category", (category !=null)?category:"%");

			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();

			for(Object object : objects) {
				SMComputation computation = (SMComputation)object;

				Query queryParameters = session.createQuery(
						"select parameter from SMEntry parameter " +
				"where parameter.computationId = :computationId");
				queryParameters.setParameter("computationId", computation.getOperationId());

				@SuppressWarnings("unchecked")
				List<Object> parameters = queryParameters.list();
				if (!parameters.isEmpty()) {
					computation.setParameters(parameters.toArray(new SMEntry[parameters.size()]));
				}		
			}

			SMComputation[] computations = objects.toArray(new SMComputation[objects.size()]);
			return new SMComputations(computations);

		} finally {
			closeSession(session);
		}
	}

	public static long addComputation(final SMComputationRequest request) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {

			SMComputation smcomputation = new SMComputation();
			smcomputation.setOperationType(SMOperationType.COMPUTED.ordinal());
			smcomputation.setPortalLogin(request.getUser());
			smcomputation.setSubmissionDate(Calendar.getInstance());
			smcomputation.setTitle(request.getTitle());
			smcomputation.setDescription(request.getDescription());

			SMAlgorithm algorithm = request.getConfig().getAlgorithm();
			smcomputation.setAlgorithm(algorithm.getName());
			smcomputation.setCategory(algorithm.getCategory());

			smcomputation.setOperationStatus(SMOperationStatus.PENDING.ordinal());
			session.save(smcomputation);

			SMEntries parameters = request.getConfig().getParameters();
			for(SMEntry parameter : parameters.getList()) {
				parameter.setComputationId(smcomputation.getOperationId());
				session.save(parameter);
			}

			t.commit();
			return smcomputation.getOperationId();

		} catch (Exception e) {
			roolbackTransaction(t);
			e.printStackTrace();
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}

	public static void addCreatedResource(
			final long operationId, SMResource resource) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {

			SMOperation operation = (SMOperation)session.get(SMOperation.class, operationId);
			session.save(resource);

			SMAbstractResource ar = new SMAbstractResource();
			ar.setResource(resource);
			ar.setAbstractResourceId(resource.getResourceId());
			session.save(ar);

			operation.setAbstractResource(ar);
			operation.setOperationStatus(SMOperationStatus.COMPLETED.ordinal());
			operation.setCompletedDate(Calendar.getInstance());
			session.saveOrUpdate(operation);
			t.commit();
		} catch (Exception e) {
			logger.debug("Error persistence ", e);
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}

	public static void setOperationStatus(long operationId, SMOperationStatus status) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMOperation smoperation = (SMOperation)session.get(SMOperation.class, operationId);
			smoperation.setOperationStatus(status.ordinal());

			if (status == SMOperationStatus.FAILED)
				smoperation.setCompletedDate(Calendar.getInstance());	

			session.saveOrUpdate(smoperation);
			t.commit();		
		} catch (Exception e) {
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static void setComputationalInfrastructure(long operationId, INFRASTRUCTURE infra) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMComputation smcomputation = (SMComputation)session.get(SMComputation.class, operationId);
			smcomputation.setInfrastructure(infra.toString());
			session.save(smcomputation);
			t.commit();		
		} catch (Exception e) {
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static void removeOperation(long operationId) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMOperation smoperation = (SMOperation)session.get(SMOperation.class, operationId);

			SMAbstractResource abstractResource = smoperation.getAbstractResource();
			if (abstractResource != null) {
				SMResource resource = abstractResource.getResource();
				if (SMResourceType.values()[resource.getResourceType()] == SMResourceType.TABULAR) {
					try {
						DataBaseManager.removeTable(resource.getResourceId());
					} catch (Exception e) {
						logger.error("Table no deleted", e);
					}
				}
			}

			session.delete(smoperation);
			t.commit();		
		} catch (Exception e) {
			logger.error("Computation not removed ",e);
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static void removeTable(String tableId) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			Query query = session
			.createQuery("select o  from SMOperation  o "
					+ "join o.abstractResource.resource, SMTable table "
					+ "where o.abstractResource.abstractResourceId = table.resourceId "
					+ "and table.resourceId = :tableId");

			query.setParameter("tableId", tableId);
			

			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();
			SMOperation operation = (SMOperation)objects.get(0);
			
			SMAbstractResource abstractResource = operation.getAbstractResource();
			if (abstractResource != null) {
				SMResource resource = abstractResource.getResource();
				if (SMResourceType.values()[resource.getResourceType()] == SMResourceType.TABULAR) {
					try {
						DataBaseManager.removeTable(resource.getResourceId());
					} catch (Exception e) {
						logger.error("Table no deleted", e);
					}
				}
			}
			
			session.delete(operation);
			t.commit();		
		} catch (Exception e) {
			logger.error("Computation not removed ",e);
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static void checkImport(long importId) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMImport smimport = (SMImport)session.get(SMOperation.class, importId);
			smimport.setChecked(true);
			session.saveOrUpdate(smimport);
			t.commit();		
		} catch (Exception e) {
			logger.error("Computation not removed ",e);
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static SMImporters getUncheckedImports(String user) {

		Session session = getSessionFactory().openSession();

		try {
			Query query = session
			.createQuery("select importer from SMImport  importer "
					+ "where importer.checked = false "
					+ "and importer.portalLogin like :name ");

			query.setParameter("name",
					(user != null) ? user : "%");

			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();

			SMImport[] importers = objects
			.toArray(new SMImport[objects.size()]);
			return new SMImporters(importers);	

		} finally {
			closeSession(session);
		}
	}

}
