package org.gcube.data.analysis.tabulardata.cube.metadata;

import java.util.List;

import javax.enterprise.inject.Default;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;

import org.gcube.data.analysis.tabulardata.cube.metadata.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.metadata.model.JPATable;
import org.gcube.data.analysis.tabulardata.cube.metadata.model.JPATableFactory;
import org.gcube.data.analysis.tabulardata.cube.metadata.model.TableFactory;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

@Default
@Singleton
public class JPACubeMetadataWrangler implements CubeMetadataWrangler {

	Logger log = LoggerFactory.getLogger(JPACubeMetadataWrangler.class);

	EntityManager em = null;

	ISEntityManagerProvider emp;

	@Inject
	public JPACubeMetadataWrangler(ISEntityManagerProvider emp) {
		super();
		this.emp = emp;
	}

	// public Table update(Table table){
	// TableConsistencyChecker.checkTableConsistency(table);
	// JPATable jpaTable = JPATableFactory.createJPATable(table);
	// mergeEntity(jpaTable);
	// return TableFactory.createTable(jpaTable);
	// }

	@Override
	public Table save(Table table, boolean overwrite) {
		TableConsistencyChecker.checkTableConsistency(table);
		JPATable jpaTable = null;

		if (overwrite)
			try {
				jpaTable = JPATableFactory.updateJPATable(getJPATableById(table.getId().getValue()), table);
			} catch (NoSuchTableException e) {
				log.warn("the table with id {} is not persisted, cannot overwrite ",table.getId().getValue());
			}
		 
		if (jpaTable==null)
			jpaTable = JPATableFactory.createJPATable(table);
		
		persistEntity(jpaTable);
		return TableFactory.createTable(jpaTable);
	}

	private void initializeIfNot() {
		if (em == null)
			em = emp.get();
	}

	@Override
	public Table get(TableId id) throws NoSuchTableException {
		return TableFactory.createTable(getJPATableById(id.getValue()));
	}

	@Override
	public List<Table> getAll() {
		return Lists.transform(getAllJPATables(), new Function<JPATable, Table>() {

			@Override
			public Table apply(JPATable input) {
				return TableFactory.createTable(input);
			}

		});
	}

	@Override
	public List<Table> getAll(TableType tableType) {
		return Lists.transform(getAllJPATablesByType(tableType), new Function<JPATable, Table>() {

			@Override
			public Table apply(JPATable input) {
				return TableFactory.createTable(input);
			}

		});
	}

	@Override
	public void remove(TableId id) throws NoSuchTableException {
		removeEntity(getJPATableById(id.getValue()));
	}

	private JPATable getJPATableById(long id) throws NoSuchTableException {
		JPATable table = getEntityManager().find(JPATable.class, id);
		if (table==null)throw new NoSuchTableException(id);
		return table;
	}

	private List<JPATable> getAllJPATables() {
		TypedQuery<JPATable> query = getEntityManager().createNamedQuery("Table.findAll", JPATable.class);
		return query.getResultList();
	}

	private List<JPATable> getAllJPATablesByType(TableType tableType) {
		TypedQuery<JPATable> query = getEntityManager().createNamedQuery("Table.findAllByType", JPATable.class);
		query.setParameter("TableType", tableType);
		return query.getResultList();
	}

	private void persistEntity(JPATable entity) {
		try{
			getEntityManager().getTransaction().begin();
						
			if (!em.contains(entity))
				getEntityManager().persist(entity);
			else getEntityManager().merge(entity);
			getEntityManager().flush();
			getEntityManager().getTransaction().commit();
		}catch(Throwable t){
			log.warn("DB error",t);
			em.getTransaction().rollback();
		}
		//log.debug("Saved entity: " + entity);
	}

	// private void mergeEntity(Object entity){
	// log.debug("Updating entity: " + entity);
	// getEntityManager().getTransaction().begin();
	// getEntityManager().merge(entity);
	// getEntityManager().getTransaction().commit();
	// log.debug("Updated entity: " + entity);
	// }

	private void removeEntity(Object entity) {
		getEntityManager().getTransaction().begin();
		getEntityManager().remove(entity);
		getEntityManager().flush();
		getEntityManager().getTransaction().commit();
	}

	private EntityManager getEntityManager() {
		initializeIfNot();
		return em;
	}

}
