package org.gcube.data.analysis.tabulardata.cube.tablemanagers;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.IdColumn;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnHasLabel;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnHasName;
import org.gcube.data.analysis.tabulardata.model.metadata.CubeMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;

public abstract class DefaultTableCreator implements TableCreator {

	protected DatabaseWrangler dbWrangler;
	protected CubeMetadataWrangler mdWrangler;
	protected TableManager tableManager;

	public DefaultTableCreator(DatabaseWrangler dbWrangler, CubeMetadataWrangler mdWrangler, TableManager tableManager) {
		super();
		this.dbWrangler = dbWrangler;
		this.mdWrangler = mdWrangler;
		this.tableManager = tableManager;
	}
	
	@Override
	public TableCreator addColumn(Column column) {
		return addColumns(column);
	}

	@Override
	public TableCreator addColumns(Column... columns) {
		checkColumnsToAdd(columns);
		for (Column column : columns) {
			addNewColumn(column);
		}
		return this;
	}
	
	private void checkColumnsToAdd(Column... columns){
		for (Column column : columns)
			if (!isAllowedColumn(column))
				throw new IllegalArgumentException("Invalid column type: " + column.getColumnType());
	}

	@Override
	public TableCreator like(Table table, boolean withData) {
		return like(table, withData, new ArrayList<Column>());
	}

	@Override
	public TableCreator like(Table table, boolean withData, Collection<Column> columnsToRemove) {
		//Check if provided table is allowed
		if (!isAllowedCloneableTable(table))
			throw new IllegalArgumentException("Provided table to clone is not of the right type: "
					+ table.getTableType());
	
		//Check if a table to clone was already provided
		if (getTableToClone() != null)
			throw new IllegalStateException("A table to clone was already provided.");
	
		// Check if a column to remove is an IDColumn
		for (Column column : columnsToRemove) {
			if (column.getColumnType() == ColumnType.ID)
				throw new IllegalArgumentException("Cannot remove ID column");
		}
	
		// Check if provided table contains columns to remove
		for (Column column : columnsToRemove) {
			if (!table.getColumns().contains(column))
				throw new IllegalArgumentException("The provided table does not contain the column to remove: " + column);
		}
		
		setTableToClone(table);
		setCloneWithData(withData);
		setColumnsToRemove(columnsToRemove);
	
		return this;
	}
	
	@Override
	public Table create() throws TableCreationException {
		
		checkConsistency();

		setColumnNames(getAllColumns());
		
		String tableName = null;
		if (getTableToClone() != null) {
			// Handle cloning
			tableName = dbWrangler.cloneTable(getTableToClone().getName(), isCloneWithData(), false);

			for (Column column : getColumnsToRemove()) {
				dbWrangler.removeColumn(tableName, column.getName());
			}

			for (Column column : getNewColumns()) {
				dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			}

		} else {
			// Handle simple creation of empty table
			tableName = dbWrangler.createTable();
			for (Column column : getNewColumns()) {
				dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			}
		}
		
		addIndexes(tableName, getAllColumns());
		
		Collection<Column> columns = getAllColumns();
		columns.add(new IdColumn());
		Table newTable = createBaseTable(tableName, columns); 
		
		if (getTableToClone() != null) {
			//Set parent table id
			newTable.setParentTableId(getTableToClone().getId());
			
			//Clone metadata
			cloneMetadata(getTableToClone(), newTable);
		}

		// Register codelist on metadata DB
		return mdWrangler.save(newTable);
		
	}
	
	protected abstract void addIndexes(String tableName, Collection<Column> columns);
	
	protected abstract Table createBaseTable(String tableName, Collection<Column> columns);

	protected void checkDuplicateLabels(Collection<Column> columns) throws TableCreationException {
		// Check for duplicate labels
		List<String> labels = Lists.newArrayList();
		for (Column c : columns) {
			if (labels.contains(c.getLabel()))
				throw new TableCreationException("Two columns with the same label " + c.getLabel() + " were provided.");
			labels.add(c.getLabel());
		}
	}

	protected void setColumnNames(Collection<Column> columns) {
		int i = 0;
		String eligibleName = "field" + i;
		for (Column column : columns) {

			if (column.hasName()) {
				continue;
			}
			while (!Collections2.filter(columns, new ColumnHasName(eligibleName)).isEmpty()) {
				eligibleName = "field" + (++i);
			}
			column.setName(eligibleName);
		}

	}

	protected boolean isLabelPresent(String label, List<Column> columns) {
		Collection<Column> sameLabelColumns = Collections2.filter(columns, new ColumnHasLabel(label));
		if (!sameLabelColumns.isEmpty())
			return true;
		else
			return false;
	}

	protected void checkColumnsRelationship(Collection<Column> columns) throws Exception {
		for (Column column : columns) {
			if (column.hasRelationship()) {
				
				try {
					tableManager.get(column.getRelationship().getTargetTableRef()
							.getTableId());
				} catch (NoSuchTableException e) {
					throw new Exception(String.format("Referenced Codelist with ID %1$s does not exists.", column
							.getRelationship().getTargetTableRef().getTableId()));
				}
			}
		}
	}

	protected void cloneMetadata(Table sourceTable, Table destTable) {
		for (CubeMetadata<Serializable> m : sourceTable.getAllMetadata()){
			if (m.isInheritable())
				destTable.setMetadata(m);
		}
	}

	protected abstract void checkConsistency() throws TableCreationException;

	protected abstract boolean isAllowedColumn(Column column);

	protected abstract Collection<Column> getAllColumns();

	protected abstract Collection<Column> getNewColumns();

	protected abstract void addNewColumn(Column column);

	protected abstract boolean isAllowedCloneableTable(Table table);

	protected abstract Table getTableToClone();

	protected abstract void setTableToClone(Table table);
	
	protected abstract boolean isCloneWithData();
	
	protected abstract void setCloneWithData(boolean cloneWithData);
	
	protected abstract Collection<Column> getColumnsToRemove();
	
	protected abstract void setColumnsToRemove(Collection<Column> columns);

}
