package org.gcube.data.analysis.tabulardata.cube.time;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.AttributeColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.factories.CodeColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.factories.IdColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.DateType;
import org.gcube.data.analysis.tabulardata.model.datatype.IntegerType;
import org.gcube.data.analysis.tabulardata.model.datatype.TextType;
import org.gcube.data.analysis.tabulardata.model.metadata.table.TimePeriodTypeMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.type.TimeCodelistTableType;
import org.gcube.data.analysis.tabulardata.model.time.PeriodType;

import com.google.common.collect.Lists;

@Singleton
public class DayCodelistCreator implements TimeCodelistCreator {

	@Inject
	DatabaseWrangler dbWrangler;

	@Inject
	CubeMetadataWrangler cmWrangler;

	@Override
	public Table createTable() {
		Table dayCodelist = createTableMeta();

		String tableName = createTableOnDBWithColumns(dayCodelist.getColumns());
		dayCodelist.setName(tableName);
		
		return cmWrangler.save(dayCodelist, false);
	}

	private Table createTableMeta() {
		List<Column> columns = Lists.newArrayList(IdColumnFactory.create(), createCodeColumn(), createDayColumn(),
				createMonthColumn(), createQuarterColumn(), createYearColumn(), createDecadeColumn(),
				createIsoWeekColumn(), createIsoYearColumn());
		Table dayCodelist = new Table(new TimeCodelistTableType());
		dayCodelist.setColumns(columns);
		dayCodelist.setMetadata(new TimePeriodTypeMetadata(getManagedPeriodType()));
		return dayCodelist;
	}

	private String createTableOnDBWithColumns(List<Column> columns) {
		String tableName = dbWrangler.createTable();
		for (Column column : columns) {
			if (column.getColumnType() instanceof IdColumnType)
				continue;
			dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			dbWrangler.setNullable(tableName, column.getName(), false);
			if (column.getColumnType() instanceof CodeColumnType)
				dbWrangler.createUniqueIndex(tableName, column.getName());
		}
		dbWrangler.createInsertUpdateTriggerOnTable("update_time_dimension_"+tableName, tableName, "update_gregorian_day_dimension");
		return tableName;
	}

	private Column createIsoYearColumn() {
		Column resultColumn = AttributeColumnFactory.create(new IntegerType());
		resultColumn.setName("iso_year");
		return resultColumn;
	}

	private Column createDecadeColumn() {
		Column resultColumn = AttributeColumnFactory.create(new IntegerType());
		resultColumn.setName("decade");
		return resultColumn;
	}

	private Column createYearColumn() {
		Column yearColumn = AttributeColumnFactory.create(new IntegerType());
		yearColumn.setName("year");
		return yearColumn;
	}

	private Column createQuarterColumn() {
		Column quarterColumn = AttributeColumnFactory.create(new IntegerType());
		quarterColumn.setName("quarter_of_year");
		return quarterColumn;
	}

	private Column createCodeColumn() {
		Column codeColumn = CodeColumnFactory.create();
		codeColumn.setName("day_code");
		codeColumn.setDataType(new TextType(10));
		return codeColumn;
	}

	private Column createDayColumn() {
		Column dayColumn = AttributeColumnFactory.create(new DateType());
		dayColumn.setName("the_day");
		return dayColumn;
	}

	private Column createMonthColumn() {
		Column monthColumn = AttributeColumnFactory.create(new IntegerType());
		monthColumn.setName("month_of_year");
		return monthColumn;
	}

	private Column createIsoWeekColumn() {
		Column isoWeekColumn = AttributeColumnFactory.create(new IntegerType());
		isoWeekColumn.setName("week_of_iso_year");
		return isoWeekColumn;
	}

	@Override
	public PeriodType getManagedPeriodType() {
		return PeriodType.DAY;
	}

}
