package org.gcube.data.analysis.tabulardata.operation.worker;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import org.apache.commons.dbutils.DbUtils;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class BaseWorker implements Worker {

	private static final Logger log = LoggerFactory.getLogger(BaseWorker.class);

	private ImmutableJob job;
	
	protected OperationInvocation invocation;

	public BaseWorker(OperationInvocation invocation) {
		this.invocation = invocation;
		job = new ImmutableJob(0, ActivityStatus.INITIALIZING, invocation);
	}

	protected void inProgress(float progress) {
		job = new ImmutableJob(progress, ActivityStatus.IN_PROGRESS, job.getSourceInvocation());
	}

	protected void fail(OperationException exception) {
		job = new ImmutableJob(job.getProgress(), ActivityStatus.FAILED, job.getSourceInvocation(),
				new ImmutableJobResult(exception));
	}

	protected void succeed(Table resultTable) {
		job = new ImmutableJob(1, ActivityStatus.SUCCEDED, job.getSourceInvocation(), new ImmutableJobResult(
				resultTable));
	}

	public ImmutableJob getJob() {
		return job;
	}

	protected void executeSQLCommand(String sqlCommand, DatabaseConnectionProvider connectionProvider)
			throws OperationException {
		Connection conn = null;
		Statement stmt = null;
		try {
			conn = connectionProvider.getConnection();
			stmt = conn.createStatement();
			log.debug("Executing SQL command: " + sqlCommand);
			stmt.execute(sqlCommand);
		} catch (SQLException e) {
			String msg = "Unable to execute sql command : " + sqlCommand;
			log.error(msg);
			throw new OperationException(msg, e);
		} finally {
			DbUtils.closeQuietly(stmt);
			DbUtils.closeQuietly(conn);
		}
	}
	
	protected void executeSQLBatchCommands(DatabaseConnectionProvider connectionProvider, String... sqlCommands) throws OperationException {
		Connection conn = null;
		Statement stmt = null;
		try {
			conn = connectionProvider.getConnection();
			stmt  = conn.createStatement();
			
			for (int i = 0; i < sqlCommands.length; i++) {
				log.debug("Adding to batch: " + sqlCommands[i]);
				stmt.addBatch(sqlCommands[i]);
			}
			
			stmt.executeBatch();
		} catch (SQLException e){
			String msg = "Unable to execute batch sql command";
			log.error(msg,e);
			throw new OperationException(msg, e);
		} finally {
			DbUtils.closeQuietly(stmt);
			DbUtils.closeQuietly(conn);
		}
	}

}
