package org.gcube.data.analysis.tabulardata.operation.worker;

import java.util.Observable;

import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Worker extends Observable implements Runnable {

	private static final Logger log = LoggerFactory.getLogger(Worker.class);

	private float progress = 0;

	private WorkerException exception = null;

	private WorkerResult result = null;

	private WorkerStatus status = WorkerStatus.INITIALIZING;

	private OperationInvocation sourceInvocation;

	public Worker(OperationInvocation sourceInvocation) {
		this.sourceInvocation = sourceInvocation;
	}

	protected void updateProgress(float progress) {
		if (progress <= 0f || progress >= 1f)
			throw new IllegalArgumentException(
					"When updating progress, progress value must be between 0 and 1 (excluded)");
		this.progress = progress;
		status = WorkerStatus.IN_PROGRESS;
		setChanged();
		notifyObservers(this);
	}

	protected abstract WorkerResult execute() throws WorkerException;

	public void run() {
		try {
			log.debug("Starting worker: " + this.getClass().getSimpleName());
			WorkerResult result = execute();
			succeded(result);
			log.debug("Worker ended execution succesfully: " + this.getClass().getSimpleName());
		} catch (WorkerException e) {
			log.debug("Worker failed execution: " + this.getClass().getSimpleName());
			failed(e);
		} catch (Exception e) {
			log.debug("Worker failed execution: " + this.getClass().getSimpleName());
			failed(new WorkerException("Internal operation error", e));
		}
	}

	private void succeded(WorkerResult result) {
		this.progress = 1f;
		this.result = result;
		this.status = WorkerStatus.SUCCEDED;
		setChanged();
		notifyObservers(this);
	}

	private void failed(WorkerException e) {
		this.status = WorkerStatus.FAILED;
		this.exception = e;
		setChanged();
		notifyObservers(this);
	}

	public float getProgress() {
		return progress;
	}

	public WorkerException getException() {
		return exception;
	}

	public WorkerResult getResult() {
		return result;
	}

	public WorkerStatus getStatus() {
		return status;
	}

	public OperationInvocation getSourceInvocation() {
		return sourceInvocation;
	}

}
