package org.gcube.data.analysis.tabulardata.operation.comet;

import java.io.IOException;
import java.io.StringWriter;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map.Entry;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;

import org.apache.commons.dbutils.DbUtils;
import org.fao.fi.comet.mapping.model.Mapping;
import org.fao.fi.comet.mapping.model.MappingData;
import org.fao.fi.comet.mapping.model.MappingDetail;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDTypeValue;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.comet.model.MappedRow;
import org.gcube.data.analysis.tabulardata.operation.comet.model.MappedValue;
import org.gcube.data.analysis.tabulardata.operation.comet.model.Rule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MappingParser {

	private static Logger logger = LoggerFactory.getLogger(MappingParser.class);
	
	public static enum MappingDirection{
		FORWARD,
		BACKWARD
	}
	
	public static class ParserConfiguration{
		private Table previousCodelistVersion;
		private Table currentCodelistVersion;
		private double scoreThreshold=1.0d;
		private boolean skipOnError=true;
		private MappingDirection direction=MappingDirection.BACKWARD; //Cotrix behaviour
		
		public ParserConfiguration(Table previousCodelistVersion,
				Table currentCodelistVersion) {
			super();
			this.previousCodelistVersion = previousCodelistVersion;
			this.currentCodelistVersion = currentCodelistVersion;
		}

		public Table getCurrentCodelistVersion() {
			return currentCodelistVersion;
		}
		public Table getPreviousCodelistVersion() {
			return previousCodelistVersion;
		}
		
		public void setDirection(MappingDirection direction) {
			this.direction = direction;
		}
		
		public MappingDirection getDirection() {
			return direction;
		}
		
		
		public boolean isSkipOnError() {
			return skipOnError;
		}
		
		/**
		 * @return the scoreThreshold
		 */
		public double getScoreThreshold() {
			return scoreThreshold;
		}
		/**
		 * @param scoreThreshold the scoreThreshold to set
		 */
		public void setScoreThreshold(double scoreThreshold) {
			this.scoreThreshold = scoreThreshold;
		}
		/**
		 * @param skipOnError the skipOnError to set
		 */
		public void setSkipOnError(boolean skipOnError) {
			this.skipOnError = skipOnError;
		}
	}

	private ParserConfiguration config;
	private SQLExpressionEvaluatorFactory evaluatorFactory;
	private CubeManager cubeManager;
	private DatabaseConnectionProvider connectionProvider;
	
	
	public MappingParser(ParserConfiguration config,SQLExpressionEvaluatorFactory evaluatorFactory,CubeManager cubeManager, DatabaseConnectionProvider connectionProvider) throws IOException, JAXBException {
		this.config=config;
		this.evaluatorFactory=evaluatorFactory;
		this.cubeManager=cubeManager;
		this.connectionProvider=connectionProvider;
		
		
		// Marshall expression
		JAXBContext jaxbContext = JAXBContext.newInstance(Expression.class);
		marshaller = jaxbContext.createMarshaller();
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
		
	}

	
	private Marshaller marshaller;
	
	
	private long foundMappingsCount=0;
	private long parsedRulesCount=0;
	
	private Table rulesTable=null;
	private Connection conn=null;
	private PreparedStatement stmt=null;
	
	

	public void parse(String xmlFile) throws Exception{
		XMLStreamReader xmler =null;
		try{
			if(rulesTable==null) rulesTable=Rule.createTable(cubeManager);
			
			//TODO parse mapping by mapping instead of DOM for less memory usage
			logger.debug("Parsing file "+xmlFile);
			XMLInputFactory xmlif = XMLInputFactory.newInstance();
			xmler= xmlif.createXMLStreamReader(OperationHelper.getInputStreamFromUrl(xmlFile));
			JAXBContext ctxUnmarshall = JAXBContext.newInstance(MappingData.class);
			MappingData deserialized = (MappingData)ctxUnmarshall.createUnmarshaller().unmarshal(xmler);
			Collection<Mapping> mappings=deserialized.getMappings();
			logger.debug("Parsed {} mappings",mappings.size());	
			foundMappingsCount=mappings.size();
			for(Mapping mapping:mappings){
				try{
					MappedRow sourceRow=new MappedRow(mapping.getSource());
					for(MappingDetail targetCandidate:mapping.getTargets()){
						if(targetCandidate.getScore()>=config.getScoreThreshold()){
							MappedRow targetRow=new MappedRow(targetCandidate.getTargetElement());
							
							List<MappedValue> mappedValueList=config.getDirection().equals(MappingDirection.FORWARD)?
									getChangeSet(sourceRow,targetRow):getChangeSet(targetRow, sourceRow);
									
							for(MappedValue mapped:mappedValueList){
								Rule rule=getRule(mapped);
								storeRule(rule);
								System.out.println(rule);
								parsedRulesCount++;
							}
						}
					}
				}catch(Throwable t){
					logger.debug("Skipping mapping for source element {} ",mapping.getSource().getId().getElementId());
				}
			}	
			logger.debug(String.format("Generated %s rules out of %s mappings into %s table.",parsedRulesCount,foundMappingsCount,rulesTable));
		}finally{
			if(xmler!=null)xmler.close();
			if(stmt!=null)DbUtils.closeQuietly(stmt);
			if(conn!=null)DbUtils.closeQuietly(conn);
		}
	}




	

	private Rule getRule(MappedValue mapped) throws Exception{
		Column referred=getReferredColumnByLabel(mapped.getFieldLabel());
		TDTypeValue targetValue=referred.getDataType().fromString(mapped.getTargetValue());
		TDTypeValue sourceValue=referred.getDataType().fromString(mapped.getSourceValue());		
		return new Rule(sourceValue,targetValue,
				getConfig().getCurrentCodelistVersion().getColumnReference(referred),true,
				evaluatorFactory.getEvaluator(sourceValue).evaluate(),
				evaluatorFactory.getEvaluator(targetValue).evaluate());
	}

	private Column getReferredColumnByLabel(String label) throws Exception{
		for(Column col:getConfig().getCurrentCodelistVersion().getColumnsExceptTypes(IdColumnType.class,ValidationColumnType.class)){
			if(col.getName().equals(label)) return col;
			try{
				for(LocalizedText localizedLabel:col.getMetadata(NamesMetadata.class).getTexts())
					if(localizedLabel.getValue().equals(label)) return col;
			}catch(NoSuchMetadataException e){
				//skip column
			}
		}
		logger.debug(String.format("Unable to get Column with label %s in table %s",label,getConfig().getCurrentCodelistVersion()));
		throw new Exception("Unable to find a column named "+label);
	}
	
	
	private int storeRule(Rule r) throws SQLException, JAXBException{
		
		
		if(conn==null) conn=connectionProvider.getConnection();
		
		
		
		if(stmt==null) {
			String insertStatement="INSERT INTO "+rulesTable.getName()+"("+
					Rule.ENABLED+","+
					Rule.REFERRED_CODELIST_COLUMN+","+
					Rule.TO_CHANGE_VALUE_FIELD+","+
					Rule.TO_CHANGE_VALUE_DESCRIPTION+","+
					Rule.TO_SET_VALUE_FIELD+","+
					Rule.TO_SET_VALUE_DESCRIPTION+") VALUES (?,?,?,?,?,?)";
			stmt=conn.prepareStatement(insertStatement);			
		}
		
		stmt.setBoolean(1, r.isEnabled());		
		stmt.setString(2, r.getReferredCodelistColumn().getColumnId().toString());
		stmt.setString(3, serializeExpression(r.getToChangeValue()));
		stmt.setString(4, r.getToChangeValueDescription());
		stmt.setString(5, serializeExpression(r.getToSetValue()));
		stmt.setString(6, r.getToSetValueDescription());
		
		return stmt.executeUpdate();
	}

	
	
	
	private List<MappedValue> getChangeSet(MappedRow original,MappedRow newRow) throws Exception{		 
		ArrayList<MappedValue> toReturn=new ArrayList<>();		
		for(Entry<String,String> originalEntry:original.entrySet()){
			if(newRow.containsKey(originalEntry.getKey())){
				if(!originalEntry.getValue().equals(newRow.get(originalEntry.getKey())))
					toReturn.add(new MappedValue(originalEntry.getValue(), newRow.get(originalEntry.getKey()), originalEntry.getKey()));
			}
		}
		return toReturn;
	}
	
	
	private String serializeExpression(Expression toSerialize) throws JAXBException{
		StringWriter stringWriter = new StringWriter();
		marshaller.marshal(toSerialize, stringWriter);
		return stringWriter.toString();
	}
	
	
	
	public ParserConfiguration getConfig() {
		return config;
	}
	public long getParsedRulesCount() {
		return parsedRulesCount;
	}
	public long getFoundMappingsCount() {
		return foundMappingsCount;
	}
	
	public Table getRulesTable() {
		return rulesTable;
	}
}
