package org.gcube.data.analysis.tabulardata.operation.csv.importer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URL;
import java.util.List;
import java.util.Map;

import net.sf.csv4j.CSVReaderProcessor;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.type.GenericTableType;
import org.gcube.data.analysis.tabulardata.operation.worker.BaseWorker;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationException;
import org.postgresql.PGConnection;
import org.postgresql.copy.CopyManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.gcube.data.analysis.tabulardata.operation.csv.Constants.*;

public class CSVImport extends BaseWorker {

	private static Logger logger = LoggerFactory.getLogger(CSVImport.class);

	private String encoding;
	private Character separator;
	private Boolean hasHeader;
	private String url;

	private CubeManager cubeManager; 
	private DatabaseConnectionProvider connectionProvider;

	public CSVImport(OperationInvocation invocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		super(invocation);
		retrieveParameters();
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	@Override
	public void run() {
		logger.trace("starting import CSV");
		inProgress(0f);
		File csvFile;
		try {
			csvFile= getInputFile(url);
		} catch (Exception e) {
			fail(new OperationException("failed downloading csv file",e));
			return;
		}
		inProgress(0.3f);
		InitializerProcessor initializerProcessor;
		try {
			initializerProcessor = getInitializerProcessor(csvFile);
		}catch (Exception e) {
			fail(new OperationException("failed reading csv header",e));
			return;
		}
		inProgress(0.5f);

		Table table;
		try{
			table = createTable(initializerProcessor.getColumns());
		}catch (Exception e) {
			fail(new OperationException("failed creating table",e));
			return;
		}
		inProgress(0.6f);
		try{
			copy(csvFile, table);
		}catch (Exception e) {
			fail(new OperationException("failed copying lines to table",e));
			return;
		}
		inProgress(1f);
		succeed(table);
		logger.trace("finished import CSV");
	}


	private Table createTable(List<Column> columns) throws Exception{
		TableCreator tableCreator = cubeManager.createTable(new GenericTableType());
		for (Column column : columns)
			tableCreator.addColumn(column);
		return tableCreator.create();
	}

	private InitializerProcessor getInitializerProcessor(File csvFile) throws Exception{
		InitializerProcessor initializer = new InitializerProcessor();

		//starting csv processing
		CSVReaderProcessor csvReaderProcessor = new CSVReaderProcessor();
		csvReaderProcessor.setDelimiter(separator);
		csvReaderProcessor.setHasHeader(hasHeader);
		InputStreamReader isrInitializer =  null;
		try{
			isrInitializer =  new InputStreamReader(new FileInputStream(csvFile), encoding); 
			new CSVReaderProcessor().processStream(isrInitializer, initializer);
			return initializer;
		}finally{
			if (isrInitializer!=null) isrInitializer.close(); 
		}

	}

	private File getInputFile(String storageUrl) throws Exception{
		URL url = new URL(storageUrl);
		InputStream inputStream = url.openConnection().getInputStream();

		File tempFile = File.createTempFile("import", ".csv");

		OutputStream outputStream = new FileOutputStream(tempFile);

		int read = 0;
		byte[] bytes = new byte[1024];

		while ((read = inputStream.read(bytes)) != -1) 
			outputStream.write(bytes, 0, read);

		return tempFile;
	}

	private long copy(File csvFile, Table table) throws Exception {
		PGConnection conn = connectionProvider.getPostgreSQLConnection();
		CopyManager cpManager = conn.getCopyAPI();
		StringBuilder columns = new StringBuilder();
		for (Column c : table.getColumns()) {
			if (!(c.getColumnType() instanceof IdColumnType))
				columns.append(c.getName() + ",");
		}
		columns.deleteCharAt(columns.length() - 1);

		String sqlCmd = String.format("COPY %s ( %s ) FROM STDIN ( FORMAT CSV ,DELIMITER '%c', HEADER %b, ENCODING '%s');", table.getName(),
				columns.toString(), separator, hasHeader, encoding);
		logger.info("executing copy for csv import with query {}",sqlCmd);
		InputStreamReader inputStreamReader =null;
		try{
			inputStreamReader = new InputStreamReader(new FileInputStream(csvFile), encoding);
			long lines = cpManager.copyIn(sqlCmd, inputStreamReader);
			inputStreamReader.close();
			return lines;
		}finally{
			if (inputStreamReader!=null)
				inputStreamReader.close();
		}
	}

	private void retrieveParameters() {
		Map<String, Object> parameters = invocation.getParameterInstances();
		url = (String) parameters.get(URL);
		separator = (Character) parameters.get(SEPARATOR);
		hasHeader = (Boolean) parameters.get(HASHEADER);
		encoding = (String) parameters.get(ENCODING);
	}

}
