package org.gcube.data.analysis.tabulardata.operation.data.add;

import java.util.Map;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.TableReferenceReplacer;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.factories.BaseColumnFactory;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;

public class AddColumn extends Worker {

	private CubeManager cubeManager;
	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;
	private DatabaseConnectionProvider connectionProvider;
	
	public AddColumn(OperationInvocation sourceInvocation,
			CubeManager cubeManager,SQLExpressionEvaluatorFactory sqlEvaluatorFactory,
			DatabaseConnectionProvider connectionProvider) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.sqlEvaluatorFactory=sqlEvaluatorFactory;
		this.connectionProvider=connectionProvider;
	}


	private Table targetTable;
	private Column theNewColumn;
	private Table resultTable;
	private Table diffTable;
	
	private Expression toSetValue=null;
	
	
	@Override
	protected WorkerResult execute() throws WorkerException {
		updateProgress(0.1f, "Initiating");
		targetTable=cubeManager.getTable(getSourceInvocation().getTargetTableId());
		updateProgress(0.4f,"Creating column");
		theNewColumn=createColumn();
		resultTable=cubeManager.createTable(targetTable.getTableType()).like(targetTable, true).addColumn(theNewColumn).create();
		diffTable=cubeManager.createTable(targetTable.getTableType()).addColumn(theNewColumn).create();
		updateProgress(0.5f,"Filling with values");
		fillWithDefault();
		updateProgress(0.9f,"Finalizing");
		return new ImmutableWorkerResult(resultTable, diffTable);
	}

	
	private Column createColumn(){
		Map<String,Object> params=getSourceInvocation().getParameterInstances();
		
		ColumnType colType=(ColumnType) params.get(AddColumnFactory.COLUMN_TYPE.getIdentifier());
		
		DataType dataType=null;
		if(params.containsKey(AddColumnFactory.DATA_TYPE.getIdentifier()))dataType=(DataType) params.get(AddColumnFactory.DATA_TYPE.getIdentifier());
		else dataType=colType.getDefaultDataType();
		
		LocalizedText label=null;
		if(params.containsKey(AddColumnFactory.LABEL.getIdentifier()))label=(LocalizedText) params.get(AddColumnFactory.LABEL.getIdentifier());
		else label=new ImmutableLocalizedText("New Column"); 
		
		return BaseColumnFactory.getFactory(colType).create(label, dataType);
	}

	private void initializeValueExpression(){
		try{
			toSetValue= OperationHelper.getParameter(AddColumnFactory.VALUE_PARAMETER, getSourceInvocation());			
			TableReferenceReplacer replacer=new TableReferenceReplacer(toSetValue);
			replacer.replaceTableId(targetTable.getId(), resultTable.getId());
			
			toSetValue=replacer.getExpression();
		}catch(Throwable t){
			//Expression not set, use default
			toSetValue=theNewColumn.getDataType().getDefaultValue();
		}
	}
	
	private void fillWithDefault() throws WorkerException{		
		initializeValueExpression();
		String stmt=String.format("UPDATE %s SET %s = %s",
				resultTable.getName(),
				theNewColumn.getName(),
				sqlEvaluatorFactory.getEvaluator(toSetValue).evaluate()
				);
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, stmt);
		} catch (Exception e) {
			throw new WorkerException("Error occurred while executing SQL command "+stmt, e);
		}
	}
	
}
