package org.gcube.data.analysis.tabulardata.operation.data.add;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.functions.Cast;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.CompositeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetColumnParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.types.DataWorker;
import org.gcube.data.analysis.tabulardata.operation.worker.types.RollbackWorker;

@Singleton
public class UnionFactory extends TableTransformationWorkerFactory{

	private static final OperationId OPERATION_ID=new OperationId(3208);
	
	public static final TargetColumnParameter SOURCE_COLUMN_PARAMETER=new TargetColumnParameter("source", "Source", "Source column", Cardinality.ONE);
	public static final TargetColumnParameter TARGET_COLUMN_PARAMETER=new TargetColumnParameter("target", "Target", "Target Column", Cardinality.ONE);
	
	public static final CompositeParameter MAPPINGS_PARAMETER=new CompositeParameter(
			"mappings", "Mappings", "Mappings betweeen target and source table columns", new Cardinality(1, Integer.MAX_VALUE),
			Arrays.asList(new Parameter[]{SOURCE_COLUMN_PARAMETER,TARGET_COLUMN_PARAMETER}));
	
	
	
	private CubeManager cubeManager;
	private DatabaseConnectionProvider connectionProvider;
	private SQLExpressionEvaluatorFactory evaluatorFactory;
	
	@Inject
	public UnionFactory(CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			SQLExpressionEvaluatorFactory evaluatorFactory) {
		super();
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evaluatorFactory=evaluatorFactory;
	}
	
	@Override
	public DataWorker createWorker(OperationInvocation invocation)
			throws InvalidInvocationException {
		performBaseChecks(invocation, cubeManager);
		checkMappings(invocation);
		return new UnionWorker(invocation,cubeManager,connectionProvider,evaluatorFactory);
	}
	
	@Override
	protected String getOperationDescription() {
		return "Import data from a compatible table";
	}
	
	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}

	@Override
	protected String getOperationName() {
		return "Union";
	}
	
	@Override
	protected List<Parameter> getParameters() {
		return Collections.singletonList((Parameter)MAPPINGS_PARAMETER);
	}
	
	
	@Override
	public String describeInvocation(OperationInvocation invocation)
			throws InvalidInvocationException {
		performBaseChecks(invocation, cubeManager);
		ColumnReference ref=(ColumnReference) getMappings(invocation).get(0).get(SOURCE_COLUMN_PARAMETER.getIdentifier());
		Table sourceTable=cubeManager.getTable(ref.getTableId());
		return String.format("Import %s rows",OperationHelper.retrieveTableLabel(sourceTable));
	}
	
	private void checkMappings(OperationInvocation invocation)throws InvalidInvocationException{
		List<Map<String,Object>> mappings=getMappings(invocation);
		Table source=null;
		Table target=cubeManager.getTable(invocation.getTargetTableId());
		for(Map<String,Object> mapping:mappings){
			// all source columns must belong to same table
			ColumnReference sourceRef=(ColumnReference) mapping.get(SOURCE_COLUMN_PARAMETER.getIdentifier());
			if(source==null)source=cubeManager.getTable(sourceRef.getTableId());
			else {
				if(!sourceRef.getTableId().equals(source.getId())) throw new InvalidInvocationException(invocation, String.format("Incoherent source table id %s, expected %s",sourceRef.getTableId(),source.getId()));
			}
			// all target columns must belong to target table
			ColumnReference targetRef=(ColumnReference) mapping.get(TARGET_COLUMN_PARAMETER.getIdentifier());
			if(!targetRef.getTableId().equals(target.getId())) throw new InvalidInvocationException(invocation, String.format("Incoherent target table id %s, expected %s",targetRef.getTableId(),target.getId()));
			// all source-target columns must have compatible data types
			
			Column sourceCol=source.getColumnById(sourceRef.getColumnId());		
			DataType sourceType=sourceCol.getDataType();
			Column targetCol=target.getColumnById(targetRef.getColumnId());
			DataType targetType=targetCol.getDataType();
			if(!Cast.isCastSupported(sourceType, targetType)) 
				throw new InvalidInvocationException(invocation,String.format("Cannot map %s values to %s column.", OperationHelper.retrieveColumnLabel(sourceCol),OperationHelper.retrieveColumnLabel(targetCol)));
		}
	}
	
	protected static List<Map<String,Object>> getMappings(OperationInvocation invocation){		
		Object actualParam=invocation.getParameterInstances().get(MAPPINGS_PARAMETER.getIdentifier());
		if(actualParam instanceof Map){
			// one param
			return Collections.singletonList((Map<String, Object>) actualParam);
		}else{
			Iterable<Map<String,Object>> it=(Iterable<Map<String, Object>>) actualParam;
			ArrayList<Map<String,Object>> toReturn=new ArrayList<Map<String,Object>>();
			for(Map<String,Object> elem:it)
				toReturn.add(elem);
			return toReturn;
			
		}		
	}
	
	@Override
	public boolean isRollbackable() {
		return true;
	}
	
	@Override
	public RollbackWorker createRollbackWoker(Table diffTable,
			Table createdTable, OperationInvocation oldInvocation) {
		return new UnionRollbackWorker(diffTable, createdTable, oldInvocation, cubeManager, connectionProvider);
	}
}
