package org.gcube.data.analysis.tabulardata.operation.table;

import java.util.List;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.MultivaluedStringParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.EligibleOperation;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationNotEligibleException;
import org.reflections.Reflections;
import org.reflections.scanners.SubTypesScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

@Singleton
public class ChangeTableTypeFactory extends TableTransformationWorkerFactory {

	static final String TABLE_TYPE_PARAMETER_ID = "tableType";

	private static final OperationId OPERATION_ID = new OperationId(ChangeTableTypeFactory.class);

	private final static List<TableType> availableTableTypes = Lists.newArrayList();

	private static final Logger log = LoggerFactory.getLogger(ChangeTableTypeFactory.class);

	static {
		Reflections reflections = new Reflections("org.gcube.data.analysis.tabulardata.model.table",
				new SubTypesScanner());
		Set<Class<? extends TableType>> classes = reflections.getSubTypesOf(TableType.class);
		for (Class<? extends TableType> type : classes) {
			String msg = "Unable to instantiate table type";
			try {
				availableTableTypes.add(type.newInstance());
			} catch (InstantiationException e) {
				log.error(msg, e);
			} catch (IllegalAccessException e) {
				log.error(msg, e);
			}
		}
		log.trace("Loaded available table types: " + availableTableTypes);
	}

	@Inject
	private CubeManager cubeManager;

	public EligibleOperation getEligibleOperation(TableId tableId) throws OperationNotEligibleException {
		return getEligibleOperation(cubeManager.getTable(tableId));
	}

	public EligibleOperation getEligibleOperation(Table table) {
		TableType tableType = table.getTableType();
		List<String> admittedValues = getAdmittedValues(tableType);
		Parameter tableTypeParameter = new MultivaluedStringParameter(TABLE_TYPE_PARAMETER_ID, "Table Type",
				"Table type", Cardinality.ONE, admittedValues);
		List<Parameter> parameters = Lists.newArrayList(tableTypeParameter);
		return new EligibleOperation(getOperationDescriptor(), parameters, table.getId());

	}

	public List<String> getAdmittedValues(TableType tableType) {
		List<String> admittedValues = Lists.newArrayList();
		for (TableType availableTableType : availableTableTypes) {
			if (availableTableType.equals(tableType))
				continue;
			admittedValues.add(availableTableType.getName());
		}
		return admittedValues;
	}

	public Worker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		checkInvocation(invocation);
		Table targetTable = cubeManager.getTable(invocation.getTargetTableId());
		String targetTableTypeName = (String) invocation.getParameterInstances().get(TABLE_TYPE_PARAMETER_ID);
		TableType targetTableType = getTableType(targetTableTypeName);
		return new ChangeTableType(invocation, cubeManager, targetTable, targetTableType);
	}

	public TableType getTableType(String targetTableTypeName) {
		TableType targetTableType = null;
		for (TableType availableTableType : availableTableTypes) {
			if (availableTableType.getName().equals(targetTableTypeName))
				targetTableType = availableTableType;
		}
		return targetTableType;
	}

	private void checkInvocation(OperationInvocation invocation) throws InvalidInvocationException {
		try {
			List<String> admittedValues = getAdmittedValues(cubeManager.getTable(invocation.getTargetTableId())
					.getTableType());
			String tableTypeName = (String) invocation.getParameterInstances().get(TABLE_TYPE_PARAMETER_ID);
			if (!admittedValues.contains(tableTypeName))
				throw new InvalidInvocationException(invocation,
						"Provided value is not admitted: " + tableTypeName);
		} catch (InvalidInvocationException e) {
			throw e;
		} catch (Exception e) {
			log.error("Invalid invocation caused by exception", e);
			throw new InvalidInvocationException(invocation, "Uncaught exception caused invocation check to fail",e);
		}
	}

	@Override
	protected String getOperationName() {
		return "Change table type";
	}

	@Override
	protected String getOperationDescription() {
		return "Modify the table type";
	}

	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}

}
