package org.gcube.data.analysis.tabulardata.operation.column;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.ColumnTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.LocalizedTextChoiceParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.LocalizedTextParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.EligibleOperation;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationNotEligibleException;

import com.google.common.collect.Lists;

@Singleton
public class ChangeLabelFactory extends ColumnTransformationWorkerFactory {

	public static final String NEW_LABEL_ID = "newLabel";

	public static final String TARGET_LABEL_ID = "targetLabel";

	@Inject
	private CubeManager cubeManager;

	public EligibleOperation getEligibleOperation(TableId tableId, ColumnLocalId columnId) throws OperationNotEligibleException {
		return getEligibleOperation(cubeManager.getTable(tableId), columnId);
	}

	public EligibleOperation getEligibleOperation(Table table, ColumnLocalId columnId) throws OperationNotEligibleException {
		Column column = table.getColumnById(columnId);
		List<LocalizedText> labels = getLabels(column);
		if (labels == null || labels.isEmpty())
			throw new OperationNotEligibleException(table.getId(), columnId, "Column has no labels");
		List<Parameter> parameters = createParameters(labels);
		return new EligibleOperation(getOperationDescriptor(), parameters, table.getId(), columnId);
	}

	public List<Parameter> createParameters(List<LocalizedText> labels) {
		List<Parameter> parameters = new ArrayList<Parameter>();
		parameters.add(new LocalizedTextChoiceParameter(TARGET_LABEL_ID, "Target label", "Label to modify",
				Cardinality.ONE, labels));
		parameters.add(new LocalizedTextParameter(NEW_LABEL_ID, "New label", "Label to be created", Cardinality.ONE));
		return parameters;
	}

	private List<LocalizedText> getLabels(Column column) {
		try {
			return column.getMetadata(NamesMetadata.class).getTexts();
		} catch (NoSuchMetadataException e) {
			return Lists.newArrayList();
		}
	}

	public Worker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		checkTableIdPresence(invocation);
		checkColumnIdPresence(invocation);
		LocalizedText newLabel = getParameterValue(NEW_LABEL_ID, LocalizedText.class, invocation);
		LocalizedText targetLabel = getParameterValue(TARGET_LABEL_ID, LocalizedText.class, invocation);
		return new ChangeLabel(invocation,cubeManager,invocation.getTargetTableId(), invocation.getTargetColumnId(), targetLabel, newLabel);
	}

	@Override
	protected String getOperationName() {
		return "Change label";
	}

	@Override
	protected String getOperationDescription() {
		return "Modify an existing label";
	}
	
	@Override
	protected OperationId getOperationId() {
			return new OperationId(1005);
	}
}
