package org.gcube.data.analysis.tabulardata.operation.validation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableMetaCreator;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.logical.IsNotNull;
import org.gcube.data.analysis.tabulardata.expression.logical.Or;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.type.AnnotationColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeDescriptionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeNameColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataLocaleMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.Validation;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ValidationsMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerStatus;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerWrapper;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CodelistValidator extends Worker {

	private static final Logger log = LoggerFactory.getLogger(CodelistValidator.class);

	private static final Class[] validColumnTypes=new Class[]{
		CodeColumnType.class,		
		CodeNameColumnType.class,
		CodeDescriptionColumnType.class,
		AnnotationColumnType.class,
		IdColumnType.class
	};


	// Factories


	private DuplicateValuesInColumnValidatorFactory duplicateInColumnFactory;
	private ValidateDataWithExpressionFactory validateDataWithExpressionFactory;
	private DuplicateRowValidatorFactory duplicateRowsFactory;




	CubeManager cubeManager;

	DatabaseConnectionProvider connectionProvider;

	Table targetTable;


	List<Validation> toSetValidations;
	HashMap<ColumnLocalId,List<Validation>> columnValidations=new HashMap<ColumnLocalId, List<Validation>>();

	public CodelistValidator(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			DuplicateValuesInColumnValidatorFactory duplicateInColumnFactory,
			ValidateDataWithExpressionFactory validateWithExpressionFactory,
			DuplicateRowValidatorFactory duplicateRowFactory) {
		super(sourceInvocation);		
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.duplicateInColumnFactory=duplicateInColumnFactory;
		this.validateDataWithExpressionFactory=validateWithExpressionFactory;
		this.duplicateRowsFactory=duplicateRowFactory;		
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		retrieveTargetTable();
		updateProgress(0.1f);
		//perform metadata checks
		log.debug("Checking metadata constraints for table "+targetTable.getId());
		initializeValidationsMetadata();
		log.debug("Unique code column : "+checkExistingUniqueCodeColumn());			
		log.debug("Existing code name column : "+checkExistingCodeNameColumn());
		log.debug("Duplicate locales : "+checkDuplicateCodeNameDataLocales());
		log.debug("Allowed column types only : "+checkInvalidColumnTypes());
		log.debug("Needed locale and labels : "+checkLocaleAndLabels());
		updateProgress(0.3f);
		updateProgress(0.4f);		

		// Data validation
		log.debug("Validating duplicates in code columns..");
		checkDuplicatesInCodeColumn();
		updateProgress(0.5f);
		log.debug("Validating not null in code columns ...");
		checkNullValuesInCodeColumn();
		updateProgress(0.7f);
		log.debug("Validating name presence in each tuple...");
		checkNamePresenceInEachTuple();
		updateProgress(0.8f);
		log.debug("Validating tuple uniqueness... ");
		checkDuplicates();
		updateProgress(0.9f);

		//TODO Not sure if to apply
		//checkDuplicatesInAnyNames();


		return new ImmutableWorkerResult(createMetaValidatedTable());
	}

	private void retrieveTargetTable() {
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
	}

	private void initializeValidationsMetadata(){
		toSetValidations=getToEnrichTableMetaValidations(targetTable);
	}

	//********************* Metadata Validation


	private boolean checkExistingUniqueCodeColumn(){
		List<Column> codeColumns=targetTable.getColumnsByType(CodeColumnType.class);
		int codeColumnCount=codeColumns.size();
		boolean valid= codeColumnCount==1;
		if(!valid){
			for(Column col:codeColumns)
				addColumnValidation("Duplicated Code column",valid,col);
		}
		toSetValidations.add(new Validation("Must have one and only code column", valid));
		return valid;
	}

	private boolean checkExistingCodeNameColumn(){
		List<Column> codeNameColumns=targetTable.getColumnsByType(CodeNameColumnType.class);		
		boolean valid=codeNameColumns.size()>0;
		toSetValidations.add(new Validation("Must have at least one codename columns", valid));
		return valid;
	}

	private boolean checkDuplicateCodeNameDataLocales(){
		List<Column> codeNameColumns=targetTable.getColumnsByType(CodeNameColumnType.class);

		// Construct locale map
		HashMap<String,List<Column>> codeNamesLocales=new HashMap<String,List<Column>>();
		for(Column c:codeNameColumns)
			if (c.contains(DataLocaleMetadata.class)){
				String locale=c.getMetadata(DataLocaleMetadata.class).getLocale();
				if(!codeNamesLocales.containsKey(locale))
					codeNamesLocales.put(locale, new ArrayList<Column>());
				codeNamesLocales.get(locale).add(c);
			}


		//******** Check locale duplicates

		boolean valid=true;
		for(Entry<String,List<Column>> entry:codeNamesLocales.entrySet()){
			boolean duplicate=entry.getValue().size()>1;			
			if(duplicate)valid=false;			
			for(Column col:entry.getValue())
				addColumnValidation("Code column with unique locale",!duplicate,col);
		}

		toSetValidations.add(new Validation("Must have at most one CodeName column for each data locale", valid));
		return valid;
	}

	private boolean checkInvalidColumnTypes(){
		boolean globalValid=true;
		for(Column col:targetTable.getColumnsByType(validColumnTypes))
			addColumnValidation("Allowed column type ", true, col);

		List<Column> invalidCols=targetTable.getColumnsExceptTypes(validColumnTypes);
		globalValid=invalidCols.isEmpty();
		if(!globalValid)
			for(Column col:invalidCols) addColumnValidation("Allowed column type ", false, col);


		toSetValidations.add(new Validation("Only Code, CodeName, CodeDescription and Annotation column types allowed.", globalValid));
		return globalValid;
	}



	private boolean checkLocaleAndLabels(){
		boolean globalValid=true;
		Class[] toCheckColumnTypes=new Class[]{
				CodeNameColumnType.class,
				CodeDescriptionColumnType.class,
				AnnotationColumnType.class,
		};

		for(Column col:targetTable.getColumnsByType(toCheckColumnTypes)){
			boolean validColumn=checkLocaleAndLabels(col);
			if(!validColumn) globalValid=false;
			addColumnValidation("Must have Data locale metadata and at least one label", validColumn, col);
		}

		toSetValidations.add(new Validation("Each CodeName, CodeDescription and Annotation column with DataLocale and at least one label",globalValid));
		return globalValid;
	}



	//**************** Data Validation

	private void checkDuplicatesInCodeColumn() throws WorkerException{
		List<Column> toCheckColumns=targetTable.getColumnsByType(CodeColumnType.class);
		WorkerWrapper wrapper=new WorkerWrapper(duplicateInColumnFactory);
		for(Column col : toCheckColumns){
			try{
				WorkerStatus status=wrapper.execute(targetTable.getId(), col.getLocalId(), null);
				processStep(status,wrapper.getResult());
			}catch(InvalidInvocationException e){
				throw new WorkerException("Unable to execute wrapped worker ",e);
			}
		}
	}

	private void checkNullValuesInCodeColumn() throws WorkerException{
		List<Column> toCheckColumns=targetTable.getColumnsByType(CodeColumnType.class);

		WorkerWrapper wrapper=new WorkerWrapper(validateDataWithExpressionFactory);
		for(Column col : toCheckColumns){
			try{
				//Form Expression
				ColumnReference targetColumnReference =  new ColumnReference(targetTable.getId(), col.getLocalId());
				IsNotNull condition=new IsNotNull(targetColumnReference);
				HashMap<String,Object> map=new HashMap<String,Object>();
				map.put(validateDataWithExpressionFactory.EXPRESSION_PARAMETER.getIdentifier(), condition);


				WorkerStatus status=wrapper.execute(targetTable.getId(), col.getLocalId(), map);
				processStep(status,wrapper.getResult());
			}catch(InvalidInvocationException e){
				throw new WorkerException("Unable to execute wrapped worker ",e);
			}
		}	
	}

	private void checkNamePresenceInEachTuple() throws WorkerException{
		// Form Expression nameX isnot null || name Y is not null etc...

		List<Column> toCheckColumns=targetTable.getColumnsByType(CodeNameColumnType.class);
		List<Expression> orArguments=new ArrayList<Expression>();
		for(Column col : toCheckColumns){
			ColumnReference targetColumnReference =  new ColumnReference(targetTable.getId(), col.getLocalId());
			IsNotNull condition=new IsNotNull(targetColumnReference);
			orArguments.add(condition);
		}

		//Checking number of found or parameters
		Expression toApplyCondition=null;
		if(orArguments.size()==0) log.debug("The table hasn't codenames columns");
		else {
			if (orArguments.size()==1) toApplyCondition=orArguments.get(0);
			else toApplyCondition=new Or(orArguments);
			try{
				WorkerWrapper wrapper=new WorkerWrapper(validateDataWithExpressionFactory);
				HashMap<String,Object> map=new HashMap<String,Object>();
				map.put(validateDataWithExpressionFactory.EXPRESSION_PARAMETER.getIdentifier(), toApplyCondition);
				WorkerStatus status=wrapper.execute(targetTable.getId(), null, map);
				processStep(status,wrapper.getResult());
			}catch(InvalidInvocationException e){
				throw new WorkerException("Unable to execute wrapped worker ",e);
			}
		}
	}

	private void checkDuplicates() throws WorkerException{
		try{
			WorkerWrapper wrapper=new WorkerWrapper(duplicateRowsFactory);
			WorkerStatus status=wrapper.execute(targetTable.getId(), null, null);
			processStep(status,wrapper.getResult());
		}catch(InvalidInvocationException e){
			throw new WorkerException("Unable to execute wrapped worker ",e);
		}
	}

	private void checkDuplicatesInAnyNames() throws WorkerException{

	}


	//***************** Instance misc


	private void addColumnValidation(String msg, boolean valid, Column col){

		ColumnLocalId id=col.getLocalId();
		if(!columnValidations.containsKey(id)){
			columnValidations.put(id, getToEnrichColumnMetaValidations((targetTable.getColumnById(id))));
		}
		columnValidations.get(id).add(new Validation(msg,valid));

	}



	private Table createMetaValidatedTable(){		
		TableMetaCreator tmc =cubeManager.modifyTableMeta(targetTable.getId());		
		tmc.setTableMetadata(new ValidationsMetadata(toSetValidations));
		for(Entry<ColumnLocalId,List<Validation>> entry:columnValidations.entrySet()){
			ValidationsMetadata columnValidMeta=new ValidationsMetadata(entry.getValue());			
			tmc.setColumnMetadata(entry.getKey(), columnValidMeta);
		}
		return tmc.create();
	}

	/**
	 * Updates operating table depending on wrapped worker
	 * 
	 */

	private void processStep(WorkerStatus status, WorkerResult result)throws WorkerException{
		if(!status.equals(WorkerStatus.SUCCEDED))
			throw new WorkerException("Wrapped step has failed, see previous log");	
		log.debug("Sub Worker Succeded. Switching working table from "+targetTable+" to "+result.getResultTable());
		targetTable=result.getResultTable();		
	}


	//************* Static misc

	private static List<Validation> getToEnrichTableMetaValidations(Table table){
		List<Validation> foundValidations=new ArrayList<Validation>();		
		ValidationsMetadata validationsMetadata = table.getMetadata(ValidationsMetadata.class);
		if (validationsMetadata!=null)	
			foundValidations.addAll(validationsMetadata.getValidations());			
		else
			log.debug("No validation metadata found, returned empty List");
		
		return foundValidations;
	}	

	private static List<Validation> getToEnrichColumnMetaValidations(Column column){
		List<Validation> foundValidations=new ArrayList<Validation>();		
		ValidationsMetadata validationsMetadata = column.getMetadata(ValidationsMetadata.class);
		if ( validationsMetadata !=null){
			foundValidations.addAll(validationsMetadata.getValidations());			
		} else log.debug("No validation metadata found, returned empty List");
		return foundValidations;
	}

	private static boolean checkLocaleAndLabels(Column col){
		try {
			col.getMetadata(DataLocaleMetadata.class);
			return !col.getMetadata(NamesMetadata.class).getTexts().isEmpty();
		} catch (Exception e) {
			return false;
		}
	}

}
