package org.gcube.data.analysis.tabulardata.operation.validation;

import java.util.ArrayList;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableMetaCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DuplicateRowValidator extends Worker {
	
	private static final Logger log = LoggerFactory.getLogger(DuplicateRowValidator.class);

	CubeManager cubeManager;

	DatabaseConnectionProvider connectionProvider;

	Table targetTable;

	Column validationColumn;

	List<Column> toCheckColumns=null;
	
	public DuplicateRowValidator(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		retrieveTargetTable();
		updateProgress(0.2f);
		createNewTableWithValidationColumn();
		updateProgress(0.4f);
		fillNewTableWithData();
		updateProgress(0.8f);
		evaluateValidityAndUpdateTableMeta();
		return new ImmutableWorkerResult(targetTable);
	}

	private void evaluateValidityAndUpdateTableMeta() throws WorkerException {
		boolean valid = ValidationHelper.evaluateValidationColumnValidity(connectionProvider, targetTable.getName(), validationColumn.getName());
		DataValidationMetadata validationMetadata = createDataValidationMetadata(valid);
		targetTable = cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), validationMetadata).create();
	}

	private void fillNewTableWithData() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, createSetAllTrueSQL(), createSetDuplicateASFalse());
		} catch (Exception e) {
			String msg = "Unable to perform SQL operation";
			log.error(msg,e);
			throw new WorkerException(msg);
		}
	}

	private String createSetAllTrueSQL() {
		return String.format("UPDATE %1$s as newtable SET %2$s = true;",targetTable.getName(), validationColumn.getName());
	}

	private String createSetDuplicateASFalse() {
		
//		
//		String columnNamesSnippet = getColumnNamesSnippet(toCheckColumns);
//		String selectDuplicateTuples = String.format(
//				"SELECT %1$s FROM ( SELECT %1$s, COUNT(*) AS tuples FROM %2$s GROUP BY %1$s) AS temp	WHERE tuples > 1",
//				columnNamesSnippet, targetTable.getName());
//		return String.format("UPDATE %1$s as newtable SET %2$s = false WHERE (%3$s) IN (%4$s);",
//				targetTable.getName(), validationColumn.getName(), columnNamesSnippet, selectDuplicateTuples);
		
		String columnNamesSnippet = getColumnNamesSnippet(toCheckColumns);
		return String.format("WITH duplicates AS (" +
				"Select %1$s from %2$s group by %1$s  having count(*)>1)," +
				"firsts AS (" +
				"Select distinct(first_value(%2$s .id) over (partition by %1$s )) AS id  from %2$s  where (%1$s)  IN (Select %1$s  from duplicates))" +
				"Update %2$s  SET %3$s = false WHERE id not in (SELECT id from firsts) AND (%1$s) in (SELECT %1$s  from duplicates)",columnNamesSnippet,targetTable.getName(),validationColumn.getName());
		
	}

	private String getColumnNamesSnippet(List<Column> columnsToCheck) {
		StringBuilder columnNamesSnippet = new StringBuilder();
		for (Column column : columnsToCheck) {
			columnNamesSnippet.append(column.getName() + ", ");
		}
		columnNamesSnippet.delete(columnNamesSnippet.length() - 2, columnNamesSnippet.length() - 1);
		return columnNamesSnippet.toString();
	}

	private void createNewTableWithValidationColumn() {
		DataValidationMetadata dataValidationMetadata = createDataValidationMetadata(false);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText("Not duplicate"),dataValidationMetadata);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);
		TableMetaCreator creator=cubeManager.modifyTableMeta(targetTable.getId());
		for(Column col:this.toCheckColumns){
			
			ValidationReferencesMetadata referencesMeta;
			if (col.contains(ValidationReferencesMetadata.class)){
				referencesMeta = col.getMetadata(ValidationReferencesMetadata.class);
				referencesMeta.add(validationColumn.getLocalId());
			} else referencesMeta = new ValidationReferencesMetadata(validationColumn.getLocalId());
			
			creator.setColumnMetadata(col.getLocalId(), referencesMeta);
		}
		targetTable=creator.create();
	}

	private DataValidationMetadata createDataValidationMetadata(boolean valid) {
		return new DataValidationMetadata(new ImmutableLocalizedText("True when the tuple is not a duplicate, false otherwise"),valid);
	}

	@SuppressWarnings("unchecked")
	private void retrieveTargetTable() {
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		if(getSourceInvocation().getParameterInstances().containsKey(DuplicateRowValidatorFactory.KEY.getIdentifier())){
			this.toCheckColumns=new ArrayList<Column>();
			Object colParam=getSourceInvocation().getParameterInstances().get(DuplicateRowValidatorFactory.KEY.getIdentifier());
			if(colParam instanceof Iterable<?>){
				for(ColumnReference col:(Iterable<ColumnReference>)colParam)
					this.toCheckColumns.add(targetTable.getColumnById(col.getColumnId()));
			}else{
				this.toCheckColumns.add(targetTable.getColumnById(((ColumnReference)colParam).getColumnId()));
			}			
		}else this.toCheckColumns=targetTable.getColumnsExceptTypes(IdColumnType.class, ValidationColumnType.class);
	System.out.println(this.toCheckColumns);
	}

}
