package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.SQLException;
import java.util.Map;
import java.util.Map.Entry;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GlobalDataValidationReportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.EmptyType;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ValidationWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidateAmbiguousReference extends ValidationWorker {

	private static Logger logger = LoggerFactory.getLogger(ValidateAmbiguousReference.class);
	
	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private Table targetTable;

	private Column targetColumn;

	private Column validationColumn;

	private Table externalTable ;

	private Column externalColumn ;

	private Map<Long, Long> mapping;

	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;

	private static Logger log = LoggerFactory.getLogger(ValidateAmbiguousReference.class);

	public ValidateAmbiguousReference(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider, SQLExpressionEvaluatorFactory sqlEvaluatorFactory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.sqlEvaluatorFactory = sqlEvaluatorFactory;
	}

	@Override
	protected EmptyType execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.1f,"Creating validation structure");
		addValidationColumn();
		updateProgress(0.4f,"Filling validation column");
		fillValidationColumn();
		updateProgress(0.7f,"Finalizing validation");
		evaluateValidityAndUpdateTableMeta();
		return EmptyType.instance();
	}

	@SuppressWarnings({ "unchecked" })
	private void retrieveParameters(){
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
		log.debug("targetColumn is "+targetColumn);
		mapping = (Map<Long, Long>)getSourceInvocation().getParameterInstances().get(ValidateAmbiguousReferenceFactory.MAPPING_PARAMETER.getIdentifier());
		ColumnReference externalReference = (ColumnReference) getSourceInvocation().getParameterInstances().get(ValidateAmbiguousReferenceFactory.TARGET_COLUMN_PARAMETER.getIdentifier());
		externalTable = cubeManager.getTable(externalReference.getTableId());
		externalColumn = externalTable.getColumnById(externalReference.getColumnId());
	}

	private void evaluateValidityAndUpdateTableMeta() throws WorkerException {
		try{
			int invalidCount=ValidationHelper.getErrorCount(connectionProvider, targetTable, validationColumn, sqlEvaluatorFactory);
			GlobalDataValidationReportMetadata globalMeta=ValidationHelper.createDataValidationReport(validationColumn);
			DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText("Ambiguous values on external reference validation"),invalidCount);
			targetTable = cubeManager.modifyTableMeta(targetTable.getId())
					.setColumnMetadata(validationColumn.getLocalId(), 
							validationMeta).
							setTableMetadata(globalMeta).create();
		}catch(Exception e){
			throw new WorkerException("Unable to evaluate global validation",e);
		}
	}

	private String generateSetAllFalseSqlCommand() {
		return String.format("UPDATE %s SET %s = true;", targetTable.getName(), validationColumn.getName());
	}
	
	private void fillValidationColumn() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, generateSetAllFalseSqlCommand(),
					generateValidationSqlCommand());
		} catch (SQLException e) {
			logger.error("error filling validation column",e.getNextException());
			throw new WorkerException("Error occurred while executing SQL command", e.getNextException());
		}
	}
	
	private String generateValidationSqlCommand() throws WorkerException {
		try{
			StringBuffer mappingFrom = new StringBuffer();
			String mappingWhere = "";
			if (mapping!=null && !mapping.isEmpty()){		
				mappingFrom.append("(VALUES");
				for (Entry<Long,Long> mapEntry : mapping.entrySet())
					mappingFrom.append(" (").append(mapEntry.getKey()).append(",").append(mapEntry.getValue()).append("),"); 
				mappingFrom.deleteCharAt(mappingFrom.lastIndexOf(","));
				mappingFrom.append(") AS mapping (key, value), ");
				
				mappingWhere=" OR ( count>1 AND id = mapping.key AND ARRAY[mapping.value] <@ (reference.ids) )";
			}
			
			String query = String.format("UPDATE %1$s SET %2$s = false FROM %3$s (SELECT %4$s as value , array_agg(id) AS ids,  count(*) AS count FROM %5$s GROUP BY %4$s) AS reference WHERE " +
					" (reference.value = %6$s AND NOT(count=1  %7$s )) ", targetTable.getName(), validationColumn.getName(), mappingFrom.toString(), externalColumn.getName(), 
					externalTable.getName(), targetColumn.getName(), mappingWhere ) ;
			
			return query;
						
		}catch(Exception e){
			log.error("error updating validation column",e);
			throw new WorkerException("error updating validation column",e);
		}
	}

	private void addValidationColumn() throws WorkerException {	

		DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText("Ambiguous values on external reference validation"),0);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText("Ambiguous values"),validationMeta);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);	
		log.debug("Added validation column:\n" + targetTable);
		ValidationReferencesMetadata meta=new ValidationReferencesMetadata(targetColumn);
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), meta).create();
	}

}
