package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.SQLException;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GlobalDataValidationReportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.datatype.TypeTransitionSQLHandler;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.EmptyType;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ValidityResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ValidationWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ColumnTypeCastValidator extends ValidationWorker {

	private static final Logger log = LoggerFactory.getLogger(ColumnTypeCastValidator.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;
	private SQLExpressionEvaluatorFactory evaluatorFactory;

	private Table targetTable;

	private Column targetColumn;

	private DataType targetType;

	private Column validationColumn;

	public ColumnTypeCastValidator(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,SQLExpressionEvaluatorFactory sqlFactory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evaluatorFactory=sqlFactory;
	}

	@Override
	protected ValidityResult execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.1f,"Configuring validation");
		addValidationColumnToTable();
		updateProgress(0.4f,"Validating rows");
		fillNewTableWithData();
		updateProgress(0.8f,"Evaluating result");
		return new ValidityResult(createUpdatedTableMeta()==0);
	}

	private void retrieveParameters() {
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
		targetType = OperationHelper.getParameter(ColumnTypeCastValidatorFactory.TARGET_TYPE_PARAMETER,
				getSourceInvocation());
	}

	private void addValidationColumnToTable() {
		createValidationColumn();
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);

		ValidationReferencesMetadata referenceMeta=new ValidationReferencesMetadata(targetColumn);
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), referenceMeta).create();

	}

	private void createValidationColumn() {
		String columnLabel = OperationHelper.retrieveColumnLabel(targetColumn);
		DataValidationMetadata validationMeta = createDataValidationMetadata(0);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText(String.format("Is %s a valid %s?", columnLabel, targetType.getName())),validationMeta);
	}

	private DataValidationMetadata createDataValidationMetadata(int count) {
		String columnLabel = OperationHelper.retrieveColumnLabel(targetColumn);
		LocalizedText validationText = new ImmutableLocalizedText(String.format(
				"Tells whether %s can be casted to a %s value", columnLabel, targetType.getName()));
		DataValidationMetadata validationMeta = new DataValidationMetadata(validationText, count);
		return validationMeta;
	}


	private void fillNewTableWithData() throws WorkerException {
		String sqlCommand = generateFillTableSQL();
		try {
			SQLHelper.executeSQLCommand(sqlCommand, connectionProvider);
		} catch (SQLException e) {
			throw new WorkerException("Unable to evaluate validation result", e);
		}
	}

	private String generateFillTableSQL() {
		TypeTransitionSQLHandler typeTransitionHandler = TypeTransitionSQLHandler.getHandler(
				targetColumn.getDataType(), targetType);
		log.debug("Using Type transition handler: " + typeTransitionHandler.getClass().getSimpleName());
		return typeTransitionHandler.getFillValidationColumnSQLCommand(targetTable, validationColumn, targetColumn);
	}

	private int createUpdatedTableMeta() throws WorkerException {
		try{
			int invalidCount=ValidationHelper.getErrorCount(connectionProvider, targetTable, validationColumn, evaluatorFactory);
			GlobalDataValidationReportMetadata globalMeta=ValidationHelper.createDataValidationReport(validationColumn);

			targetTable = cubeManager.modifyTableMeta(targetTable.getId())
					.setColumnMetadata(validationColumn.getLocalId(), 
							createDataValidationMetadata(invalidCount)).
							setTableMetadata(globalMeta).create();
			return invalidCount;
		}catch(Exception e){
			throw new WorkerException("Unable to evaluate global validation",e);
		}
	}

}
