package org.gcube.data.analysis.tabulardata.operation.validation;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GlobalDataValidationReportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.EmptyType;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ValidityResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ValidationWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DuplicateValuesInColumnValidator extends ValidationWorker{

	private static final Logger log = LoggerFactory.getLogger(DuplicateValuesInColumnValidator.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private SQLExpressionEvaluatorFactory evaluatorFactory;

	private Table targetTable;

	private Column targetColumn;

	private Column validationColumn;


	public DuplicateValuesInColumnValidator(
			OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,SQLExpressionEvaluatorFactory factory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evaluatorFactory=factory;
	}

	@Override
	protected ValidityResult execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.1f,"Configuring validationg");
		createNewTableWithValidationColumn();
		updateProgress(0.5f,"Validating rows");
		fillNewTableWithData();
		updateProgress(0.8f,"Evaluating result");
		return new ValidityResult(evaluateValidityAndUpdateTableMeta()==0);
	}

	private void retrieveParameters() {		
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
	}


	private void createNewTableWithValidationColumn(){
		DataValidationMetadata dataValidationMetadata = createDataValidationMetadata(0);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText("Unique "+OperationHelper.retrieveColumnLabel(targetColumn)),dataValidationMetadata);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);
		ValidationReferencesMetadata refreferenceMeta=new ValidationReferencesMetadata(targetColumn);
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), refreferenceMeta).create();
	}

	private DataValidationMetadata createDataValidationMetadata(int count) {
		return new DataValidationMetadata(new ImmutableLocalizedText("True when the value in column "+OperationHelper.retrieveColumnLabel(targetColumn)+" is not a duplicate, false otherwise"),count);
	}

	private void fillNewTableWithData() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider,createSetAllTrueSQL(),createSetFalseOnDuplicatesSQL());
		} catch (Exception e) {
			String msg = "Unable to perform SQL operation";
			log.error(msg,e);
			throw new WorkerException(msg);
		}
	}

	private String createSetAllTrueSQL() {
		return String.format("UPDATE %1$s as newtable SET %2$s = true;",targetTable.getName(), validationColumn.getName());
	}

	private String createSetFalseOnDuplicatesSQL(){
		return String.format("UPDATE %2$s AS target SET %3$s=false WHERE  " +
				" id NOT IN (SELECT distinct min(id) from " +
				"%2$s GROUP BY %1$s) ", targetColumn.getName(), targetTable.getName(), validationColumn.getName());
	}

	private int evaluateValidityAndUpdateTableMeta() throws WorkerException {
		try{
			int invalidCount=ValidationHelper.getErrorCount(connectionProvider, targetTable, validationColumn, evaluatorFactory);
			GlobalDataValidationReportMetadata globalMeta=ValidationHelper.createDataValidationReport(validationColumn);

			targetTable = cubeManager.modifyTableMeta(targetTable.getId())
					.setColumnMetadata(validationColumn.getLocalId(), 
							createDataValidationMetadata(invalidCount)).
							setTableMetadata(globalMeta).create();
			return invalidCount;
		}catch(Exception e){
			throw new WorkerException("Unable to evaluate global validation",e);
		}
	}
}
