package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.SQLException;
import java.util.Map;
import java.util.Map.Entry;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GlobalDataValidationReportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ValidityResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ValidationWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidateAmbiguousReference extends ValidationWorker {

	private static Logger logger = LoggerFactory.getLogger(ValidateAmbiguousReference.class);
	
	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private Table targetTable;

	private Column targetColumn;

	private Column validationColumn;

	private Table externalTable ;

	private Column externalColumn ;

	private Map<Long, Long> mapping;

	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;

	private static Logger log = LoggerFactory.getLogger(ValidateAmbiguousReference.class);

	public ValidateAmbiguousReference(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider, SQLExpressionEvaluatorFactory sqlEvaluatorFactory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.sqlEvaluatorFactory = sqlEvaluatorFactory;
	}

	@Override
	protected ValidityResult execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.1f,"Creating validation structure");
		addValidationColumn();
		updateProgress(0.4f,"Filling validation column");
		fillValidationColumn();
		updateProgress(0.7f,"Finalizing validation");
		return new ValidityResult(evaluateValidityAndUpdateTableMeta()==0);
	}

	@SuppressWarnings({ "unchecked" })
	private void retrieveParameters(){
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
		log.debug("targetColumn is "+targetColumn);
		mapping = (Map<Long, Long>)getSourceInvocation().getParameterInstances().get(ValidateAmbiguousReferenceFactory.MAPPING_PARAMETER.getIdentifier());
		ColumnReference externalReference = (ColumnReference) getSourceInvocation().getParameterInstances().get(ValidateAmbiguousReferenceFactory.TARGET_COLUMN_PARAMETER.getIdentifier());
		externalTable = cubeManager.getTable(externalReference.getTableId());
		externalColumn = externalTable.getColumnById(externalReference.getColumnId());
	}

	private int evaluateValidityAndUpdateTableMeta() throws WorkerException {
		try{
			int invalidCount=ValidationHelper.getErrorCount(connectionProvider, targetTable, validationColumn, sqlEvaluatorFactory);
			GlobalDataValidationReportMetadata globalMeta=ValidationHelper.createDataValidationReport(validationColumn);
			DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText("Ambiguous values on external reference validation"),invalidCount);
			targetTable = cubeManager.modifyTableMeta(targetTable.getId())
					.setColumnMetadata(validationColumn.getLocalId(), 
							validationMeta).
							setTableMetadata(globalMeta).create();
			return invalidCount;
		}catch(Exception e){
			throw new WorkerException("Unable to evaluate global validation",e);
		}
	}

	private String generateSetAllFalseSqlCommand() {
		return String.format("UPDATE %s SET %s = true;", targetTable.getName(), validationColumn.getName());
	}
	
	private void fillValidationColumn() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, generateSetAllFalseSqlCommand(),
					generateValidationSqlCommand());
		} catch (SQLException e) {
			logger.error("error filling validation column",e.getNextException());
			System.out.println("----------------------------");e.printStackTrace();
			throw new WorkerException("Error occurred while executing SQL command", e.getNextException());
		}
		
	}
	
	private String generateValidationSqlCommand() throws WorkerException {
		try{
			StringBuffer mappingFrom = new StringBuffer();
			String query;
			if (mapping!=null && !mapping.isEmpty()){
				
				mappingFrom.append("(VALUES");
				for (Entry<Long,Long> mapEntry : mapping.entrySet())
					mappingFrom.append(" (").append(mapEntry.getKey()).append(",").append(mapEntry.getValue()).append("),"); 
				mappingFrom.deleteCharAt(mappingFrom.lastIndexOf(","));
				mappingFrom.append(") AS mapping (key, value) ");
			
				String referenceQuery = String.format(" (SELECT %1$s as value , array_agg(id) AS ids,  count(*) AS count FROM %2$s GROUP BY %1$s) AS reference", 
						externalColumn.getName(), externalTable.getName() );
				
				query = String.format("  UPDATE %1$s SET %2$s = false FROM ( %1$s as alias LEFT JOIN %3$s ON reference.value = alias.%4$s ) as l LEFT JOIN %5$s ON l.id= mapping.key " +
						" WHERE l.id=%1$s.id AND (key is null OR NOT(ARRAY[mapping.value] <@ (ids)))  and count>1 ", 
						targetTable.getName(), validationColumn.getName(), referenceQuery, targetColumn.getName(), mappingFrom.toString());
			} else 
				query = String.format("UPDATE %1$s SET %2$s = false FROM (SELECT %3$s as value , array_agg(id) AS ids,  count(*) AS count FROM %4$s GROUP BY %3$s) AS reference WHERE " +
					" (reference.value = %5$s AND count>1 ) ", targetTable.getName(), validationColumn.getName(), externalColumn.getName(), 
					externalTable.getName(), targetColumn.getName() ) ;
			return query;
						
		}catch(Exception e){
			log.error("error updating validation column",e);
			throw new WorkerException("error updating validation column",e);
		}
	}

	private void addValidationColumn() throws WorkerException {	

		DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText("Ambiguous values on external reference validation"),0);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText("Ambiguous values"),validationMeta);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);	
		log.debug("Added validation column:\n" + targetTable);
		ValidationReferencesMetadata meta=new ValidationReferencesMetadata(targetColumn);
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), meta).create();
	}

}
