package org.gcube.data.harmonization.occurrence.services;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.queries.GCUBERuntimeResourceQuery;
import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.data.harmonization.occurrence.db.model.ImportReference;
import org.gcube.data.harmonization.occurrence.db.model.MergeReference;
import org.gcube.data.harmonization.occurrence.db.model.ToMergeImported;
import org.gcube.data.harmonization.occurrence.operating.ImportHandler;
import org.gcube.data.harmonization.occurrence.operating.MergeHandler;
import org.gcube.data.harmonization.occurrence.operating.OperatingModule;
import org.gcube.data.harmonization.occurrence.operating.OperatingModuleConfiguration;
import org.gcube.data.harmonization.occurrence.util.Constants;

import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.dao.DaoManager;
import com.j256.ormlite.jdbc.JdbcConnectionSource;
import com.j256.ormlite.jdbc.JdbcPooledConnectionSource;
import com.j256.ormlite.table.TableUtils;


public class ServiceContext extends GCUBEServiceContext {

	/** Single context instance, created eagerly */
	private static ServiceContext cache = new ServiceContext();

	/** Returns cached instance */
	public static ServiceContext getContext() {return cache;}

	/** Prevents accidental creation of more instances */
	private ServiceContext(){};

	/** {@inheritDoc} */
	protected String getJNDIName() {return Constants.JNDIName;}

	
	public static enum FOLDERS{
		IMPORTED
	}

	private Properties props;


	private OperatingModule<ImportReference,ImportHandler> importModule;
	private OperatingModule<MergeReference,MergeHandler> mergeModule;

	private JdbcConnectionSource connectionSource;

	private Dao<ImportReference,String> importDao=null;
	private Dao<MergeReference,String> mergeDao=null;
	private Dao<ToMergeImported,Integer> toMergeDao=null;





	@Override
	protected void onInitialisation() throws Exception {
		//Read Configuration
		InputStream is=null;
		try{
			props=new Properties();
			is= new FileInputStream (this.getFile("config.properties", false));
			props.load(is);			
			is.close ();
			logger.debug("loaded properties file :" +props.toString());
		}catch(Exception e){
			logger.fatal("Unable to read configuration file",e);
			throw e;
		}finally{is.close();}	


		//Init Database
		String DBconnectionString=null;
		try{
			if(Boolean.parseBoolean(props.getProperty(Constants.USE_EMBEDDED_DATABASE)))			
				DBconnectionString="jdbc:h2:mem:occurrence";
			else{
				logger.trace("Looking for database runtime resource...");
				ISClient isClient=GHNContext.getImplementation(ISClient.class);
				GCUBERuntimeResourceQuery runtimeQuery=isClient.getQuery(GCUBERuntimeResourceQuery.class);
				runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Category", props.getProperty(Constants.INTERNAL_DB_CATEGORY)));
				runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Platform/Name", props.getProperty(Constants.INTERNAL_DB_PLATFORM)));
				boolean found=false;
				for(GCUBEScope scope:getStartScopes()){
					List<GCUBERuntimeResource> list=isClient.execute(runtimeQuery, scope);
					logger.debug("Found "+list.size()+" matching resources under scope "+scope);
					for(GCUBERuntimeResource rr:list){
						if(found) break;
						try{
							for(AccessPoint access:rr.getAccessPoints()){
								if(access.getEntryname().equals(props.getProperty(Constants.INTERNAL_DB_ENTRYPOINT))&&
										access.getAllPropertyNames().contains(props.getProperty(Constants.INTERNAL_DB_SCHEMA_PROPERTY))&&
										access.getProperty(props.getProperty(Constants.INTERNAL_DB_SCHEMA_PROPERTY)).equals(props.getProperty(Constants.INTERNAL_DB_SCHEMA_VALUE))){
									logger.debug("Found Access point "+access.getEntryname()+" in "+rr.getName()+" [ ID : "+rr.getID()+"]");
									DBconnectionString="jdbc:postgres:"+access.getEndpoint()+";user="+access.getUsername()+";password="+access.getPassword();
									found=true;
								}
							}
						}catch(Exception e){logger.warn("Unable to parse resource [ID :"+rr.getID()+"]",e);}
					}
					if(found) break;
				}					
			}
			connectionSource=new JdbcPooledConnectionSource(DBconnectionString);
		}catch(Exception e){
			logger.fatal("Unable to connect to "+DBconnectionString,e);
			throw e;
		}


		try{
			importDao=DaoManager.createDao(connectionSource, ImportReference.class);			
			mergeDao=DaoManager.createDao(connectionSource, MergeReference.class);
			toMergeDao=DaoManager.createDao(connectionSource, ToMergeImported.class);


			TableUtils.createTableIfNotExists(connectionSource, ImportReference.class);			
			TableUtils.createTableIfNotExists(connectionSource, MergeReference.class);
			TableUtils.createTableIfNotExists(connectionSource, ToMergeImported.class);
		}catch(Exception e){
			logger.fatal("Unable to init Database", e);
			throw e;
		}

		
		String queueConnectionHost="localhost";
		
		
		try{
			OperatingModuleConfiguration<ImportReference, ImportHandler> importConfiguration=new OperatingModuleConfiguration<ImportReference, ImportHandler>();
			importConfiguration.setDao(importDao);
			importConfiguration.setHanlderClass(ImportHandler.class);
			importConfiguration.setMaxWorkerThreads(Integer.parseInt(props.getProperty(Constants.IMPORTER_MAX_WORKER)));
			importConfiguration.setMessageRoutingKey("org.gcube.data.harmonization.occurrence.import");
			importConfiguration.setQueueConnectionHost(queueConnectionHost);
			importModule=new OperatingModule<ImportReference,ImportHandler>();
			importModule.init(importConfiguration);
		}catch(Exception e){
			logger.fatal("Unable to init Import Module",e);
			throw e;
		}

		try{
			OperatingModuleConfiguration<MergeReference, MergeHandler> mergeConfiguration=new OperatingModuleConfiguration<MergeReference, MergeHandler>();
			mergeConfiguration.setDao(mergeDao);
			mergeConfiguration.setHanlderClass(MergeHandler.class);
			mergeConfiguration.setMaxWorkerThreads(Integer.parseInt(props.getProperty(Constants.MERGE_MAX_WORKER)));
			mergeConfiguration.setMessageRoutingKey("org.gcube.data.harmonization.occurrence.merge");
			mergeConfiguration.setQueueConnectionHost(queueConnectionHost);
			mergeModule=new OperatingModule<MergeReference,MergeHandler>();
			mergeModule.init(mergeConfiguration);
		}catch(Exception e){
			logger.fatal("Unable to init Merge Module",e);
			throw e;
		}
	}

	public Dao<ImportReference, String> getImportDao() {
		return importDao;
	}
	public OperatingModule<ImportReference,ImportHandler> getImportModule() {
		return importModule;
	}
	public Dao<MergeReference, String> getMergeDao(){
		return mergeDao;
	}
	public OperatingModule<MergeReference,MergeHandler> getMergeModule() {
		return mergeModule;
	}

	public Dao<ToMergeImported, Integer> getToMergeDao() {
		return toMergeDao;
	}
	
	public String getFolderPath(FOLDERS folderName){
		String persistencePath = ServiceContext.getContext().getPersistenceRoot().getAbsolutePath()+File.separator+folderName;
		File f=new File(persistencePath);
		if(!f.exists()){
			logger.debug("Creating persistence folder "+persistencePath);
			f.mkdirs();
			try {
				Process proc=Runtime.getRuntime().exec("chmod -R 777 "+persistencePath);
				try{
					proc.waitFor();
				}catch(InterruptedException e){
					int exitValue=proc.exitValue();
					logger.debug("Permission execution exit value = "+exitValue);
				}
			} catch (IOException e) {
				logger.warn("Unexpected Exception", e);
			}
		}
		return persistencePath;
	}
}
