package org.gcube.datapublishing.sdmx.impl.repository;

import static org.gcube.datapublishing.sdmx.util.DatasourceUtil.parseReferenceString;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.ArrayList;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient.Detail;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient.References;
import org.gcube.datapublishing.sdmx.api.repository.GCubeSDMXDataRepository;
import org.gcube.datapublishing.sdmx.api.repository.dao.RegistrationDAO;
import org.gcube.datapublishing.sdmx.impl.data.TimeseriesRegistration;
import org.gcube.datapublishing.sdmx.impl.exceptions.SDMXRegistryClientException;
import org.gcube.datapublishing.sdmx.impl.repository.dao.Registration;
import org.sdmxsource.sdmx.api.constants.SDMX_ERROR_CODE;
import org.sdmxsource.sdmx.api.engine.DataReaderEngine;
import org.sdmxsource.sdmx.api.exception.SdmxException;
import org.sdmxsource.sdmx.api.model.beans.SdmxBeans;
import org.sdmxsource.sdmx.api.model.beans.datastructure.DataStructureBean;
import org.sdmxsource.sdmx.api.model.beans.reference.MaintainableRefBean;
import org.sdmxsource.sdmx.api.util.ReadableDataLocation;
import org.sdmxsource.sdmx.dataparser.manager.DataReaderManager;
import org.sdmxsource.util.io.ReadableDataLocationTmp;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;

@Slf4j
public class GCubeSDMXDataRepositoryImpl implements GCubeSDMXDataRepository {

	private RegistrationDAO registrationDAO;

	private DataReaderManager dataReaderManager;

	private SDMXRegistryClient registryClient;

	@Autowired
	public GCubeSDMXDataRepositoryImpl(RegistrationDAO registrationDAO,
			DataReaderManager dataReaderManager,
			SDMXRegistryClient registryClient) {
		super();
		this.registrationDAO = registrationDAO;
		this.dataReaderManager = dataReaderManager;
		this.registryClient = registryClient;
	}

	

	@Override
	public InputStream dataQuery(String flowRef, String key,
			String providerRef, String startPeriod, String endPeriod,
			String updatedAfter, String firstNObservations,
			String lastNObservations, String dimensionAtObservation,
			String detail) throws SdmxException {
		// TODO Replace with real implementation. this work only in test

		// Recover datastructure given the flowref
		MaintainableRefBean ref;
		try {
			ref = parseReferenceString(flowRef);
		} catch (Exception e1) {
			log.error("Unable to parse flow reference string");
			throw new SdmxException("Unable to parse flowRef",
					SDMX_ERROR_CODE.SYNTAX_ERROR);
		}

		SdmxBeans beans;
		try {
			beans = registryClient.getDataFlow(ref.getAgencyId(),
					ref.getMaintainableId(), ref.getVersion(), Detail.full,
					References.children);
		} catch (SDMXRegistryClientException e) {
			log.error("Caught SDMXRegistryClientException",e);
			throw new SdmxException(e.getMessage(), SDMX_ERROR_CODE.INTERNAL_SERVER_ERRROR);
		}
		if (beans.getDataflows().isEmpty())
			throw new SdmxException("Unable to find dataflow",
					SDMX_ERROR_CODE.NO_RESULTS_FOUND);

		if (beans.getDataStructures().isEmpty())
			throw new SdmxException("Unable to find datastructures",
					SDMX_ERROR_CODE.NO_RESULTS_FOUND);

		DataStructureBean ds = beans.getDataStructures().iterator().next();

		ReadableDataLocation rdl = new ReadableDataLocationTmp(
				"src/test/resources/Repository/Data/FAO/CAPTURE-lessdata.xml");
		final DataReaderEngine dre = dataReaderManager.getDataReaderEngine(rdl,
				ds);

		PipedInputStream pis = new PipedInputStream();
		final PipedOutputStream pos;
		try {
			pos = new PipedOutputStream(pis);
		} catch (IOException e) {
			log.error("Unable to create data output stream", e);
			throw new SdmxException("Unable to create data output stream",
					SDMX_ERROR_CODE.INTERNAL_SERVER_ERRROR);
		}
		new Thread(new Runnable() {
			@Override
			public void run() {
				log.debug("Writing data to output stream");
				dre.copyToOutputStream(pos);
			}
		}).start();

		return pis;
	}

	@Override
	public InputStream metadataQuery(String flowRef, String key,
			String providerRef, String startPeriod, String endPeriod,
			String updatedAfter) throws SdmxException {
		throw new SdmxException("Not Implemented",
				SDMX_ERROR_CODE.NOT_IMPLEMENTED);
	}

	@Override
	public void registerTimeseries(TimeseriesRegistration tsr) {

		Registration r = new TransformTimeseriesRegistration().apply(tsr);

		registrationDAO.insertRegistration(r);

	}

	@Override
	public List<TimeseriesRegistration> getRegistrations() {
		List<Registration> registrations = registrationDAO
				.getAllRegistrations();
		return new ArrayList<TimeseriesRegistration>(Collections2.transform(
				registrations, new TransformRegistration()));
	}

	@Override
	public boolean removeRegistration(String flowAgencyId, String flowId,
			String flowVersion, String providerAgencyId, String providerId) {
		return registrationDAO.removeRegistration(flowAgencyId, flowId, flowVersion, providerAgencyId, providerId);
	}

	private class TransformTimeseriesRegistration implements
			Function<TimeseriesRegistration, Registration> {

		@Override
		public Registration apply(TimeseriesRegistration input) {

			Registration output = new Registration();

			output.setFlowAgencyId(input.getFlowAgencyId());
			output.setFlowId(input.getFlowId());
			output.setFlowVersion(input.getFlowVersion());
			output.setProviderAgencyId(input.getProviderAgencyId());
			output.setProviderId(input.getProviderId());
			output.setTimeseriesScope(input.getTimeseriesScope());
			output.setTimeseriesId(input.getTimeseriesId());
			output.setRegistryScope(input.getRegistryScope());

			return output;
		}
	}

	private class TransformRegistration implements
			Function<Registration, TimeseriesRegistration> {

		@Override
		public TimeseriesRegistration apply(Registration input) {
			TimeseriesRegistration output = new TimeseriesRegistration();

			output.setFlowAgencyId(input.getFlowAgencyId());
			output.setFlowId(input.getFlowId());
			output.setFlowVersion(input.getFlowVersion());
			output.setProviderAgencyId(input.getProviderAgencyId());
			output.setProviderId(input.getProviderId());
			output.setTimeseriesScope(input.getTimeseriesScope());
			output.setTimeseriesId(input.getTimeseriesId());
			output.setRegistryScope(input.getRegistryScope());

			return output;
		}

	}

}
