package org.gcube.datapublishing.sdmx.impl.publisher;

import java.io.InputStream;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.datapublishing.sdmx.api.datasource.GCubeSDMXDatasourceClient;
import org.gcube.datapublishing.sdmx.api.model.GCubeSDMXDatasourceDescriptor;
import org.gcube.datapublishing.sdmx.api.publisher.GCubeSDMXPublisher;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient.Detail;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient.References;
import org.gcube.datapublishing.sdmx.impl.exceptions.SDMXRegistryClientException;
import org.gcube.datapublishing.sdmx.impl.publisher.data.LocalizedText;
import org.gcube.datapublishing.sdmx.impl.publisher.exceptions.PublisherException;
import org.sdmxsource.sdmx.api.constants.TERTIARY_BOOL;
import org.sdmxsource.sdmx.api.model.beans.SdmxBeans;
import org.sdmxsource.sdmx.api.model.beans.base.AgencySchemeBean;
import org.sdmxsource.sdmx.api.model.beans.base.DataProviderSchemeBean;
import org.sdmxsource.sdmx.api.model.beans.codelist.CodelistBean;
import org.sdmxsource.sdmx.api.model.beans.conceptscheme.ConceptSchemeBean;
import org.sdmxsource.sdmx.api.model.beans.datastructure.DataStructureBean;
import org.sdmxsource.sdmx.api.model.beans.datastructure.DataflowBean;
import org.sdmxsource.sdmx.api.model.beans.reference.MaintainableRefBean;
import org.sdmxsource.sdmx.api.model.beans.registry.ProvisionAgreementBean;
import org.sdmxsource.sdmx.api.model.mutable.base.AgencyMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.base.AgencySchemeMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.base.DataSourceMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.registry.RegistrationMutableBean;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.base.AgencyMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.base.DataSourceMutableBeanImpl;
import org.springframework.beans.factory.annotation.Autowired;

@Slf4j
public class GCubeSDMXPublisherImpl implements GCubeSDMXPublisher {

	private SDMXRegistryClient registryClient;

	private GCubeSDMXDatasourceClient datasourceClient;

	private GCubeSDMXDatasourceDescriptor datasourceDescriptor;

	private static String ROOT_AGENCY_SCHEME_FILENAME = "rootAgencyScheme.properties";

	private Properties props = null;

	@Autowired
	public GCubeSDMXPublisherImpl(SDMXRegistryClient registryClient,
			GCubeSDMXDatasourceClient datasourceClient,
			GCubeSDMXDatasourceDescriptor datasourceDescriptor) {
		super();
		this.registryClient = registryClient;
		this.datasourceClient = datasourceClient;
		this.datasourceDescriptor = datasourceDescriptor;
	}

	private void loadAgencyRootSchemeProperties() throws PublisherException {
		if (props != null)
			return;
		try {
			props = new Properties();
			InputStream is = this.getClass().getClassLoader()
					.getResourceAsStream(ROOT_AGENCY_SCHEME_FILENAME);
			log.debug("Properties file InputStream:" + is);
			props.load(is);
		} catch (Exception e) {
			log.error("Unable to find Root Agency Scheme coordinates file ('"
					+ ROOT_AGENCY_SCHEME_FILENAME + "')");
			throw new PublisherException(
					"Unable to find Root Agency Scheme coordinates file");
		}
	}

	@Override
	public void createAgency(String agencyId, List<LocalizedText> names)
			throws PublisherException {

		loadAgencyRootSchemeProperties();

		log.debug("Properties file: " + props);

		String schemaAgencyId = props.getProperty("agencyId");
		String schemaId = props.getProperty("artifactId");
		String schemaVersion = props.getProperty("version");
		log.debug("Retrieved Agency Scheme coordinates from properties file: ["
				+ schemaAgencyId + ", " + schemaId + ", " + schemaVersion + "]");

		SdmxBeans beans;
		try {

			beans = registryClient.getAgencyScheme(schemaAgencyId, schemaId,
					schemaVersion, Detail.full, References.none);
		} catch (SDMXRegistryClientException e) {
			String errorMsg = "Unable to find the root agency scheme";
			log.error(errorMsg, e);
			throw new PublisherException(errorMsg, e);
		}

		Set<AgencySchemeBean> agencies = beans.getAgenciesSchemes();

		if (agencies.size() != 1)
			throw new PublisherException(
					"Invalid number of root agency schemas retrieved from IS");

		AgencySchemeMutableBean rootAgencyScheme = agencies.iterator().next()
				.getMutableInstance();

		for (AgencyMutableBean agency : rootAgencyScheme.getItems()) {
			if (agency.getId().equals(agencyId)) {
				log.debug("Agency with id '" + agencyId + "' already present");
				return;
			}
		}

		if (rootAgencyScheme.getFinalStructure() == TERTIARY_BOOL.TRUE)
			throw new PublisherException(
					"Unable to modify root Agency Scheme, artifact is marked as Final");

		log.debug("Agency with id '" + agencyId
				+ "' not found in root agency schema, creating a new one...");
		AgencyMutableBean newAgency = new AgencyMutableBeanImpl();
		newAgency.setId(agencyId);
		for (LocalizedText localizedText : names) {
			newAgency.addName(localizedText.getLocale(),
					localizedText.getText());
		}

		log.debug("Submitting modified root agency schema");
		rootAgencyScheme.addItem(newAgency);
		try {
			registryClient.publish(rootAgencyScheme.getImmutableInstance());
		} catch (SDMXRegistryClientException e) {
			log.error(
					"SDMXRegistryClientException caught while publishing modified root agency schema",
					e);
			throw new PublisherException(e.getMessage());
		}
	}

	public void publish(CodelistBean codelist) throws PublisherException {
		try {
			registryClient.publish(codelist);
		} catch (SDMXRegistryClientException e) {
			throw new PublisherException(e);
		}
	}

	public void publish(ConceptSchemeBean conceptscheme)
			throws PublisherException {
		try {
			registryClient.publish(conceptscheme);
		} catch (SDMXRegistryClientException e) {
			throw new PublisherException(e);
		}
	}

	public void publish(DataStructureBean datastructure)
			throws PublisherException {
		try {
			registryClient.publish(datastructure);
		} catch (SDMXRegistryClientException e) {
			throw new PublisherException(e);
		}
	}

	public void publish(DataflowBean dataflow) throws PublisherException {
		try {
			registryClient.publish(dataflow);
		} catch (SDMXRegistryClientException e) {
			throw new PublisherException(e);
		}
	}

	public void publish(DataProviderSchemeBean dataproviderscheme)
			throws PublisherException {
		try {
			registryClient.publish(dataproviderscheme);
		} catch (SDMXRegistryClientException e) {
			throw new PublisherException(e);
		}
	}

	public void publish(ProvisionAgreementBean provisionagreement)
			throws PublisherException {
		try {
			registryClient.publish(provisionagreement);
		} catch (SDMXRegistryClientException e) {
			throw new PublisherException(e);
		}
	}

	public void publish(RegistrationMutableBean registration, String timeseriesId,
			String timeseriesServiceScope) throws PublisherException {

		// Register on datasource
		TimeseriesRegistrationData tsrd = getTimeseriesRegistrationData(registration
				.getProvisionAgreementRef().getMaintainableReference());

		String currentScope = ScopeProvider.instance.get();

		try {
			datasourceClient.registerTimeseries(tsrd.getFlowAgencyId(),
					tsrd.getFlowId(), tsrd.flowVersion,
					tsrd.getProviderAgencyId(), tsrd.getProviderId(),
					timeseriesId, currentScope, currentScope);
		} catch (Exception e) {
			String errorMsg = "Unable to register timeseries on sdmx datasource";
			log.error(errorMsg,e);
			throw new PublisherException(errorMsg);
		}

		// Register on registry
		DataSourceMutableBean dataSource = new DataSourceMutableBeanImpl();
		dataSource.setRESTDatasource(true);
		dataSource.setWebServiceDatasource(false);
		dataSource.setSimpleDatasource(false);
		dataSource.setDataUrl(datasourceDescriptor.getRest_url_V2_1());
		
		registration.setDataSource(dataSource);

		try {
			registryClient.publish(registration.getImmutableInstance());
		} catch (SDMXRegistryClientException e) {
			String errorMsg = "Unable to publish registration on sdmx registry";
			log.error(errorMsg,e);
			throw new PublisherException(errorMsg);
		}
	}

	private TimeseriesRegistrationData getTimeseriesRegistrationData(
			MaintainableRefBean provisionAgreementRef) throws PublisherException {
		SdmxBeans beans;
		try {
			beans = registryClient.getProvisionAgreement(
					provisionAgreementRef.getAgencyId(),
					provisionAgreementRef.getMaintainableId(),
					provisionAgreementRef.getVersion(), Detail.full,
					References.none);
		} catch (SDMXRegistryClientException e) {
			String errorMsg = "Unable to retrieve provision agreement from sdmx registry with reference: " + provisionAgreementRef;
			log.error(errorMsg,e);
			throw new PublisherException(errorMsg);
		}

		TimeseriesRegistrationData result = new TimeseriesRegistrationData();

		ProvisionAgreementBean pa = beans.getProvisionAgreements().iterator()
				.next();

		MaintainableRefBean dataflowRef = pa.getStructureUseage().getMaintainableReference(); 
		result.setFlowAgencyId(dataflowRef.getAgencyId());
		result.setFlowId(dataflowRef.getMaintainableId());
		result.setFlowVersion(dataflowRef.getVersion());
		
		result.setProviderAgencyId(pa.getDataproviderRef().getMaintainableReference().getAgencyId());
		result.setProviderId(pa.getDataproviderRef().getIdentifiableIds()[0]);
		
		log.trace("Built: " + result );
		
		return result;

	}

	@Data
	private class TimeseriesRegistrationData {
		String flowAgencyId;
		String flowId;
		String flowVersion;
		String providerAgencyId;
		String providerId;
	}

}
