package org.gcube.datatransfer.common.messaging;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.monitoring.GCUBEMessage;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScopeNotSupportedException;
import org.gcube.datatransfer.common.messaging.TransferMessage;
import org.gcube.messaging.common.producer.ActiveMQClient;
import org.gcube.messaging.common.producer.GCUBELocalProducer;

/** 		
 * 
 * @author Andrea Manzi(CERN)
 *
 */
public class MSGClient {
	

	protected MSGClient() throws IOException, GCUBEScopeNotSupportedException{
		
		if (isClientMode())
			this.startLocalProducer(GCUBEScope.getScope("/"+ GHNContext.getContext()
					.getProperty(GHNContext.INFRASTRUCTURE_NAME, true)));
	}
	
	/**
	 * The method create Messages and send them to the D4science MessageBroker configured in the Infrastructure.
	 *
	 * This method is intended to be executed by Service running on a GHN container.
	 * 
	 * The information about GCUBEScope, ServiceClass and ServiceName are automatically retrieved from the 
	 * ServiceContext. 
	 * 
	 * By default a message for each service scope is generated, otherwise is it possible to specify the GCUBEScope as optional parameter. 
	 * 
	 * 
	 * @param context the Service Context 
	 * @param message the map of parameters to send
	 * @param scope, if the service is running in more than one scope and we want to specify the caller scope, is it possible to use this optional parameters
	 * @throws GHNClientModeException the Exception is thrown when the method is invoked from a client
	 * @throws ReservedFieldException if one or more parameters clash the ones already reserved by the system
	 * @throws IllegalArgumentException if one or more parameters type are not supported by the system
	 * @throws Exception 
	 */
	public void sendMessage(GCUBEServiceContext context,TransferMessage message,
			GCUBEScope ...scope) 	throws GHNClientModeException  ,IllegalArgumentException, Exception {
		
		if (isClientMode()){
			throw new GHNClientModeException("Container is not running, this method is not intented to be used in GHN client mode");
		}
		
		try {
				message.setTimeNow();
			if (scope.length > 0) {
				message.createTopicName(scope[0].getInfrastructure());
				message.setScope(scope[0].getInfrastructure().toString());	
				GCUBELocalProducer.logger.debug(message.toString());
				this.sendMessage(message);
				
			}
			else 
			{//a message  for each scope is sent
				
				for (GCUBEScope sco : context.getInstance().getScopes().values()){

					message.createTopicName(sco.getInfrastructure());
					message.setScope(sco.getInfrastructure().toString());	
					GCUBELocalProducer.logger.debug(message.toString());
					this.sendMessage(message);
				}
			}
				
		}catch (Exception e){
			GCUBELocalProducer.logger.error("Error Sending Transfer message",e);
			e.printStackTrace();
			throw e;
		}
		
	}
	
	
	//The library should understand if the message has to be sent from within a running container or not
	private boolean isClientMode(){
		return GHNContext.getContext().isClientMode();
	}
	
	
	/**
	 * {@inheritDoc}
	 */
	private void sendMessage(GCUBEMessage message) {
		ActiveMQClient.getSingleton().sendMessageToQueue(message);
	}
	
	
	private void startLocalProducer(GCUBEScope scope) throws GCUBEScopeNotSupportedException{
		GCUBELocalProducer local = new GCUBELocalProducer();
		long interval = 1200;
		
		ArrayList<EndpointReferenceType> eprs = new ArrayList<EndpointReferenceType>();
		
		if (scope.getServiceMap().getEndpoints(GHNContext.MSGBROKER)!= null) 
		{
			
			for (EndpointReferenceType msgBrokerEpr:scope.getServiceMap().getEndpoints(GHNContext.MSGBROKER)) {
				eprs.add(msgBrokerEpr);
			}
		}
		
		HashMap<GCUBEScope,ArrayList<EndpointReferenceType>> monitoredScopes = new HashMap<GCUBEScope,ArrayList<EndpointReferenceType>>();
		monitoredScopes.put(scope,eprs);
		
		local.setBrokerMap(monitoredScopes);
		local.setInterval(interval);
		local.run();
		
	}

	
	class GHNClientModeException extends Exception {

		public GHNClientModeException(String string) {
			super (string);
		}

		/**
		 * 
		 */
		private static final long serialVersionUID = 2107643850291448530L;
	};
}
