package org.gcube.data.transfer.library;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import lombok.Synchronized;
import lombok.extern.slf4j.Slf4j;

import org.gcube.data.transfer.library.faults.FailedTransferException;
import org.gcube.data.transfer.library.faults.HostingNodeNotFoundException;
import org.gcube.data.transfer.library.faults.InitializationException;
import org.gcube.data.transfer.library.faults.InvalidSourceException;
import org.gcube.data.transfer.library.faults.ServiceNotFoundException;
import org.gcube.data.transfer.library.faults.SourceNotSetException;
import org.gcube.data.transfer.library.faults.UnreachableNodeException;
import org.gcube.data.transfer.library.transferers.Transferer;
import org.gcube.data.transfer.library.transferers.TransfererBuilder;

@Slf4j
public class DataTransferClient {

	private Transferer transferer=null;
	
	private DataTransferClient(Transferer transferer) {
		this.transferer=transferer;
	}
	
	public static DataTransferClient getInstanceByEndpoint(String endpoint) throws UnreachableNodeException, ServiceNotFoundException{
		log.debug("Getting transferer for endpoint : "+endpoint);
		return new DataTransferClient(TransfererBuilder.getTransfererByHost(endpoint));		
	}
	
	public static DataTransferClient getInstanceByNodeId(String id) throws HostingNodeNotFoundException, UnreachableNodeException, ServiceNotFoundException{
		log.debug("Getting transferer for nodeId : "+id);
		return new DataTransferClient(TransfererBuilder.getTransfererByhostingNodeId(id));		
	}
	
	@Synchronized("transferer")
	public TransferResult localFile(String path) throws InvalidSourceException, SourceNotSetException, FailedTransferException, InitializationException{
		if(transferer==null) throw new RuntimeException("Transferer not set, please set destination before trying to transfer");
		log.debug("Sending local file {} to {}",path,transferer.getDestinationCapabilities().getHostName());
		transferer.localFile(path);
		return transferer.transfer();
	}
	
	@Synchronized("transferer")
	public TransferResult localFile(File file) throws InvalidSourceException, SourceNotSetException, FailedTransferException, InitializationException{
		if(transferer==null) throw new RuntimeException("Transferer not set, please set destination before trying to transfer");
		log.debug("Sending local file {} to {}",file.getAbsolutePath(),transferer.getDestinationCapabilities().getHostName());
		transferer.localFile(file);
		return transferer.transfer();
	}
	
	@Synchronized("transferer")
	public TransferResult httpSource(String url) throws InvalidSourceException, SourceNotSetException, FailedTransferException, InitializationException{
		if(transferer==null) throw new RuntimeException("Transferer not set, please set destination before trying to transfer");
		log.debug("Passed url string : "+url);
		try{
			return this.httpSource(new URL(url));
		}catch(MalformedURLException e){
			throw new InvalidSourceException("Invalid url : "+url);
		}
	}
	
	@Synchronized("transferer")
	public TransferResult httpSource(URL url) throws InvalidSourceException, SourceNotSetException, FailedTransferException, InitializationException{
		if(transferer==null) throw new RuntimeException("Transferer not set, please set destination before trying to transfer");
		log.debug("Sending from url {} to {}",url,transferer.getDestinationCapabilities().getHostName());
		transferer.fromURL(url);
		return transferer.transfer();
	}
	
	@Synchronized("transferer")
	public TransferResult storageId(String id) throws InvalidSourceException, SourceNotSetException, FailedTransferException, InitializationException{
		if(transferer==null) throw new RuntimeException("Transferer not set, please set destination before trying to transfer");
		log.debug("Sending from storage id {} to {}",id,transferer.getDestinationCapabilities().getHostName());
		transferer.storageFileId(id);
		return transferer.transfer();
	}
}
