package org.gcube.datatransfer.scheduler.impl.porttype;

import static org.gcube.datatransfer.agent.library.proxies.Proxies.transferAgent;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.axis.components.uuid.UUIDGen;
import org.apache.axis.components.uuid.UUIDGenFactory;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.datatransfer.agent.library.AgentLibrary;
import org.gcube.datatransfer.agent.library.outcome.FileTransferOutcome;
import org.gcube.datatransfer.scheduler.db.DataTransferDBManager;
import org.gcube.datatransfer.scheduler.db.model.Agent;
import org.gcube.datatransfer.scheduler.db.model.DataSource;
import org.gcube.datatransfer.scheduler.db.model.DataStorage;
import org.gcube.datatransfer.scheduler.db.model.ManuallyScheduled;
import org.gcube.datatransfer.scheduler.db.model.PeriodicallyScheduled;
import org.gcube.datatransfer.scheduler.db.model.Transfer;
import org.gcube.datatransfer.scheduler.db.model.TransferObject;
import org.gcube.datatransfer.scheduler.db.model.TypeOfSchedule;
import org.gcube.datatransfer.scheduler.impl.check.CheckDBForTransfersThread;
import org.gcube.datatransfer.scheduler.impl.context.SchedulerContext;
import org.gcube.datatransfer.scheduler.impl.context.ServiceContext;
import org.gcube.datatransfer.scheduler.impl.state.SchedulerResource;
import org.gcube.datatransfer.scheduler.is.ISManager;
import org.gcube.datatransfer.scheduler.library.obj.InfoCancelSchedulerMessage;
import org.gcube.datatransfer.scheduler.library.obj.SchedulerObj;
import org.gcube.datatransfer.scheduler.library.outcome.CallingSchedulerResult;
import org.gcube.datatransfer.scheduler.stubs.datatransferscheduler.FrequencyType;
import org.gcube.datatransfer.scheduler.stubs.datatransferscheduler.StorageType;
import org.globus.wsrf.ResourceException;

import com.thoughtworks.xstream.XStream;


public class Scheduler extends GCUBEPortType {
	/** The UUIDGen */
	private static final UUIDGen uuidgen = UUIDGenFactory.getUUIDGen();

	public DataTransferDBManager dbManager;
	public ISManager<Agent> isManagerForAgents;
	public ISManager<DataSource> isManagerForSources;
	public ISManager<DataStorage> isManagerForStorages;

	@Override
	protected ServiceContext getServiceContext() {return ServiceContext.getContext();}

	/*
	 * storeInfoScheduler
	 * input: String InfoSchedulerMessage (StartTransferMessage & TypeOfScheduler)
	 * return: String with the transfer id
	 */
	public String storeInfoScheduler(String msg) throws GCUBEFault {

		ServiceContext sctx = ServiceContext.getContext();
		this.dbManager=ServiceContext.getContext().getDbManager();
		this.isManagerForAgents=ServiceContext.getContext().getIsManagerForAgents();
		this.isManagerForSources=ServiceContext.getContext().getIsManagerForSources();
		this.isManagerForStorages=ServiceContext.getContext().getIsManagerForStorages();

		SchedulerResource resource=null;
		String nameOfCheckThread=null;
		try {
			//1. **** Retrieve the Resource ..****
			resource = this.getResource();
			nameOfCheckThread=resource.getCheckDBThread();
			System.out.println("Stateful Service(storeInfoScheduler) - Thread name for checking DB:"+resource.getCheckDBThread());

		}catch (Exception e) {
			throw sctx.getDefaultException(e).toFault();
		}

		
		//2. **** Storing in the DB .. ****
		SchedulerObj schedulerObj= new SchedulerObj();
		String transferId = uuidgen.nextUUID();
		//Create the transfer in DB
		Transfer t = new Transfer();
		t.setTransferId(transferId);
		Set<TransferObject> transferObjects=new HashSet<TransferObject>();

		String tmpMsg=msg;
		tmpMsg.replaceAll("&lt;", "<");
		tmpMsg=tmpMsg.replaceAll("&gt;", ">");

		XStream xstream = new XStream();
		schedulerObj=(SchedulerObj)xstream.fromXML(tmpMsg);

		String submitter=null;
		try {
			//set submitter
			submitter=this.getResource().getName();
			t.setSubmitter(submitter);
		} catch (ResourceException e1) {
			System.out.println("Stateful Service(storeInfoScheduler) - Exception in setting the submitter in Transfer:\n");
			e1.printStackTrace();
		}
		//set status
		t.setStatus("STANDBY");

		//set the scope
		try{
			String scope = schedulerObj.getScope();
			t.setScope(scope);
		}
		catch(Exception e){
			System.out.println("Stateful Service(storeInfoScheduler) - Exception in taking the scope:\n");
			e.printStackTrace();
		}

		
		String hostnameOfAgent = schedulerObj.getAgentHostname();
		//check for the agent in IS
		String checkResultFromIS=null;
		checkResultFromIS=this.isManagerForAgents.checkIfObjExistsInIS_ByHostname(hostnameOfAgent);
		int num=0;
		while(checkResultFromIS==null){
			if(num!=0)checkResultFromIS=this.isManagerForAgents.checkIfObjExistsInIS_ByHostname(hostnameOfAgent);
			System.out.println("Stateful Service(storeInfoScheduler) " +
					"- Error!! - there is no agent with hostname= '"+hostnameOfAgent+"' in IS now\n");
			//sleeping ..........
			try {			
				Thread.sleep(4000);
			} catch (InterruptedException e) {
				System.out.println("Stateful Service ..- InterruptedException-Unable to sleep");
				e.printStackTrace();
			}
			num++;
		}
		//check for the agent in DB
		String checkResultFromDB=null;
		checkResultFromDB=this.isManagerForAgents.checkIfObjExistsInDB_ById(checkResultFromIS);
		num=0;
		while(checkResultFromDB==null){
			if(num!=0)checkResultFromDB=this.isManagerForAgents.checkIfObjExistsInDB_ById(checkResultFromIS);
			System.out.println("Stateful Service(storeInfoScheduler) " +
					"- Error!! - there is no agent with hostname= '"+hostnameOfAgent+"' in DB now or it's not 'UP' yet\n");
			//sleeping ..........
			try {			
				Thread.sleep(4000);
			} catch (InterruptedException e) {
				System.out.println("Stateful Service ..- InterruptedException-Unable to sleep");
				e.printStackTrace();
			}
			num++;
		}
		System.out.println("Stateful Service ..- agent= '"+hostnameOfAgent+"'");
		//set agent
		t.setAgentId(checkResultFromDB);
		
		
		TypeOfSchedule typeOfSchedule = new TypeOfSchedule();		
		typeOfSchedule.setTypeOfScheduleId(transferId.concat("-typeOfSchedule"));

		// ## if Direct Transfer
		if(schedulerObj.getTypeOfSchedule().isDirectedScheduled()==true){
			typeOfSchedule.setDirectedScheduled(true);
			
			CheckDBForTransfersThread checkDBForTransfersThread = (CheckDBForTransfersThread) getThread(nameOfCheckThread);
			checkDBForTransfersThread.setImmediateCheck(true);
			
		}// ## if Manual Transfer - a specific instance
		else if(schedulerObj.getTypeOfSchedule().getManuallyScheduled()!=null){
			ManuallyScheduled manuallyScheduled= new ManuallyScheduled();
			manuallyScheduled.setManuallyScheduledId(transferId.concat("-manuallyScheduled"));
			manuallyScheduled.setCalendar(schedulerObj.getTypeOfSchedule().getManuallyScheduled().getCalendar());

			
			CheckDBForTransfersThread checkDBForTransfersThread = (CheckDBForTransfersThread) getThread(nameOfCheckThread);
			long checkForTransfersIntervalMS = checkDBForTransfersThread.getCheckForTransfersIntervalMS();
			
			Calendar calendarTmp=schedulerObj.getTypeOfSchedule().getManuallyScheduled().getCalendar();
			long timeThatTransferWillHappen = calendarTmp.getTimeInMillis()-Calendar.getInstance().getTimeInMillis();
			System.out.println("\nStateful Service(storeInfoScheduler) - checkForTransfersIntervalMS="+checkForTransfersIntervalMS+" - timeThatTransferWillHappenMS="+timeThatTransferWillHappen);
			//if the transfer is about to happen during the time that the thread is sleeping, we change the time interval
			if(timeThatTransferWillHappen>0 && timeThatTransferWillHappen<=checkForTransfersIntervalMS){
				checkDBForTransfersThread.setCheckForTransfersIntervalMS(timeThatTransferWillHappen);
				System.out.println("\nStateful Service(storeInfoScheduler) - checkForTransfersIntervalMS="+timeThatTransferWillHappen+" (CHANGED)");
			}
			else if(timeThatTransferWillHappen<0){//we check also if the dateInstance is very old .. if so we replace it with the present
				manuallyScheduled.setCalendar(Calendar.getInstance());
				checkDBForTransfersThread.setImmediateCheck(true);
			}
			
			try	{ 
				typeOfSchedule.setManuallyScheduledId(transferId.concat("-manuallyScheduled"));
				ServiceContext.getContext().getDbManager().storeManuallyScheduled(manuallyScheduled);
			}
			catch(Exception e){
				System.out.println("Stateful Service(storeInfoScheduler) - Exception in storing the ManuallyScheduled:\n");
				e.printStackTrace();
			}
		

		}// ## if Periodically Transfer - every minute/hour/day/.. etc
		else if(schedulerObj.getTypeOfSchedule().getPeriodicallyScheduled()!=null){
			PeriodicallyScheduled periodicallyScheduled = new PeriodicallyScheduled();
			periodicallyScheduled.setPeriodicallyScheduledId(transferId.concat("-periodicallyScheduled"));
			FrequencyType frequency = schedulerObj.getTypeOfSchedule().getPeriodicallyScheduled().getFrequency();

			if(frequency==FrequencyType.perMinute){
				periodicallyScheduled.setFrequency(org.gcube.datatransfer.scheduler.db.utils.Utils.FrequencyType.perMinute);
			}
			else if(frequency==FrequencyType.perHour){
				periodicallyScheduled.setFrequency(org.gcube.datatransfer.scheduler.db.utils.Utils.FrequencyType.perHour);
			}
			else if(frequency==FrequencyType.perDay){
				periodicallyScheduled.setFrequency(org.gcube.datatransfer.scheduler.db.utils.Utils.FrequencyType.perDay);
			}
			else if(frequency==FrequencyType.perWeek){
				periodicallyScheduled.setFrequency(org.gcube.datatransfer.scheduler.db.utils.Utils.FrequencyType.perWeek);
			}
			else if(frequency==FrequencyType.perMonth){
				periodicallyScheduled.setFrequency(org.gcube.datatransfer.scheduler.db.utils.Utils.FrequencyType.perMonth);
			}
			else if(frequency==FrequencyType.perYear){
				periodicallyScheduled.setFrequency(org.gcube.datatransfer.scheduler.db.utils.Utils.FrequencyType.perYear);
			}
			periodicallyScheduled.setStartInstance(schedulerObj.getTypeOfSchedule().getPeriodicallyScheduled().getStartInstance());

			CheckDBForTransfersThread checkDBForTransfersThread = (CheckDBForTransfersThread) getThread(nameOfCheckThread);
			long checkForTransfersIntervalMS = checkDBForTransfersThread.getCheckForTransfersIntervalMS();
			
			//we check if the startInstance is very old .. if so we replace it with the present
			Calendar calendarTmp=schedulerObj.getTypeOfSchedule().getPeriodicallyScheduled().getStartInstance();
			long timeThatTransferWillHappen = calendarTmp.getTimeInMillis()-Calendar.getInstance().getTimeInMillis();
			if(timeThatTransferWillHappen<0 ){
				periodicallyScheduled.setStartInstance(Calendar.getInstance());
				checkDBForTransfersThread.setImmediateCheck(true);
			}
			else periodicallyScheduled.setStartInstance(calendarTmp);

			try	{ 
				typeOfSchedule.setPeriodicallyScheduledId(transferId.concat("-periodicallyScheduled"));
				ServiceContext.getContext().getDbManager().storePeriodicallyScheduled(periodicallyScheduled);
			}
			catch(Exception e){
				System.out.println("Stateful Service(storeInfoScheduler) - Exception in storing the PeriodicallyScheduled:\n");
				e.printStackTrace();
			}
			

			long frequencyInMS = this.frequencyInMS(frequency);
			//if the frequency is less than the interval for checking the DB, 
			//we change the interval for checking the DB
			System.out.println("\nStateful Service(storeInfoScheduler) - checkForTransfersIntervalMS="+checkForTransfersIntervalMS+" - frequencyInMS="+frequencyInMS);
			if(frequencyInMS>0 && frequencyInMS<checkForTransfersIntervalMS){
				checkDBForTransfersThread.setCheckForTransfersIntervalMS(frequencyInMS); 
				System.out.println("\nStateful Service(storeInfoScheduler) - checkForTransfersIntervalMS="+frequencyInMS+" (CHANGED)");
			}
		}

		try	{
			//set typeOfSchedule
			t.setTypeOfScheduleId(transferId.concat("-typeOfSchedule"));
			ServiceContext.getContext().getDbManager().storeTypeOfSchedule(typeOfSchedule);
		}
		catch(Exception e){
			System.out.println("Stateful Service(storeInfoScheduler) - Exception in storing the TypeOfSchedule:\n");
			e.printStackTrace();
		}

		//set transferType
		String transferType= schedulerObj.getTypeOfTransfer();
		t.setTransferType(transferType);
		
		String[] inputURIs=null;
		
		// **** If LocalFileBasedTransfer **** 
		if(transferType.compareTo("LocalFileBasedTransfer")==0){
			//one type here
			// uris (files) from client to the agent			
			
			String destinationFolder = schedulerObj.getDestinationFolder();
			boolean overwrite = schedulerObj.isOverwrite();
			boolean unzipFile = schedulerObj.isUnzipFile();
			inputURIs = schedulerObj.getInputUrls();

			// for every inputUrl we create a transferObject which represents a file
			// we collect all of them in a set of objects and after storing the transfer
			//we also store the transferObjectSet ... 
			TransferObject obj=null;
			for(String inputUri : inputURIs){
				obj=new TransferObject();
				try {
					obj.setURI(new URI(inputUri));
				} catch (URISyntaxException e) {
					e.printStackTrace();
				}
				obj.setTransferid(transferId);
				transferObjects.add(obj);
			}
			
			//its a LocalFileBasedTransfer so we don't need a datastorage node here... 
			t.setDestinationFolder(destinationFolder);
			t.setOverwrite(overwrite);
			t.setUnzipFile(unzipFile);
			
		}// **** If FileBasedTransfer **** 
		else if (transferType.compareTo("FileBasedTransfer")==0){
			//two types here... 
			//first urls[] or dataSourceName(containing all the info) to the agent
			//and second urls[] or dataSourceName(containing all the info) to the DataStorage node
			
			StorageType storageType= schedulerObj.getStorageType();
			String destinationFolder = schedulerObj.getDestinationFolder();	
			boolean overwrite = schedulerObj.isOverwrite();
			boolean unzipFile = schedulerObj.isUnzipFile();
			inputURIs = schedulerObj.getInputUrls();
						
			// if inputURIs is not null it means that we don't have any source node
			// we use directly the urls.. 
			if(inputURIs!=null){
				// for every inputUrl we create a transferObject which represents a file
				// we collect all of them in a set of objects and after storing the transfer
				//we also store the transferObjectSet ... 
				TransferObject obj=null;
				for(String inputUri : inputURIs){
					obj=new TransferObject();
					try {
						obj.setURI(new URI(inputUri));
					} catch (URISyntaxException e) {
						e.printStackTrace();
					}
					obj.setTransferid(transferId);
					transferObjects.add(obj);
				}
			}			
			else {// in other case we do have a source node... 
				//normally it will check for the dataSource in IS.. not manually

				String sourceHostName="";
				//String sourceHostName = infoObj.getStartTransferMessage().getSource().getHostName();

				//check for the source node in IS
				checkResultFromIS=null;
				checkResultFromIS=this.isManagerForSources.checkIfObjExistsInIS_ByHostname(sourceHostName);
				num=0;
				while(checkResultFromIS==null){
					if(num!=0)checkResultFromIS=this.isManagerForSources.checkIfObjExistsInIS_ByHostname(sourceHostName);
					System.out.println("Stateful Service(storeInfoScheduler) " +
							"- Error!! - there is no dataSource node with hostname= '"+sourceHostName+"' in IS now\n");
					//sleeping ..........
					try {			
						Thread.sleep(4000);
					} catch (InterruptedException e) {
						System.out.println("Stateful Service ..- InterruptedException-Unable to sleep");
						e.printStackTrace();
					}
					num++;
				}
				//check for the source node in DB
				checkResultFromDB=null;
				checkResultFromDB=this.isManagerForSources.checkIfObjExistsInDB_ById(checkResultFromIS);
				num=0;
				while(checkResultFromDB==null){
					if(num!=0)checkResultFromDB=this.isManagerForSources.checkIfObjExistsInDB_ById(checkResultFromIS);
					System.out.println("Stateful Service(storeInfoScheduler) " +
							"- Error!! - there is no source with hostname= '"+sourceHostName+"' in DB now or it's not 'UP' yet\n");
					//sleeping ..........
					try {			
						Thread.sleep(4000);
					} catch (InterruptedException e) {
						System.out.println("Stateful Service ..- InterruptedException-Unable to sleep");
						e.printStackTrace();
					}
				}
				//set source
				t.setSourceId(checkResultFromDB);
			}
			
			t.setDestinationFolder(destinationFolder);
			t.setOverwrite(overwrite);
			t.setUnzipFile(unzipFile);
			
			// if storageType = LocalGHN, it means we don't have any storage node 
			// we use the local node (agent) as a storage node
			if(storageType.getValue()==StorageType.LocalGHN.getValue()){
				//do nothing
			}// but if storageType = StorageManager, we do have a DataStorage node (usually remote node)
			else if(storageType.getValue()==StorageType.StorageManager.getValue()){

				DataStorage datastorage=new DataStorage();
				datastorage.setDataStorageId(transferId.concat("-datastorage"));
				datastorage.setType(StorageType.StorageManager.getValue());	

				//normally it will check for the dataStorage in IS.. not manually
				//supposed that the datastorage is the node that the agent service's running
				//datastorage.setHost("pcitgt1012.cern.ch");
				//datastorage.setPort(8080);

				try	{ // *** store the DataStorage in DB ***
					//set dataStorage
					t.setStorageId(datastorage.getDataStorageId());
					ServiceContext.getContext().getDbManager().storeStorage(datastorage);
				}
				catch(Exception e){
					System.out.println("Stateful Service(storeStorage) " +
							"- Exception in storing the DataStorage:\n");
					e.printStackTrace();
				}	
			}
			
		}// **** If TreeBasedTransfer **** 
		else if (transferType.compareTo("TreeBasedTransfer")==0){
			//##### Empty for now #####
		}

		try	{ // *** store the transfer in DB ***
			ServiceContext.getContext().getDbManager().storeTransfer(t);
		}
		catch(Exception e){
			System.out.println("Stateful Service(storeInfoScheduler) " +
					"- Exception in storing the Transfer:\n");
			e.printStackTrace();
		}

		if(inputURIs!=null && transferObjects!=null){ //if we had inputUris and not DataSource identity
			try	{// *** store the transfer objects in DB *** 
				ServiceContext.getContext().getDbManager().storeTransferObject(transferObjects);
			}
			catch(Exception e){
				System.out.println("Stateful Service(storeInfoScheduler) " +
						"- Exception in storing the Set of Transfer Objects:\n");
				e.printStackTrace();
			}
			System.out.println("");
		}
		
		//we return the transferId of the Transfer in SchedulerDB
		return transferId;
	}

	/*
	 * cancelScheduledTransfer
	 * input: String CancelTransferMessage (transferId (the one in the schedulerDB) & isForceCancel)
	 * return: String with CallingSchedulerResult obj (xml)
	 * if exception or error occurred, CallingSchedulerResult contains the specific error message
	 */
	public String cancelScheduledTransfer(String msg) throws GCUBEFault {

		this.dbManager=ServiceContext.getContext().getDbManager();
		this.isManagerForAgents=ServiceContext.getContext().getIsManagerForAgents();

		List<String> errors = new ArrayList<String>();
		CallingSchedulerResult callingSchedulerResult = new CallingSchedulerResult();

		String tmpMsg=msg;
		tmpMsg.replaceAll("&lt;", "<");
		tmpMsg=tmpMsg.replaceAll("&gt;", ">");

		XStream xstream = new XStream();
		InfoCancelSchedulerMessage cancelObj= new InfoCancelSchedulerMessage();
		cancelObj=(InfoCancelSchedulerMessage)xstream.fromXML(tmpMsg);

		String transferId = cancelObj.getCancelTransferMessage().getTransferID();
		Boolean isForceCancel = cancelObj.getCancelTransferMessage().isForceStop();
		//System.out.println("Stateful Service(cancelScheduledTransfer) - transferId="+transferId);

		Transfer transfer = this.dbManager.getPersistenceManager().getObjectById(Transfer.class,transferId);
		String typeOfScheduleId = transfer.getTypeOfScheduleId();
		TypeOfSchedule typeOfSchedule = this.dbManager.getPersistenceManager().getObjectById(TypeOfSchedule.class, typeOfScheduleId);

		try{
			//check for the status before we make the cancel
			String status = transfer.getStatus();
			System.out.println("Stateful Service(cancelScheduledTransfer) - status before Calling the cancelTransfer in Agent!! = "+status);
			if(status=="COMPLETED" || status=="FAILED"){
				//if it's a periodically schedule we have to change its status to "CANCELED" 
				if(typeOfSchedule.getPeriodicallyScheduledId()!=null){
					try {
						this.dbManager.updateTransferStatus(transferId, "CANCELED");
					} catch (Exception e) {
						System.out.println("\nStateful Service(cancelScheduledTransfer) - " +
								"Exception in updating the status to Canceled\n");
						errors.add("Stateful Service(cancelScheduledTransfer) - " +
								"Exception in updating the status to Canceled\n"+e.getMessage());
						callingSchedulerResult.setErrors(errors);
						String msgStr = callingSchedulerResult.toXML();
						e.printStackTrace();
						return msgStr;	
					}
					callingSchedulerResult.setCancelResult("DONE");
					String msgStr = callingSchedulerResult.toXML();
					return msgStr;	
				}//in other case there is no point to change it because this transfer is already done... 
				else{
					callingSchedulerResult.setCancelResult("Transfer already done");
					String msgStr = callingSchedulerResult.toXML();
					return msgStr;
				}
			}
			else if (status=="CANCELED"){
				callingSchedulerResult.setCancelResult("CANCELED");
				String msgStr = callingSchedulerResult.toXML();
				return msgStr;	
			}
			else if(status=="STANDBY"){
				//we change the status of transfer to CANCELED
				try {
					this.dbManager.updateTransferStatus(transferId, "CANCELED");
				} catch (Exception e) {
					System.out.println("\nStateful Service(cancelScheduledTransfer) - " +
							"Exception in updating the status to Canceled\n");
					errors.add("Stateful Service(cancelScheduledTransfer) - " +
							"Exception in updating the status to Canceled\n"+e.getMessage());
					callingSchedulerResult.setErrors(errors);
					String msgStr = callingSchedulerResult.toXML();
					e.printStackTrace();
					return msgStr;		
				}
				callingSchedulerResult.setCancelResult("DONE");
				String msgStr = callingSchedulerResult.toXML();
				return msgStr;
			}
			//else status "ONGOING"
			
			//retrieving the agentId
			String agentId = transfer.getAgentId();
			if(agentId==null){
				System.out.println("Stateful Service(cancelScheduledTransfer) - Error - agentId=null");
				errors.add("Stateful Service(cancelScheduledTransfer) - Error - agentId=null");
				callingSchedulerResult.setErrors(errors);
				String msgStr = callingSchedulerResult.toXML();
				return msgStr;
			}

			Agent tmpAgent = this.dbManager.getPersistenceManager().getObjectById(Agent.class,agentId);
			String hostAgent=tmpAgent.getHost();
			int portAgent = tmpAgent.getPort();
			String scope = transfer.getScope();
			
			AgentLibrary agentLibrary = null;
			ScopeProvider.instance.set(scope); 
			agentLibrary =  transferAgent().at(hostAgent, portAgent).build();
			
			//retrieving the transferId that the Agent has as identifier of this specific transfer
			String transferIdOfAgent = transfer.getTransferIdOfAgent();
			if(transferIdOfAgent==null){
				System.out.println("Stateful Service(cancelScheduledTransfer) - Error - transferIdOfAgent=null");
				errors.add("Stateful Service(cancelScheduledTransfer) - Error - transferIdOfAgent=null");
				callingSchedulerResult.setErrors(errors);
				String msgStr = callingSchedulerResult.toXML();
				return msgStr;
			}

			agentLibrary.cancelTransfer(transferIdOfAgent, isForceCancel);
		}
		catch (Exception e) {
			System.out.println("\nStateful Service(cancelScheduledTransfer) - " +
					"Exception in calling the cancelTransfer\n");		   	
			errors.add("Stateful Service(cancelScheduledTransfer) - " +
					"Exception in calling the cancelTransfer\n"+e.getMessage());
			callingSchedulerResult.setErrors(errors);
			String msgStr = callingSchedulerResult.toXML();
			e.printStackTrace();
			return msgStr;		
		}

		//we change the status of transfer to CANCELED
		try {
			this.dbManager.updateTransferStatus(transferId, "CANCELED");
		} catch (Exception e) {
			System.out.println("\nStateful Service(cancelScheduledTransfer) - " +
					"Exception in updating the status to Canceled\n");
			errors.add("Stateful Service(cancelScheduledTransfer) - " +
					"Exception in updating the status to Canceled\n"+e.getMessage());
			callingSchedulerResult.setErrors(errors);
			String msgStr = callingSchedulerResult.toXML();
			e.printStackTrace();
			return msgStr;					
		}
		System.out.println("\nStateful Service(cancelScheduledTransfer) - status="+transfer.getStatus());

		callingSchedulerResult.setCancelResult("DONE");
		String msgStr = callingSchedulerResult.toXML();
		return msgStr;	    
	}

	/*
	 * monitorScheduledTransfer
	 * input: String with the transferId (the one in the schedulerDB)
	 * return: String with result of monitoring (status in SchedulerDB)
	 */
	public String monitorScheduledTransfer(String msg) throws GCUBEFault {

		this.dbManager=ServiceContext.getContext().getDbManager();
		this.isManagerForAgents=ServiceContext.getContext().getIsManagerForAgents();

		String transferId = msg;
		String status = this.dbManager.getPersistenceManager().getObjectById(Transfer.class,transferId).getStatus();
		return status;
	}


	/*
	 * getScheduledTransferOutcomes
	 * input: String with the transferId (the one in the schedulerDB)
	 * return: String with CallingSchedulerResult obj (xml)
	 * if exception or error occurred, CallingSchedulerResult contains the specific error message
	 */
	public String getScheduledTransferOutcomes(String msg) throws GCUBEFault {

		this.dbManager=ServiceContext.getContext().getDbManager();
		this.isManagerForAgents=ServiceContext.getContext().getIsManagerForAgents();

		List<String> errors = new ArrayList<String>();
		StringBuilder outcomes = new StringBuilder();
		CallingSchedulerResult callingSchedulerResult = new CallingSchedulerResult();

		String transferId = msg;
		Transfer transfer = this.dbManager.getPersistenceManager().getObjectById(Transfer.class,transferId);

		//retrieving the transferId that the Agent has as identifier of this specific transfer
		String transferIdOfAgent = transfer.getTransferIdOfAgent();
		if(transferIdOfAgent==null){
			callingSchedulerResult.setSchedulerOutcomes("The Transfer has not started yet or It is a sync op.");
			String msgStr = callingSchedulerResult.toXML();
			return msgStr;
		}

		//retrieving the agentId
		String agentId = transfer.getAgentId();
		if(agentId==null){
			System.out.println("Stateful Service(monitorScheduledTransfer) - Error - agentId=null");
			errors.add("Stateful Service(monitorScheduledTransfer) - agentId=null");
			callingSchedulerResult.setErrors(errors);
			String msgStr = callingSchedulerResult.toXML();
			return msgStr;
		}

		Agent tmpAgent = this.dbManager.getPersistenceManager().getObjectById(Agent.class,agentId);
		String hostAgent= tmpAgent.getHost();
		int portAgent = tmpAgent.getPort();
		String scope = transfer.getScope();
		AgentLibrary agentLibrary = null;	

		try{
			ScopeProvider.instance.set(scope); 
			agentLibrary =  transferAgent().at(hostAgent, portAgent).build();
		}
		catch (Exception e) {
			System.out.println("Stateful Service(monitorScheduledTransfer) - Exception when building agentLibrary");
			errors.add("Stateful Service(monitorScheduledTransfer) - Exception when building agentLibrary\n"+e.getMessage());
			callingSchedulerResult.setErrors(errors);
			String msgStr = callingSchedulerResult.toXML();
			e.printStackTrace();
			return msgStr;
		}

		//to do - we can get the ids of objects in scheduler that failed or succeeded. 
		//String[] objectTrasferredIDs = transfer.getObjectTrasferredIDs();
		//String[] objectFailedIDs= transfer.getObjectFailedIDs();


		try{
			//*** getTransferOutcomes *** //

			ArrayList<FileTransferOutcome> outcomesArray = agentLibrary.getTransferOutcomes(transferIdOfAgent, FileTransferOutcome.class);			
			int numOfObj=0;
			for (FileTransferOutcome outcome : outcomesArray){
				outcomes.append("Outcome-"+numOfObj+"\n");
				outcomes.append("Exception: "+outcome.getException()+"\n");
				outcomes.append("FileName: "+ outcome.getFilename()+"\n");
				outcomes.append("Success?: "+ outcome.isSuccess()+"\n");
				outcomes.append("Failure?: "+ outcome.isFailure()+"\n");
				numOfObj++;
			}
		}
		catch (Exception e) {
			System.out.println("Stateful Service(monitorScheduledTransfer) - Exception when calling the getTransferOutcomes and read the outcomes");
			errors.add("Stateful Service(monitorScheduledTransfer) - Exception when calling the getTransferOutcomes and read the outcomes\n"+e.getMessage());
			callingSchedulerResult.setErrors(errors);
			String msgStr = callingSchedulerResult.toXML();
			e.printStackTrace();
			return msgStr;
		}

		callingSchedulerResult.setSchedulerOutcomes(outcomes.toString());
		String msgStr = callingSchedulerResult.toXML();
		return msgStr;
	}


	
	/*
	 * getThread
	 */
	static Thread getThread( final String name ) {
	    if ( name == null )
	        throw new NullPointerException( "Null name" );
	    final Thread[] threads = getAllThreads( );
	    for ( Thread thread : threads )
	        if ( thread.getName( ).equals( name ) )
	            return thread;
	    return null;
	}
	/*
	 * getAllThreads
	 */
	static Thread[] getAllThreads( ) {
	    final ThreadGroup root = getRootThreadGroup( );
	    final ThreadMXBean thbean = ManagementFactory.getThreadMXBean( );
	    int nAlloc = thbean.getThreadCount( );
	    int n = 0;
	    Thread[] threads;
	    do {
	        nAlloc *= 2;
	        threads = new Thread[ nAlloc ];
	        n = root.enumerate( threads, true );
	    } while ( n == nAlloc );
	    return java.util.Arrays.copyOf( threads, n );
	}
	
	static ThreadGroup rootThreadGroup = null;
	
	/*
	 * getRootThreadGroup
	 */
	static ThreadGroup getRootThreadGroup( ) {
	    if ( rootThreadGroup != null )
	        return rootThreadGroup;
	    ThreadGroup tg = Thread.currentThread( ).getThreadGroup( );
	    ThreadGroup ptg;
	    while ( (ptg = tg.getParent( )) != null )
	        tg = ptg;
	    return tg;
	}
	/*
	 * frequencyInMS
	 * input: FrequencyType (the one from the stubs)
	 * return: Long with the given frequency in MS
	 */
	public long frequencyInMS (FrequencyType frequency){
		if(frequency==FrequencyType.perMinute){
			return 1000*60;
		}
		else if(frequency==FrequencyType.perHour){
			return 1000*60*60;
		}
		else if(frequency==FrequencyType.perDay){
			return 1000*60*60*24;
		}
		else if(frequency==FrequencyType.perWeek){
			return 1000*60*60*24*7;
		}
		else if(frequency==FrequencyType.perMonth){
			return 1000*60*60*24*7*30;
		}
		else if(frequency==FrequencyType.perYear){
			return 1000*60*60*24*7*30*12;
		}
		return 0;		
	}	
	
	/*
	 * getResource
	 * input: Nothing
	 * return: Resource (the one represented from the specific submitter)
	 * ResourceException if no resource was found in the current context
	 */
	private SchedulerResource getResource() throws ResourceException {
		return (SchedulerResource) SchedulerContext.getContext().getWSHome().find();
	}



}
