/**
 * 
 */
package org.gcube.dataanalysis.copernicus.motu.client;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.Vector;

import org.apache.commons.io.FileUtils;
import org.gcube.dataanalysis.copernicus.motu.util.MotuWorkspace;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements facilities to merge all chunks downloaded for a given
 * request into a single .nc file.
 *
 * @author Paolo Fabriani
 *
 */
public class ChunkMerger {

    /**
     * A logger for this class.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(ChunkMerger.class);

    /**
     * A workspace where the merger can do its work.
     */
    private MotuWorkspace workspace;
    
    public ChunkMerger() {
        MotuWorkspace workspace = new MotuWorkspace(UUID.randomUUID().toString());
        workspace.setExecutionsRoot("/tmp/motu-chunk-merger");
        workspace.setInputLocation("input");
        workspace.setOutputLocation("output");
        workspace.ensureStructureExists();
        this.workspace = workspace;
    }

    public void mergeAll(File sourceDir,  File destFile) throws IOException, Exception {

        // organize all files in a directory structure
        this.distributeFiles(sourceDir, this.workspace.getInputLocation());

        // process
        for (File f : workspace.getInputLocation().listFiles()) {
            if (f.isDirectory()) {
                this.process(f, destFile);
            }
        }

        this.workspace.destroy();
    }

    /**
     * Take all files downloaded in 'input' and organize hierarchically in
     * 'output' to enable appending/concatenating.
     * @throws IOException
     */
    private void distributeFiles(File sourceDir, File destDir) throws IOException {
        File[] files = sourceDir.listFiles();
        for (File file : files) {
            String name = file.getName();
            name = name.replaceAll("-", "/");
            FileUtils.copyFile(file, new File(destDir, name));
        }
    }

    /**
     * This method assumes the directory is already organized hierachically to drive the merge/concatenation.
     * Recursively (pre-visit) process all subdirectories.
     * 
     * @param sourceDir
     * @param destination
     * @throws IOException
     * @throws Exception
     */
    private void process(File sourceDir, File destination)
            throws IOException, Exception {

        // for each dir, if there's no corresponding .nc file, process (dir)
        for (File f : sourceDir.listFiles()) {
            if (f.isDirectory()) {
                File nc = new File(sourceDir, f.getName() + ".nc");
                if (!nc.exists()) {
                    // the destination file
                    this.process(f, nc);
                }
            }
        }
        
        LOGGER.debug("all files are ready for appending/concatenating");

        // get the split parameter
        String type = "";
        for (File f : sourceDir.listFiles()) {
            type = f.getName().substring(0, 1);
        }

        // concatenate or merge
        if ("t".equals(type)) {
            this.concatenate(sourceDir, destination);
//            FileUtils.deleteDirectory(sourceDir);
        } else if ("v".equals(type)) {
            this.append(sourceDir, destination);
//            FileUtils.deleteDirectory(sourceDir);
        } else {
            throw new Exception("unsupported dimension for merging: " + type);
        }
    }

    /**
     * Concatenate all nc files in the directory.
     * @param sourceDir
     * @param destFile
     */
    private void concatenate(final File sourceDir, final File destFile)
            throws Exception {
        // select .nc files only
        List<File> files = new Vector<>();
        for (File f : sourceDir.listFiles()) {
            if (!f.isDirectory() && f.getName().endsWith(".nc")) {
                files.add(f);
            }
        }

        // we need them sorted alphabetically
        Collections.sort(files);

        if (files.isEmpty()) {
            // no files here is an unexpected status
            throw new Exception("no files to concatenate");
        } else if (files.size() == 1) {
            // nothing to merge; copy it to the upper level. Does it ever
            // happen?
            FileUtils.moveFile(files.get(0), destFile);
        } else {
            // concatenate files
            File first = files.get(0);
            File timedFirst = new File(first.getAbsolutePath() + "-time.nc");
            String cmd1 = "ncks -O -h --mk_rec_dmn time "
                    + first.getAbsolutePath() + " "
                    + timedFirst.getAbsolutePath();
            LOGGER.debug(cmd1);
            this.workspace.exec(cmd1, sourceDir, "log.txt");
            String cmd2 = "ncrcat -h " + timedFirst.getAbsolutePath();
            for (int i = 1; i < files.size(); i++) {
                cmd2 += " " + files.get(i).getAbsolutePath();
            }
            cmd2 += " " + destFile.getAbsolutePath();
            LOGGER.debug(cmd2);
            this.workspace.exec(cmd2, sourceDir, "log.txt");
        }
    }

    /**
     * Concatenate all nc files in the directory.
     *
     * @param sourceDir the directory containing source .nc files.
     * @param destFile the appended output file.
     * 
     * @throws Exception 
     */
    private void append(final File sourceDir, final File destFile)
            throws Exception {
        // select .nc files only
        List<File> files = new Vector<>();
        for (File f : sourceDir.listFiles()) {
            if (!f.isDirectory() && f.getName().endsWith(".nc")) {
                files.add(f);
            }
        }

        // we need them sorted alphabetically
        Collections.sort(files);

        if (files.size() == 0) {
            // no files here is an unexpected status
            throw new Exception("no files to append");
        } else if (files.size() == 1) {
            // nothing to merge; copy it to the upper level. Does it ever
            // happen?
            FileUtils.moveFile(files.get(0), destFile);
        } else {
            File tmpMerged = new File(sourceDir, "appended.nc");
            for (File f : files) {
                String cmd = "ncks -h -A " + f.getAbsolutePath() + " "
                        + tmpMerged.getAbsolutePath();
                this.workspace.exec(cmd, sourceDir, "log.txt");
            }
            FileUtils.moveFile(tmpMerged, destFile);
        }
    }

}
